package itez.core.runtime.session;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.servlet.http.HttpServletRequest;

import com.jfinal.core.Controller;

import itez.core.runtime.modules.ModuleBase;
import itez.core.runtime.modules.ModuleManager;
import itez.core.runtime.modules.ModuleManager.ControllerConfig;
import itez.core.runtime.service.common.IComp;
import itez.kit.EJson;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.ERegex;
import itez.kit.EStr;
import itez.kit.EWeb;
import itez.kit.rsa.RsaKeys;
import itez.kit.rsa.RsaKit;

public class EAttr {

	private boolean devMode; //是否调试模式
	
	private String scheme; //请求协议名称

	private String host; //主机头名称
	private String web; //应用根地址，主要用于访问本地的静态资源
	private String src; //带子域的根地址，主要用于访问租户的服务端资源

	private IComp compDef; //默认公司信息
	private IComp comp; //当前会话的公司信息
	private String domain; //当前的二级域名（未启用二级域名解析，或未匹配到域名时，将使用默认域名）

	private Boolean commonUseCDN; //是否启用CDN
	private String commonDocLibPath; //公共文档资源访问地址（HTTP），可设置为七牛等云存储访问域名
	private String commonResLibPath; //公共静态资源访问地址（HTTP），可设置为七牛等云存储访问域名
	private String commonPicLibPath; //公共图片库访问地址（HTTP），可设置为七牛等云存储访问域名
	
	private String indexCtrl; //默认控制器
	private String indexUrl; //首页地址
	private String loginUrl; //登录页面地址
	private String errorUrl; //错误页面地址
	
	private String tempLayoutPath; //公共布局文件本地路径
	private String tempCommonPath; //公共视图文件本地路径

	private Integer page = 1; //当前页码
	
	private String clientIp = null; //客户端IP地址
	private String agent = null; //客户端user-agent信息
	private String browser = null; //浏览器名称
	private String browserVer = null; //浏览器版本
	private Boolean isAjax = false; //是否Ajax访问
	private Boolean isMobile = false; //是否为移动环境访问
	private Boolean isWx = false; //是否为微信内访问
	private Boolean isWxa = false; //是否为微信小程序内访问
	private String wxaAppId = null; //小程序的AppId

	private String moduleUrl = null; //当前模块完整路径
	private String moduleCode = null; //当前模块Code
	private String modulePath = null; //当前模块Path
	private String moduleName = null; //当前模块名称
	private String moduleView = null; //当前模块资源文件夹名称
	private String moduleAsset = null; //模块资源文件访问地址（HTTP）
	private String moduleTemp = null; //模块视图文件访问地址（HTTP）
	private String ctrl = null; //当前控制器的访问地址，不包含action，便于视图端快速组成向当前控制器交互的请求地址
	
	private String requestUrl = null; //本次请求的完整地址
	private String requestReferer = null; //本次请求的来源地址
	private String fileStoreUrl = null; //文件存储服务根路径
	
	private String rsaPublicKey = null; //RSA公钥

	private final String wxaServer = "https://servicewechat.com/"; //微信小程序请求路径
	private Map<String, Object> items = new ConcurrentHashMap<String, Object>(); //其他参数

	/**
	 * 构造方法
	 * @param request
	 * @param compDef 默认子域
	 * @param comp 请求子域
	 * @param appendDomain 是否需要在根路径中自动附加子域
	 */
	public EAttr(HttpServletRequest request, IComp compDef, IComp comp, boolean appendDomain){
		
		/**
		 * 初始化默认单位及当前单位
		 */
		this.compDef = compDef;
		this.comp = comp;
		
		/**
		 * 初始化基本参数
		 */
		domain = comp.getDomain();
		devMode = EProp.DevMode;
		scheme = EWeb.getScheme(request);
		host = EWeb.getHostPath(request);
		web = EWeb.getContextPath(request);
		requestUrl = EWeb.getFullUrl(request);
		requestReferer = EWeb.getReferer(request);
		isAjax = EWeb.isAjax(request);
		
		/**
		 * 初始化分页参数
		 */
		String pageStr = request.getParameter("page");
		if(EStr.notEmpty(pageStr) && pageStr.matches("\\d+")) {
			page = Integer.parseInt(pageStr);
		}
		
		/**
		 * 处理公共资源库及公共图片库的访问路径
		 */
		commonUseCDN = EProp.CommonUseCDN;
		commonResLibPath = formatUrl(EProp.CommonResLibPath);
		commonDocLibPath = formatUrl(EProp.CommonDocLibPath);
		commonPicLibPath = formatUrl(EProp.CommonPicLibPath);

		/**
		 * 初始化后台服务端资源根地址
		 */
		src = web;
		if(EProp.DomainHandle && appendDomain) src = src + "/" + domain;
		
		/**
		 * 初始化默认模块
		 */
		ModuleBase defaultModule = ModuleManager.me.getDefaultModule();
		String platDefModule = defaultModule.getModulePath();			
		String compDefModule = comp.get("index");						
		if(EStr.notEmpty(compDefModule)){ //租户默认模块
			int at = compDefModule.indexOf(":");
			compDefModule = compDefModule.substring(at == -1 ? 0 : at + 1);
			if(EStr.isEmpty(compDefModule)) compDefModule = "site";
		}else{ //平台默认模块
			compDefModule = platDefModule;
		}
		
		/**
		 * 自动为默认模块路径前面附加路径分隔符，便于URL拼接
		 */
		if(EStr.notEmpty(platDefModule) && !platDefModule.startsWith("/")) platDefModule = "/" + platDefModule;
		if(EStr.notEmpty(compDefModule) && !compDefModule.startsWith("/")) compDefModule = "/" + compDefModule;
		
		/**
		 * 初始化常用根路径
		 */
		indexCtrl = src + platDefModule; 				//平台默认控制器
		loginUrl = 	src + platDefModule + "/login"; 	//登录页地址
		errorUrl = 	src + platDefModule + "/error"; 	//错误页地址
		indexUrl = 	src + compDefModule; 				//租户默认首页地址
		
		/**
		 * 初始化默认模块，在Caller时会覆盖成当前请求模块
		 */
		moduleCode = defaultModule.getModuleCode();
		modulePath = defaultModule.getModulePath();
		moduleView = defaultModule.getModuleView();
		moduleName = defaultModule.getModuleName();
		moduleAsset = web + moduleView + "/asset";
		moduleTemp = web + moduleView + "/temp";
		moduleUrl = src + modulePath;
		
		/**
		 * 初始化默认的文件本地存储根路径
		 */
		fileStoreUrl = String.join("/", formatUrl(EProp.LocalFileUrl), domain);
		
		/**
		 * 初始化最常用的模板位置参数
		 */
		tempLayoutPath = EProp.TempLayoutPath;
		tempCommonPath = EProp.TempCommonPath;
		
		/**
		 * 初始化客户端环境参数
		 */
		clientIp = EWeb.getIpAddr(request);
		agent = request.getHeader("user-agent");
		initAgent();
		
		/**
		 * 获得默认公钥
		 */
		try {
			RsaKeys keys = RsaKit.getDefKeys();
			rsaPublicKey = keys.getPublicKey();
		} catch (Exception e) {
			ELog.error("获取默认RsaKeys失败！");
			if(EProp.DevMode) e.printStackTrace();
		}
	}

	/**
	 * 设置Session的当前请求信息
	 * @param controller 当前请求的控制器
	 * @return
	 */
	public EAttr Caller(Controller controller){
		//获得控制器类
		String clazz = controller.getClass().getName();
		//获得控制器注册信息
		ControllerConfig ctrlCfg = ModuleManager.me.getController(clazz);
		//获得控制器所在模块信息
		ModuleBase module = ModuleManager.me.getModuleByPkg(ctrlCfg.getPack());
		
		//初始化当前访问的模块参数
		moduleCode = module.getModuleCode();
		modulePath = module.getModulePath();
		moduleView = module.getModuleView();
		moduleName = module.getModuleName();
		moduleAsset = web + moduleView + "/asset";
		moduleTemp = web + moduleView + "/temp";
		moduleUrl = src + modulePath;
		
		//初始化当前控制器的访问地址，便于在模板中引用本控制器的根地址
		String ctrlKey = ctrlCfg.getKey();
		ctrl = src + modulePath + (ctrlKey.equals("/") ? "" : ctrlKey);
		
		return this;
	}
	
	/**
	 * <p>
	 * 处理URL地址
	 * 1、替换其中的${scheme}占位符
	 * 2、未携带http(s)的地址，附加当前的web服务器根地址
	 * </p>
	 * 
	 * @param url
	 * @return
	 */
	public String formatUrl(String url){
		if(EStr.isEmpty(url)) return url;
		if(ERegex.has(url, "^(http|\\/\\/)")) return url;
		if(!url.startsWith("/")) url = "/" + url;
		return "//" + host + url;
	}

	/**
	 * 获取当前的客户端环境
	 */
	protected void initAgent(){
		if(agent == null) return;
		String str = agent.toLowerCase();
		if(ERegex.has(str, "(ucbrowser)")){ //UC
			browser = "UC";
			browserVer = ERegex.findFirst(str, "ucbrowser[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(360se|360cse|360ee)")){ //360
			browser = "360";
			browserVer = "";
		}else if(ERegex.has(str, "(micromessenger)")){ //微信
			browser = "微信";
			browserVer = ERegex.findFirst(str, "micromessenger[ \\/]([\\w.]+)");
			isWx = true;
		}else if(ERegex.has(str, "(qqbrowser)")){ //QQ
			browser = "QQ";
			browserVer = ERegex.findFirst(str, "qqbrowser[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(edge)")){ //edge
			browser = "EDGE";
			browserVer = ERegex.findFirst(str, "edge[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(msie)")){ //IE 6~10
			browser = "IE";
			browserVer = ERegex.findFirst(str, "msie[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(trident\\/[\\w.]+\\; rv)")){ //IE 11
			browser = "IE";
			browserVer = ERegex.findFirst(str, "rv[ \\/\\:]([\\w.]+)");
		}else if(ERegex.has(str, "(firefox)")){ //Firefox
			browser = "Firefox";
			browserVer = ERegex.findFirst(str, "firefox[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(chrome)")){ //Chrome
			browser = "Chrome";
			browserVer = ERegex.findFirst(str, "chrome[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(safari)")){ //Safari
			browser = "Safari";
			browserVer = ERegex.findFirst(str, "safari[ \\/]([\\w.]+)");
		}else{
			browser = "未知";
			browserVer = "";
		}
		if(browserVer == null) browserVer = "0";
		if(ERegex.has(str, "(mobile|android|phone|pod|pad)")) isMobile = true;
		
		//根据requestReferer判断请求来源是否为小程序
		if(EStr.notEmpty(requestReferer) && requestReferer.startsWith(wxaServer)){
			isWxa = true;
			wxaAppId = ERegex.findFirst(requestReferer.substring(wxaServer.length()), "^([^/]+)\\/");
		}
	}
	
	/**
	 * 检测终端是否满足要求
	 * @return
	 */
	public Boolean checkAgent(){
		if(isMobile) return true;
		if(browser.equals("IE")){
			Integer ver = Integer.parseInt(browserVer.split("\\.")[0]);
			if(ver >= 11) return true;
			else return false;
		}else if(browser.equals("Firefox") || browser.equals("Chrome") || browser.equals("Safari") || browser.equals("EDGE")){
			return true;
		}
		return false;
	}
	
	/**
	 * 转换为JSON
	 * @return
	 */
	public String getJson(){
		return EJson.toJsonFilter(this, "devMode", "host", "web", "src", "domain", "commonUseCDN", "commonDocLibPath", "commonResLibPath", "commonPicLibPath", "indexCtrl", "indexUrl", "loginUrl", "page", "ctrl", "moduleCode", "moduleUrl", "modulePath", "moduleName", "moduleAsset", "isWx", "isMobile", "fileStoreUrl", "rsaPublicKey");
	}
	
	/**
	 * 添加其他参数
	 * @param key
	 * @param value
	 */
	public void setItem(String key, Object value){
		items.put(key, value);
	}
	
	/**
	 * 返回其他参数值
	 * @param key
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public <T> T getItem(String key){
		Object value = items.get(key);
		if(value == null) return null;
		return (T)value;
	}
	
	/**
	 * <p>
	 * 返回定义的租户根域名，非动态获取，主要用于生成静态页面
	 * </p>
	 * 
	 * @return
	 */
	public String getDomainUrl(){
		String temp = EProp.DomainTemplate;
		String root = temp.replaceAll("\\$\\{domain\\}", domain);
		return root;
	}
	
	public boolean getDevMode() {
		return devMode;
	}

	public String getScheme() {
		return scheme;
	}
	
	public String getHost() {
		return host;
	}

	public String getWeb() {
		return web;
	}

	public String getSrc() {
		return src;
	}

	public IComp getCompDef() {
		return compDef;
	}

	public IComp getComp() {
		return comp;
	}

	public String getDomain() {
		return domain;
	}

	public Boolean getCommonUseCDN() {
		return commonUseCDN;
	}

	public String getCommonDocLibPath() {
		return commonDocLibPath;
	}

	public String getCommonResLibPath() {
		return commonResLibPath;
	}

	public String getCommonPicLibPath() {
		return commonPicLibPath;
	}

	public String getIndexCtrl() {
		return indexCtrl;
	}

	public String getIndexUrl() {
		return indexUrl;
	}

	public String getLoginUrl() {
		return loginUrl;
	}

	public String getErrorUrl() {
		return errorUrl;
	}

	public String getTempLayoutPath() {
		return tempLayoutPath;
	}

	public String getTempCommonPath() {
		return tempCommonPath;
	}

	public Integer getPage() {
		return page;
	}

	public String getClientIp() {
		return clientIp;
	}

	public String getAgent() {
		return agent;
	}

	public String getBrowser() {
		return browser;
	}

	public String getBrowserVer() {
		return browserVer;
	}

	public boolean getIsAjax() {
		return isAjax;
	}

	public boolean getIsMobile() {
		return isMobile;
	}

	public boolean getIsWx() {
		return isWx;
	}

	public boolean getIsWxa() {
		return isWxa;
	}

	public String getWxaAppId() {
		return wxaAppId;
	}

	public String getModuleUrl() {
		return moduleUrl;
	}

	public String getModuleCode() {
		return moduleCode;
	}

	public String getModulePath() {
		return modulePath;
	}

	public String getModuleName() {
		return moduleName;
	}

	public String getModuleView() {
		return moduleView;
	}

	public String getModuleAsset() {
		return moduleAsset;
	}

	public String getModuleTemp() {
		return moduleTemp;
	}

	public String getCtrl() {
		return ctrl;
	}

	public String getRequestUrl() {
		return requestUrl;
	}

	public String getRequestReferer() {
		return requestReferer;
	}

	public String getFileStoreUrl() {
		return fileStoreUrl;
	}

	public String getRsaPublicKey() {
		return rsaPublicKey;
	}
	
}
