package itez.core.runtime.service;

import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Method;
import java.util.Set;

import com.google.inject.Binder;
import com.google.inject.Guice;
import com.google.inject.Injector;
import com.google.inject.Module;
import com.google.inject.matcher.AbstractMatcher;
import com.google.inject.matcher.Matcher;
import com.google.inject.matcher.Matchers;
import com.google.inject.name.Names;
import com.jfinal.aop.Before;

import itez.core.launch.JW;
import itez.core.runtime.service.common.ICompService;
import itez.core.runtime.service.common.IUserService;
import itez.core.runtime.service.interceptor.AopInterceptor;
import itez.kit.EClassSearch;
import itez.kit.ELog;
import itez.kit.EStr;
import itez.kit.log.ELogBase;

/**
 * 业务逻辑管理
 * @author netwild
 *
 */
public class ServiceManager implements Module {

	private static final ELogBase log = ELog.log(ServiceManager.class);
	public static final ServiceManager me = new ServiceManager();
	private Injector injector;
	
	private ServiceManager(){
		injector = Guice.createInjector(this);
	}
	
	@Override
	public void configure(Binder binder) {
		bindAop(binder);
		bindService(binder);
	}
	
	private void bindAop(Binder binder){
        Matcher<AnnotatedElement> matcher = Matchers.annotatedWith(Define.class).or(Matchers.annotatedWith(Before.class));
        AbstractMatcher<Method> notSynthetic = new AbstractMatcher<Method>() {
            @Override
            public boolean matches(Method method) {
                return !method.isSynthetic();
            }
        };
        binder.bindInterceptor(matcher, notSynthetic, new AopInterceptor());
	}
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private void bindService(Binder binder){
		Set<Class<?>> serviceList = EClassSearch.getClassList(JW.ServicePackage, true, EService.class, Define.class);
		for(Class clazz : serviceList){
			Class<?>[] interfaceClasses = clazz.getInterfaces();
			if(interfaceClasses.length == 0) continue;
			Define service = (Define)clazz.getAnnotation(Define.class);
			String name = service.value();
			for (Class<?> interfaceClass : interfaceClasses) {
	            try {
	            	if(EStr.isEmpty(name)){
	                    binder.bind(interfaceClass).to(clazz);
	            	}else{
	                    binder.bind(interfaceClass).annotatedWith(Names.named(name)).to(clazz);
	                }
	            } catch (Throwable ex) {
	                System.err.println(String.format("Service绑定失败： [%s] to [%s]", interfaceClass, clazz));
	            }
			}
		}
	}

    public Injector getInjector() {
        return injector;
    }
    
    public void injectMembers(Object object){
    	injector.injectMembers(object);
    }
    
	/**
	 * 根据服务别名，返回服务对象的单列代理实例
	 * @param code
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public <T> T getService(Class<?> interfaceClazz){
		Object ins = injector.getInstance(interfaceClazz);
		return ins == null ? null : (T) ins;
	}
	
	/**
	 * 对常用服务进行特殊处理，方便操作
	 * @return
	 */
	public ICompService getCompService() {
		ICompService compService = getService(ICompService.class);
		if(compService == null) log.error("还未初始化compService");
		return compService;
	}
	public IUserService getUserService() {
		IUserService userService = getService(IUserService.class);
		if(userService == null) log.error("还未初始化userService");
		return userService;
	}
	
}
