package itez.core.runtime.service;

import java.util.List;

import com.jfinal.plugin.activerecord.Page;

import itez.core.wrapper.dbo.model.EModel;
import itez.core.wrapper.dbo.model.Querys;
import itez.core.wrapper.dbo.model.Selector;

public interface IModelService<M extends EModel<M>> extends IService {

	//======================= 基本数据检索 ==================================
	
	/**
	 * 根据主键ID返回Model
	 * @category 根据主键ID返回Model
	 *
	 * @param id 主键ID
	 */
	public M findById(String id);
	
	/**
	 * 根据多个主键ID返回Model列表
	 * @category 根据多个主键ID返回Model列表
	 *
	 * @param ids 主键ID列表，数组或者不定个数参数
	 */
	public List<M> findByIda(String...ids);
	
	/**
	 * 根据多个主键ID返回Model列表
	 * @category 根据多个主键ID返回Model列表
	 *
	 * @param ids 主键ID列表，用逗号分隔
	 */
	public List<M> findByIds(String ids);

	/**
	 * 根据别名（code）返回Model
	 * @category 根据别名（code）返回Model
	 *
	 * @param code 别名
	 */
	public M findByCode(String code);
	
	/**
	 * 根据多个别名（code）返回Model列表
	 * @category 根据多个别名（code）返回Model列表
	 *
	 * @param ids 别名（code）列表，数组或者不定个数参数
	 */
	public List<M> findByCodea(String...codes);
	
	/**
	 * 根据多个别名（code）返回Model列表
	 * @category 根据多个别名（code）返回Model列表
	 *
	 * @param ids 别名（code）列表，用逗号分隔
	 */
	public List<M> findByCodes(String codes);
	
	/**
	 * 根据子域返回Model列表
	 * @return
	 */
	public List<M> findByDomain();
	
	/**
	 * 根据子域返回单条Model
	 * @return
	 */
	public M findByDomainSingle();

	//======================= 复杂数据检索 ==================================
	
	/**
	 * 返回全部Model列表
	 * @category 返回全部Model列表
	 *
	 * @return 返回当前子域的,used=1的记录，默认按sort asc排序
	 */
	public List<M> selectAll();
	
	/**
	 * 返回全部Model列表，只返回指定字段的数据
	 * @category 返回全部Model列表，只返回指定字段的数据
	 *
	 * @param columnNames 属性别名列表，数组或不定个数参数
	 * @return 返回当前子域的,used=1的记录，默认按sort asc排序
	 */
	public List<M> selectAll(String...columnNames);
	
	/**
	 * 返回全部Model列表，指定排序方式
	 * @category 返回全部Model列表，指定排序方式
	 *
	 * @return 返回当前子域的,used=1的记录，自定义排序
	 */
	public List<M> selectAllBy(String orderBy);
	
	/**
	 * 返回全部Model列表，指定排序方式，只返回指定字段的数据
	 * @category 返回全部Model列表，指定排序方式，只返回指定字段的数据
	 *
	 * @param columnNames 属性别名列表，数组或不定个数参数
	 * @return 返回当前子域的,used=1的记录，自定义排序
	 */
	public List<M> selectAllBy(String orderBy, String...columnNames);
	
	/**
	 * 返回第一条Model记录
	 * @category 返回第一条Model记录
	 *
	 * @return 返回当前子域的,used=1的，按sort asc排序的第一条记录
	 */
	public M selectFirst();
	
	/**
	 * 返回第一条Model记录，只返回指定字段的数据
	 * @category 返回第一条Model记录，只返回指定字段的数据
	 *
	 * @param columnNames 属性别名列表，数组或不定个数参数
	 * @return 返回当前子域的,used=1的，按sort asc排序的第一条记录
	 */
	public M selectFirst(String...columnNames);
	
	/**
	 * 返回第一条Model记录，自定义检索条件
	 * @category 返回第一条Model记录，自定义检索条件
	 *
	 * @return 返回当前子域的,used=1的，且符合检索条件的，按sort asc排序的第一条记录
	 */
	public M selectFirst(Querys querys);
	
	/**
	 * 返回第一条Model记录，自定义检索条件，指定排序方式
	 * @category 返回第一条Model记录，自定义检索条件，指定排序方式
	 *
	 * @return 返回当前子域的,used=1的，且符合检索条件的，按指定排序方式的第一条记录
	 */
	public M selectFirst(Querys querys, String orderBy);
	
	/**
	 * 返回第一条Model记录，自定义检索条件，指定排序方式，只返回指定字段的数据
	 * @category 返回第一条Model记录，自定义检索条件，指定排序方式，只返回指定字段的数据
	 *
	 * @param columnNames 属性别名列表，数组或不定个数参数
	 * @return 返回当前子域的,used=1的，且符合检索条件的，按指定排序方式的第一条记录
	 */
	public M selectFirst(Querys querys, String orderBy, String...columnNames);

	//======================= 自定义数据检索 ==================================
	
	/**
	 * 按条件检索
	 * @category 按条件检索
	 * @return 返回当前子域的,used=1的，且符合检索条件的记录列表
	 */
	public List<M> select(Querys querys);
	
	/**
	 * 按条件检索，指定排序方式
	 * @category 按条件检索，指定排序方式
	 * @return 返回当前子域的,used=1的，按指定排序方式，且符合检索条件的记录列表
	 */
	public List<M> select(Querys querys, String orderBy);
	
	/**
	 * 按条件检索，指定排序方式，记录条数
	 * @category 按条件检索，指定排序方式，记录条数
	 * @return 返回当前子域的,used=1的，按指定排序方式，记录条数，且符合检索条件的记录列表
	 */
	public List<M> select(Querys querys, String orderBy, Object limit);
	
	/**
	 * 按条件检索，指定排序方式，记录条数，只返回指定字段的数据
	 * @category 按条件检索，指定排序方式，记录条数
	 * @return 返回当前子域的,used=1的，按指定排序方式，记录条数，且符合检索条件的记录列表
	 */
	public List<M> select(Querys querys, String orderBy, Object limit, String...columnNames);
	
	/**
	 * 按条件检索，指定排序方式，记录条数，只返回指定字段的数据
	 * @category 按条件检索，指定排序方式，记录条数
	 * @return 返回当前子域的,used=1的，按指定排序方式，记录条数，且符合检索条件的记录列表
	 */
	public List<M> select(Querys querys, String orderBy, Object limit, Boolean autoCheck, String...loadColumns);
	
	/**
	 * 按条件检索，指定分组方式，指定排序方式，记录条数，只返回指定字段的数据
	 * @category 按条件检索，指定排序方式，记录条数
	 * @return 返回当前子域的,used=1的，按指定排序方式，记录条数，且符合检索条件的记录列表
	 */
	public List<M> select(Querys querys, String groupBy, String orderBy, Object limit, Boolean autoCheck, String...loadColumns);

	/**
	 * 使用SQL选择器检索
	 * @param selector
	 * @return
	 */
	public List<M> select(Selector selector);
	
	/**
	 * 使用SQL选择器检索
	 * @param selector
	 * @param autoCheck
	 * @return
	 */
	public List<M> select(Selector selector, Boolean autoCheck);
	
	//======================= 分页检索 ==================================
	
	/**
	 * 分页检索
	 * @param pageNumber
	 * @param pageSize
	 * @param selector
	 * @return
	 */
	public Page<M> paginate(int pageNumber, int pageSize, Selector selector);
	
	/**
	 * 分页检索
	 * @param pageNumber
	 * @param pageSize
	 * @param selector
	 * @param autoCheck
	 * @return
	 */
	public Page<M> paginate(int pageNumber, int pageSize, Selector selector, Boolean autoCheck);
	
	//======================= 单列检索 ==================================
	
    /**
     * 按条件、指定列
     * @category 按条件、指定列
     * @param querys
     * @param column
     * @return
     */
	public <T> List<T> selectCol(Querys querys, String column);
	
    /**
     * 按条件、指定列、排序方式
     * @category 按条件、指定列、排序方式
     * @param querys
     * @param column
     * @param orderBy
     * @return
     */
	public <T> List<T> selectCol(Querys querys, String column, String orderBy);
    
    /**
     * 按条件、指定列、排序方式、返回条数
     * @category 按条件、指定列、排序方式、返回条数
     * @param querys
     * @param column
     * @param orderBy
     * @param limit
     * @return
     */
	public <T> List<T> selectCol(Querys querys, String column, String orderBy, Object limit);
	
	//======================= 保存或更新 ==================================
	
    /**
     * 保存或更新
     * @category 保存或更新
     * @param model
     * @return
     */
	public boolean saveOrUpdate(M model);
	
    /**
     * 保存
     * @category 保存
     * @param model
     * @return
     */
	public boolean save(M model);
	
    /**
     * 更新
     * @category 更新
     * @param model
     * @return
     */
	public boolean update(M model);
	
	//======================= 事务处理 ==================================

    /**
     * 批量添加
     * @param list
     * @return
     */
	public boolean batchSave(List<M> list);

    /**
     * 批量修改
     * @param list
     * @return
     */
	public boolean batchUpdate(List<M> list);
	
	//======================= 物理删除及逻辑删除 ==================================
	
    /**
     * 物理删除（按ID）
     * @category 物理删除（按ID）
     * @param idValue
     * @return
     */
	public boolean deleteById(String idValue);
	
    /**
     * 物理删除（批量）
     * @category 物理删除（批量）
     * @param idValues
     * @return
     */
	public boolean deleteByIda(String...idValues);
	
    /**
     * 物理删除（批量）
     * @category 物理删除（批量）
     * @param idValues
     * @return
     */
	public boolean deleteByIds(String idValues);
	
    /**
     * 物理删除（批量）
     * @category 物理删除（批量）
     * @param codes
     * @return
     */
	public boolean deleteByCodes(String codes);
	
    /**
     * 获取可用状态
     * @category 获取可用状态
     * @param model
     * @return
     */
	public boolean able(M model);
	
    /**
     * 逻辑删除（按ID）
     * @category 逻辑删除（按ID）
     * @param model
     * @return
     */
	public boolean disable(String idValue);
	
    /**
     * 逻辑删除（批量）
     * @category 逻辑删除（批量）
     * @param model
     * @return
     */
	public boolean disableByIda(String...idValues);
	
    /**
     * 逻辑删除（批量）
     * @category 逻辑删除（批量）
     * @param model
     * @return
     */
	public boolean disableByIds(String idValues);
	
    /**
     * 逻辑删除
     * @category 逻辑删除
     * @param model
     * @return
     */
	public boolean disable(M model);
	
    /**
     * 逻辑恢复（按ID）
     * @category 逻辑恢复（按ID）
     * @param model
     * @return
     */
	public boolean enable(String idValue);
	
    /**
     * 逻辑删除（批量）
     * @category 逻辑删除（批量）
     * @param model
     * @return
     */
	public boolean enableByIda(String...idValues);
	
    /**
     * 逻辑删除（批量）
     * @category 逻辑删除（批量）
     * @param model
     * @return
     */
	public boolean enableByIds(String idValues);
	
    /**
     * 逻辑恢复
     * @category 逻辑恢复
     * @param model
     * @return
     */
	public boolean enable(M model);
	
	/**
	 * <p>
	 * 根据指定属性去重
	 * </p>
	 * 
	 * @param list
	 * @param key
	 */
	public void removeRepeat(List<M> list, String key);
	
}
