package itez.core.runtime.service;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jfinal.plugin.activerecord.Model;
import com.jfinal.plugin.activerecord.Record;

import itez.core.runtime.EContext;
import itez.core.runtime.service.common.IComp;
import itez.core.runtime.service.common.ICompService;
import itez.core.runtime.service.common.IUser;
import itez.core.runtime.session.EAttr;
import itez.core.wrapper.dbo.model.EModel;
import itez.kit.EClass;
import itez.kit.EProp;
import itez.kit.EUid;

public abstract class EService implements IService {

	
	//======================= 上下文环境 ==================================
	
	/**
	 * 获得UUID
	 * @return
	 */
	protected String $UUID(){
		return EUid.generator();
	}

	/**
	 * 获得当前租户的子域
	 * @return
	 */
	protected String $domain() {
		EAttr attr = EContext.getAttr();
		return null == attr ? EProp.DomainDefault : attr.getDomain();
	}

	/**
	 * 获得当前租户的实例
	 * @return
	 */
	protected IComp $comp() {
		EAttr attr = EContext.getAttr();
		if(null == attr){
			ICompService compSer = Ioc.getComp();
			return compSer.findByDomain($domain());
		}else{
			return attr.getComp();
		}
	}
	
	/**
	 * 获得当前用户的登录状态
	 * @return
	 */
	protected Boolean $logined(){
		return EContext.getSession().getLogined();
	}

	/**
	 * 获得当前登录用户的ID
	 * @return
	 */
	protected String $uid() {
		return EContext.getSession().getUid();
	}

	/**
	 * 获得当前登录用户的实例
	 * @return
	 */
	protected IUser $user() {
		return EContext.getSession().getUser();
	}
	
	//======================= 表连接（1 vs n） ==================================

	/**
	 * 合并主细表（1 vs n）
	 * @category 合并主细表
	 * 
	 * 例如：
	 * 		主表（mainList）	：[{id: 'P1', name: '主1'} ..]
	 * 		细表（subList）	：[{id: 'S1', name: '细1', pId: 'P1'} ..]
	 * 合并：
	 * 		append(mainList, "id", subList, "pId", "sub")
	 * 结果：
	 * 		[{id: 'P1', name: '主1', sub: [{id: 'S1', name: '细1', pId: 'P1'} ..]} ..]
	 * 
	 * @param parentList 主表集合
	 * @param subList 子表集合
	 * @param foreignKey 子表外键
	 * @param joinName 链接别名
	 * @return
	 */
	@Override
	public <T extends EModel<T>, V extends EModel<V>> List<T> append(List<T> parentList, List<V> subList, String foreignKey, String joinName){
		return append(parentList, "id", subList, foreignKey, joinName);
	}
    
	/**
	 * 合并主细表（1 vs n）
	 * @category 合并主细表
	 * 
	 * 例如：
	 * 		主表（mainList）	：[{id: 'P1', name: '主1'} ..]
	 * 		细表（subList）	：[{id: 'S1', name: '细1', pId: 'P1'} ..]
	 * 合并：
	 * 		append(mainList, "id", subList, "pId", "sub")
	 * 结果：
	 * 		[{id: 'P1', name: '主1', sub: [{id: 'S1', name: '细1', pId: 'P1'} ..]} ..]
	 * 
	 * @param parentList 主表集合
	 * @param primaryKey 主表主键
	 * @param subList 子表集合
	 * @param foreignKey 子表外键
	 * @param joinName 链接别名
	 * @return
	 */
	@Override
	public <T extends EModel<T>, V extends EModel<V>> List<T> append(List<T> parentList, String primaryKey, List<V> subList, String foreignKey, String joinName){
		Map<String, List<V>> parentStore = Maps.newHashMap();
		List<T> parentCopy = Lists.newArrayList();
		parentList.forEach(p -> {
			parentCopy.add(p.copy());
			parentStore.put(p.get(primaryKey).toString(), Lists.newArrayList());
		});
		subList.forEach(s -> {
			String linkId = s.getStr(foreignKey);
			if(parentStore.containsKey(linkId)) parentStore.get(linkId).add(s);
		});
		parentCopy.forEach(p -> {
			p.put(joinName, parentStore.get(p.get(primaryKey)));
		});
		return parentCopy;
	}
	
	/**
	 * 表连接（left join right）
	 * @category 表连接（left join right）
	 * 
	 * @param leftList 左侧主表集合
	 * @param leftKey 左侧主表连接字段名称
	 * @param rightList 右侧表集合
	 * @param rightKey 右侧表连接字段名称
	 * @param rightFields 右侧表字段重命名（格式：field 或 field > fieldNew）
	 */
	@Override
	public <T extends EModel<T>, V extends EModel<V>> List<T> join(List<T> leftList, String leftKey, List<V> rightList, String rightKey, String...rightFields){
		if(leftList.size() == 0) return leftList;
		if(rightList.size() == 0) return leftList;
		if(rightFields.length == 0) rightFields = rightList.get(0)._getAttrNames();
		List<String[]> rightAs = Lists.newArrayList();
		Arrays.stream(rightFields).forEach(rf -> {
			String[] sarr = null;
			if(rf.indexOf(">") != -1) sarr = rf.replaceAll("\\s+", "").split(">");
			else sarr = new String[]{ rf, rf };
			rightAs.add(sarr);
		});
		Map<String, V> mapStore = Maps.newHashMap();
		rightList.forEach(r -> { mapStore.put(r.get(rightKey).toString(), r); });
		List<T> fullCopy = Lists.newArrayList();
		leftList.forEach(l -> {
			T n = l.copy();
			V r = mapStore.get(l.get(leftKey).toString());
			if(null != r) rightAs.forEach(as -> { n.put(as[1], r.get(as[0])); });
			fullCopy.add(n);
		});
		return fullCopy;
	}
	
	protected <T extends Model<T>> List<Record> parseRecords(List<T> models){
		return models.stream().map(r -> r.toRecord()).collect(Collectors.toList());
	}
	
	protected <T extends Model<T>> List<T> parseModel(List<Record> recs, Class<T> T){
		return recs.stream().map(r -> (EClass.newInstance(T)._setAttrs(r.getColumns()))).collect(Collectors.toList());
	}
	
}
