package itez.core.runtime.service;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.sql.SQLException;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import com.google.common.collect.Sets;
import com.jfinal.plugin.activerecord.Db;
import com.jfinal.plugin.activerecord.DbPro;
import com.jfinal.plugin.activerecord.IAtom;
import com.jfinal.plugin.activerecord.Page;

import itez.core.runtime.EContext;
import itez.core.wrapper.dbo.model.EModel;
import itez.core.wrapper.dbo.model.Query;
import itez.core.wrapper.dbo.model.Querys;
import itez.core.wrapper.dbo.model.Selector;
import itez.kit.EArr;
import itez.kit.EClass;
import itez.kit.EStr;
import itez.kit.exception.ErrException;

public abstract class EModelService<M extends EModel<M>> extends EService {
	
	//======================= ORM对象初始化 ==================================
	
	protected M dao = null;
	
	/**
	 * 根据泛型，自动初始化dao（Model）对象
	 */
	@SuppressWarnings("unchecked")
	protected EModelService(){
		Class<M> modelClass = null;
		Type type = EClass.getUsefulClass(getClass()).getGenericSuperclass();
        if (type instanceof ParameterizedType) {
            Type[] p = ((ParameterizedType) type).getActualTypeArguments();
            modelClass = (Class<M>) p[0];
        }
        if (modelClass == null) {
            throw new ErrException("未找到Model的POJO类");
        }
        dao = EClass.newInstance(modelClass).dao();
	}
	
	/**
	 * 返回Model所属的连接池名称
	 * @return
	 */
	protected String getConfigName(){
		return dao._getConfig().getName();
	}
	
	/**
	 * 获得Db对象
	 * @return
	 */
	protected DbPro dbo(){
		return Db.use(getConfigName());
	}
	
	//======================= 基本数据检索 ==================================
	
	/**
	 * 按主键ID检索
	 * @param id
	 * @return
	 */
    public M findById(String id) {
        return dao.findById(id);
    }
	
	/**
	 * 按多个主键ID检索
	 * @param id
	 * @return
	 */
    public List<M> findByIda(String...ids) {
        return dao.findByIda(ids);
    }
	
	/**
	 * 按多个用逗号隔开的主键ID检索
	 * @param id
	 * @return
	 */
    public List<M> findByIds(String ids) {
        return dao.findByIds(ids);
    }
    
    /**
     * 按别名检索
     * @param code
     * @return
     */
    public M findByCode(String code){
    	Querys qs = Querys.and(Query.eq("code", code));
    	return selectFirst(qs);
    }
    
    /**
     * 按子域检索
     * @param code
     * @return
     */
    public List<M> findByDomain(){
    	Querys qs = Querys.and(Query.eq("domain", EContext.getAttr().getDomain()));
    	return select(qs);
    }
    
    /**
     * 按子域检索
     * @param code
     * @return
     */
    public M findByDomainSingle(){
    	Querys qs = Querys.and(Query.eq("domain", EContext.getAttr().getDomain()));
    	return selectFirst(qs);
    }
    
    /**
     * 按多个别名检索
     * @param codes
     * @return
     */
    public List<M> findByCodea(String...codes){
    	Querys qs = Querys.and(Query.in("code", EStr.ida2sqlIn(codes)));
    	return select(qs);
    }
    
    /**
     * 按多个用逗号隔开的别名检索
     * @param codes
     * @return
     */
    public List<M> findByCodes(String codes){
    	Querys qs = Querys.and(Query.in("code", EStr.ids2sqlIn(codes)));
    	return select(qs);
    }

	//======================= 复杂数据检索 ==================================
    
    /**
     * 返回全部
     * @return
     */
    public List<M> selectAll(){
    	Querys qs = Querys.and();
    	return select(qs);
    }
    
    /**
     * 返回全部，指定列
     * @param columnNames
     * @return
     */
    public List<M> selectAll(String...columnNames){
    	Querys qs = Querys.and();
    	return select(qs, null, null, columnNames);
    }
    
    /**
     * 返回全部，指定排序
     * @param orderBy
     * @return
     */
    public List<M> selectAllBy(String orderBy){
    	Querys qs = Querys.and();
    	return select(qs, orderBy);
    }
    
    /**
     * 返回全部，指定排序，指定列
     * @param orderBy
     * @param columnNames
     * @return
     */
    public List<M> selectAllBy(String orderBy, String...columnNames){
    	Querys qs = Querys.and();
    	return select(qs, orderBy, null, columnNames);
    }
    
    /**
     * 返回第一条
     * @return
     */
    public M selectFirst(){
    	Querys qs = Querys.and();
    	return selectFirst(qs, null);
    }
    
    /**
     * 返回第一条，指定列
     * @param columnNames
     * @return
     */
    public M selectFirst(String...columnNames){
    	Querys qs = Querys.and();
    	return selectFirst(qs, null, columnNames);
    }
    
    /**
     * 按检索条件返回第一条
     * @param querys
     * @return
     */
    public M selectFirst(Querys querys){
    	return selectFirst(querys, null);
    }
    
    /**
     * 按检索条件、排序方式返回第一条
     * @param querys
     * @param orderBy
     * @return
     */
    public M selectFirst(Querys querys, String orderBy){
    	return dao.selectFirst(querys, orderBy);
    }
    
    /**
     * 按检索条件、排序方式、指定列返回第一条
     * @param querys
     * @param orderBy
     * @param columnNames
     * @return
     */
    public M selectFirst(Querys querys, String orderBy, String...columnNames){
    	return dao.selectFirst(querys, orderBy, columnNames);
    }
    
    /**
     * 按检索条件、排序方式、指定列返回第一条
     * @param querys
     * @param orderBy
     * @param columnNames
     * @return
     */
    public M selectFirst(Querys querys, String orderBy, Boolean autoCheck, String...columnNames){
    	return dao.selectFirst(querys, orderBy, autoCheck, columnNames);
    }
    
    /**
     * 使用SQL选择器检索
     * @param selector
     * @return
     */
    public M selectFirst(Selector selector){
    	return dao.selectFirst(selector);
    }

	//======================= 自定义数据检索 ==================================
    
    /**
     * 按条件检索
     * @param querys
     * @return
     */
    public List<M> select(Querys querys){
    	return select(querys, null, null);
    }
    
    /**
     * 按条件、排序方式检索
     * @param querys
     * @param orderBy
     * @return
     */
    public List<M> select(Querys querys, String orderBy){
    	return select(querys, orderBy, null);
    }
    
    /**
     * 按条件、排序方式、返回条数检索
     * @param querys
     * @param orderBy
     * @param limit
     * @return
     */
    public List<M> select(Querys querys, String orderBy, Object limit){
    	return dao.select(querys, orderBy, limit);
    }
    
    /**
     * 按条件、排序方式、返回条数、指定列检索
     * @param querys 检索条件
     * @param orderBy 排序方式
     * @param limit 返回条数限制
     * @param loadColumns
     * @return
     */
    public List<M> select(Querys querys, String orderBy, Object limit, String...columnNames){
    	return dao.select(querys, orderBy, limit, columnNames);
    }
    
    /**
     * 按条件、排序方式、返回条数、指定列检索
     * @param querys 检索条件
     * @param orderBy 排序方式
     * @param limit 返回条数限制
     * @param autoCheck 是否自动附加其他条件（租户子域、租户ID、可用标识）
     * @param loadColumns
     * @return
     */
    public List<M> select(Querys querys, String orderBy, Object limit, Boolean autoCheck, String...columnNames){
    	return dao.select(querys, orderBy, limit, autoCheck, columnNames);
    }
    
    /**
     * 按条件、排序方式、返回条数、指定列检索
     * @param querys 检索条件
     * @param groupBy 分组方式
     * @param orderBy 排序方式
     * @param limit 返回条数限制
     * @param autoCheck 是否自动附加其他条件（租户子域、租户ID、可用标识）
     * @param loadColumns
     * @return
     */
    public List<M> select(Querys querys, String groupBy, String orderBy, Object limit, Boolean autoCheck, String...columnNames){
    	return dao.select(querys, groupBy, orderBy, limit, autoCheck, columnNames);
    }
    
    /**
     * 使用SQL选择器检索
     * @param selector
     * @return
     */
    public List<M> select(Selector selector){
    	return dao.select(selector);
    }
    
    /**
     * 使用SQL选择器检索
     * @param selector
     * @param autoCheck
     * @return
     */
    public List<M> select(Selector selector, Boolean autoCheck){
    	return dao.select(selector, autoCheck);
    }

	//======================= 分页检索 ==================================
    
    /**
     * 使用SQL选择器实现分页检索
     * @param pageNumber
     * @param pageSize
     * @param selector
     * @return
     */
    public Page<M> paginate(int pageNumber, int pageSize, Selector selector){
    	return dao.paginate(pageNumber, pageSize, selector);
    }
    
    /**
     * 使用SQL选择器实现分页检索
     * @param pageNumber
     * @param pageSize
     * @param selector
     * @param autoCheck
     * @return
     */
    public Page<M> paginate(int pageNumber, int pageSize, Selector selector, Boolean autoCheck){
    	return dao.paginate(pageNumber, pageSize, selector, autoCheck);
    }

	//======================= 单列检索 ==================================
    
    /**
     * 按条件、指定列
     * @param querys
     * @param column
     * @return
     */
    public <T> List<T> selectCol(Querys querys, String column){
    	return selectCol(querys, column, null, null);
    }
    
    /**
     * 按条件、指定列、排序方式
     * @param querys
     * @param column
     * @param orderBy
     * @return
     */
    public <T> List<T> selectCol(Querys querys, String column, String orderBy){
    	return selectCol(querys, column, orderBy, null);
    }
    
    /**
     * 按条件、指定列、排序方式、返回条数
     * @param querys
     * @param column
     * @param orderBy
     * @param limit
     * @return
     */
    public <T> List<T> selectCol(Querys querys, String column, String orderBy, Object limit){
    	return dao.selectCol(querys, orderBy, limit, column);
    }
	
	//======================= 保存或更新 ==================================
    
    /**
     * 保存或更新
     * @param model
     * @return
     */
    public boolean saveOrUpdate(M model){
    	return model.saveOrUpdate();
    }
    
    /**
     * 保存
     * @param model
     * @return
     */
    public boolean save(M model){
    	return model.save();
    }
    
    /**
     * 更新
     * @param model
     * @return
     */
    public boolean update(M model){
    	return model.update();
    }
	
	//======================= 事务处理 ==================================
    
    /**
     * 批量添加
     * @param list
     * @return
     */
    public boolean batchSave(List<M> list){
		return dbo().tx(new IAtom() {
			@Override
			public boolean run() throws SQLException {
		    	int[] ret = dbo().batchSave(list, list.size());
		    	return EArr.vali(ret);
			}
		});
    }
    
    /**
     * 批量修改
     * @param list
     * @return
     */
    public boolean batchUpdate(List<M> list){
		return dbo().tx(new IAtom() {
			@Override
			public boolean run() throws SQLException {
		    	int[] ret = dbo().batchUpdate(list, list.size());
		    	return EArr.vali(ret);
			}
		});
    }
	
	//======================= 物理删除及逻辑删除 ==================================
    
    /**
     * 物理删除（按ID）
     * @param idValue
     * @return
     */
    public boolean deleteById(String idValue) {
    	return dao.deleteById(idValue);
    }
    
    /**
     * 物理删除（批量）
     * @param idValues
     * @return
     */
    public boolean deleteByIda(String...idValues){
    	List<M> list = findByIda(idValues);
    	if(list == null) return false;
    	for(M item : list){
    		item.delete();
    	}
    	return true;
    }
    
    /**
     * 物理删除（批量）
     * @param idValues
     * @return
     */
    public boolean deleteByIds(String idValues){
    	List<M> list = findByIds(idValues);
    	if(list == null) return false;
    	for(M item : list){
    		item.delete();
    	}
    	return true;
    }
    
    /**
     * 物理删除（批量）
     * @param codes
     * @return
     */
    public boolean deleteByCodes(String codes){
    	List<M> list = findByCodes(codes);
    	if(list == null) return false;
    	for(M item : list){
    		item.delete();
    	}
    	return true;
    }
    
    /**
     * 获取可用状态
     * @param model
     * @return
     */
    public boolean able(M model) {
    	return model.able();
    }
    
    /**
     * 逻辑删除（按ID）
     * @param model
     * @return
     */
    public boolean disable(String idValue) {
    	M model = findById(idValue);
    	return disable(model);
    }
    
    /**
     * 逻辑删除（批量）
     * @param model
     * @return
     */
    public boolean disableByIda(String...idValues) {
    	List<M> list = findByIda(idValues);
    	if(list == null) return false;
    	for(M item : list){
    		disable(item);
    	}
    	return true;
    }
    
    /**
     * 逻辑删除（批量）
     * @param model
     * @return
     */
    public boolean disableByIds(String idValues) {
    	List<M> list = findByIds(idValues);
    	if(list == null) return false;
    	for(M item : list){
    		disable(item);
    	}
    	return true;
    }
    
    /**
     * 逻辑删除
     * @param model
     * @return
     */
    public boolean disable(M model) {
    	if(model == null) return false;
    	return model.disable();
    }
    
    /**
     * 逻辑恢复（按ID）
     * @param model
     * @return
     */
    public boolean enable(String idValue) {
    	M model = findById(idValue);
    	return enable(model);
    }
    
    /**
     * 逻辑删除（批量）
     * @param model
     * @return
     */
    public boolean enableByIda(String...idValues) {
    	List<M> list = findByIda(idValues);
    	if(list == null) return false;
    	for(M item : list){
    		enable(item);
    	}
    	return true;
    }
    
    /**
     * 逻辑删除（批量）
     * @param model
     * @return
     */
    public boolean enableByIds(String idValues) {
    	List<M> list = findByIds(idValues);
    	if(list == null) return false;
    	for(M item : list){
    		enable(item);
    	}
    	return true;
    }
    
    /**
     * 逻辑恢复
     * @param model
     * @return
     */
    public boolean enable(M model) {
    	if(model == null) return false;
    	return model.enable();
    }
    
    /**
     * <p>
     * 按指定属性去重
     * </p>
     * 
     * @param list
     * @param key
     */
    public void removeRepeat(List<M> list, String key){
    	Set<Object> exists = Sets.newHashSet();
		ListIterator<M> listIt = list.listIterator();
		while(listIt.hasNext()){
			M d = listIt.next();
			if(exists.contains(d.get(key))){
				listIt.remove();
			}else{
				exists.add(d.get(key));
			}
		}
    }
		
}
