package itez.plat.main.controller;

import com.google.inject.Inject;
import com.jfinal.aop.Clear;
import com.jfinal.core.NotAction;
import com.jfinal.kit.HashKit;
import com.jfinal.upload.UploadFile;

import itez.core.runtime.auth.AuthCode;
import itez.core.runtime.auth.AuthRequire;
import itez.core.runtime.service.common.ICompService;
import itez.core.runtime.service.common.IUser;
import itez.core.runtime.service.common.IUserService;
import itez.core.runtime.session.ESessionBase;
import itez.core.wrapper.controller.ControllerDefine;
import itez.kit.EProp;
import itez.kit.ERet;
import itez.kit.EStr;
import itez.plat.main.model.Comp;
import itez.plat.main.model.User;
import itez.plat.main.service.UserService;
import itez.plat.main.util.ConfigKit;
import itez.plat.wrapper.controller.EControllerDef;

/**
 * 平台默认控制器
 * 只实现最通用的几个入口Action
 * 
 * @author netwild
 *
 */
@ControllerDefine(key = "/", summary = "平台首页", view = "/")
public class IndexController extends EControllerDef {

	@Inject
	ICompService compService;

	@Inject
	IUserService userService;

	@Inject
	UserService userSer;
	
	/**
	 * 平台首页
	 */
	@AuthRequire.Logined
	public void index(){
		Comp comp = (Comp)attr().getComp();
		//如果当前单位设置了自定义首页，则直接跳转
		if(EStr.notEmpty(comp.getIndex())){
			redirect(attr().getIndexUrl());
			return;
		}
		render(EProp.TempIndexPath);
	}

	/**
	 * 修改密码
	 */
	@Override
	public void pass() {
		render("pass.html");
	}
	
	public void passEvent(String passOld, String pass1, String pass2){
		if(EStr.isEmpty(passOld)){
			setFlashMsg("请填写当前密码");
			redirect(attr().getCtrl().concat("/pass"));
			return;
		}else if(EStr.isEmpty(pass1)){
			setFlashMsg("请填写新密码");
			redirect(attr().getCtrl().concat("/pass"));
			return;
		}else if(!pass1.equals(pass2)){
			setFlashMsg("两次输入的新密码不相同");
			redirect(attr().getCtrl().concat("/pass"));
			return;
		}
		User user = (User)session().getUser();
		String salt = user.getSalt();
		String pass = user.getLoginPass();
		String passOldHash = HashKit.md5(String.format("%s%s", passOld, salt));
		if(!passOldHash.equals(pass)){
			setFlashMsg("当前密码错误");
			redirect(attr().getCtrl().concat("/pass"));
			return;
		}
		user.setLoginPass(HashKit.md5(String.format("%s%s", pass1, salt)));
		userSer.modify(user);
		setFlashMsg("密码修改完成，下次请使用新密码登录");
		redirect(attr().getCtrl().concat("/pass"));
	}
	
	/**
	 * 平台登录页
	 */
	@Clear
	public void login(){
		Comp comp = (Comp)attr().getComp();
		//如果当前单位设置了自定义首页，则直接跳转
		if(EStr.notEmpty(comp.getLogin())){
			redirect(attr().getLoginUrl());
			return;
		}
		String from = getPara("from"); //登入前的页面，用于登录成功后返回
		if(EStr.notEmpty(from)) setAttr("previousPageUrl", from);
		render(EProp.TempLoginPath);
	}

	/**
	 * 验证码
	 */
	@Clear
	public void valicode(){
		renderCaptcha();
	}
	
	/**
	 * 二维码
	 * @param code
	 * @param width
	 * @param height
	 */
	@Clear
	public void qrcode(String code, Integer width, Integer height){
		if(EStr.isEmpty(width)) width = 200;
		if(EStr.isEmpty(height)) height = 200;
		renderQrCode(code, width, height);
	}
	
	/**
	 * 登录逻辑
	 * @return
	 */
	@NotAction
	private ERet loginLogic(ESessionBase.ClientType type){
		if(type == ESessionBase.ClientType.PLAT && ConfigKit.getBool("CaptchaAble") && !validateCaptcha("valicode")){
			return ERet.fail("验证码错误");
		}
		String loginName = getPara("loginName");
		String loginPass = getPara("loginPass");
		if(EStr.isEmpty(loginName) || EStr.isEmpty(loginPass)){
			return ERet.fail("用户名及密码不允许为空");
		}
		ERet ret = ERet.create();
		IUser user = userService.findByLogin(attr().getDomain(), loginName, loginPass);
		if(null == user){
			return ret.setFail().setMsg("用户名或密码错误");
		}
		if(user.getUsed() != 1){
			return ret.setFail().setMsg("用户已被注销");
		}
		ret.setOk().set("user", user);
		return ret;
	}
	
	/**
	 * 平台登录验证（Web方式）
	 */
	@Clear
	public void login4plat(){
		String previousPageUrl = getPara("previousPageUrl");
		ERet ret = loginLogic(ESessionBase.ClientType.PLAT);
		if(ret.isFail()){
			setFlashMsg(ret.getMsg());
			setFlashAttr("previousPageUrl", previousPageUrl);
			redirect(attr().getLoginUrl());
		}else if(ret.isOk()){
			IUser user = ret.getAs("user");
			session().setLogined(true);
			session().setUid(user.getId());
			String url = EStr.ifEmpty(previousPageUrl, attr().getIndexUrl());
			redirect(url);
		}
	}
	
	/**
	 * 平台登录验证（App方式）
	 */
	@Clear
	public void login4app(){
		ERet ret = loginLogic(ESessionBase.ClientType.APP);
		if(ret.isFail()){
			renderJson(ERet.fail(ret.getMsg()));
		}else if(ret.isOk()){
			IUser user = ret.getAs("user");
			session().setLogined(true);
			session().setUid(user.getId());
			renderJson(ERet.ok().set("user", user).set("token", session().getToken()));
		}
	}
	
	/**
	 * 注销身份
	 */
	@Clear
	public void logout(){
		session().setLogined(false);
		session().setUid("");
		setFlashMsg("身份已注销，请重新登录！");
		redirect(attr().getLoginUrl());
	}
	
	/**
	 * 错误页
	 */
	@Clear
	public void error(){
		AuthCode authCode = AuthCode.valueOf(getPara("authCode"));
		String errMsg = getPara("errMsg");
		String helpUrl = getPara("helpUrl");
		if(EStr.isEmpty(errMsg)) renderErr(authCode);
		else if(EStr.isEmpty(helpUrl)) renderErrMsg(authCode, errMsg);
		else renderErrHelp(authCode, errMsg, helpUrl);
		return;
	}
	
	/**
	 * 管理员工具箱
	 */
	public void tools(){
		if(session().getUser().getLevel() < 2){
			renderErrMsg(AuthCode.LOGIC, "您的权限不足，无法使用管理员工具箱");
			return;
		}
		render("tools.html");
	}
	
	/**
	 * 自定义编辑器上传
	 */
	@Clear
	public void ueditorUpload(){
		String action = getPara("action", "config");
		/**
		 * 百度编辑器加载出按钮图标前 会将所有控件的路径 先通过config.json
		 * 文件加载出来(包括上传图片路径，视频路径等路径都是通过config.json 文件读取的)
		 * 所以某些控件点击不了 是因为 config.json文件没有找到 或者是文件里面的路径有问题
		 */
		if("config".equals(action)){
			render("/main/asset/js/ueditorConfig.json");
		}else{
			UploadFile upfile = getFile("upfile");
			if(upfile != null){
				String fileName = upfile.getFileName();
				String url = getUpload(upfile.getFile(), "editor");
				ERet ret = ERet.create();
				ret.set("state", "SUCCESS");
				ret.set("url", url);
				ret.set("title", fileName);
				ret.set("original", upfile.getOriginalFileName());
				ret.set("type", fileName.split("\\.")[1]);
				ret.set("size", upfile.getFile().length());
				renderJson(ret);
			}
		}
	}

}
