package itez.kit.server;

import java.io.File;
import java.io.IOException;
import java.net.DatagramSocket;
import java.net.ServerSocket;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.session.FileSessionDataStoreFactory;
import org.eclipse.jetty.util.resource.ResourceCollection;
import org.eclipse.jetty.webapp.WebAppContext;

import com.jfinal.kit.FileKit;
import com.jfinal.kit.LogKit;
import com.jfinal.kit.PathKit;
import com.jfinal.kit.StrKit;
import com.jfinal.server.IServer;

import itez.kit.EFile;

public class LocalServer implements IServer {

	protected String webAppDir;
	protected int port;
	protected String context;
	protected int scanIntervalSeconds;
	protected boolean running = false;
	protected Server server;
	protected WebAppContext webApp;

	LocalServer(String webAppDir, int port, String context, int scanIntervalSeconds) {
		if (webAppDir == null) throw new IllegalStateException("Invalid webAppDir of web server: " + webAppDir);
		if (port < 0 || port > 65535) throw new IllegalArgumentException("Invalid port of web server: " + port);
		if (StrKit.isBlank(context)) throw new IllegalStateException("Invalid context of web server: " + context);
		
		this.webAppDir = webAppDir;
		this.port = port;
		this.context = context;
		this.scanIntervalSeconds = scanIntervalSeconds;
	}
	
	public void start() {
		if (!running) {
			try {
				running = true;
				doStart();
			} catch (Exception e) {
				System.err.println(e.getMessage());
				LogKit.error(e.getMessage(), e);
			}
		}
	}
	
	public void stop() {
		if (running) {
			try {server.stop();} catch (Exception e) {LogKit.error(e.getMessage(), e);}
			running = false;
		}
	}
	
	protected void doStart() {
		if (!available(port)) throw new IllegalStateException("端口: " + port + " 已被占用!");
		
		deleteSessionData();
		System.out.println("开始启动JWinner...");

		/**
		 * 创建默认的 HTTP 服务
		 */
        server = new Server();
        ServerConnector http = new ServerConnector(server);
        http.setPort(port);
        server.addConnector(http);

        /**
         * 创建 HTTPS 服务
         */
        setSslContext();
		
		webApp = new WebAppContext();
		webApp.setContextPath(context);
		webApp.setResourceBase(webAppDir);	// webApp.setWar(webAppDir);
		webApp.setDisplayName("EServer by Jetty");
		
		// webApp.addSystemClass("sun.");
		// webApp.addSystemClass("com.sun.");
		webApp.getSystemClasspathPattern().add("sun.");				// 支持 IDEA
		webApp.getSystemClasspathPattern().add("com.sun.");
		webApp.getSystemClasspathPattern().add("org.apache.");		// 支持 jsp
		webApp.getSystemClasspathPattern().add("org.glassfish.");	// 支持 jsp

		webApp.setConfigurationDiscovered(true);
		webApp.setParentLoaderPriority(true);
		webApp.setThrowUnavailableOnStartupException(true);	// 在启动过程中允许抛出异常终止启动并退出 JVM
		webApp.setInitParameter("org.eclipse.jetty.servlet.Default.dirAllowed", "false");
		webApp.setInitParameter("org.eclipse.jetty.servlet.Default.useFileMappedBuffer", "false");	// webApp.setInitParams(Collections.singletonMap("org.mortbay.jetty.servlet.Default.useFileMappedBuffer", "false"));
		webApp.setMaxFormContentSize(128 * 1024 * 1024);
		
		webApp.setServer(server);
		
		/**
		 * 将Jar包中的META-INF目录加载到WebRoot
		 */
		setBaseResources();
		
		persistSession();
		
		server.setHandler(webApp);
		
		server.setStopAtShutdown(true);
		server.setDumpAfterStart(false);
		server.setDumpBeforeStop(false);
		
        setClassLoader();
        
		if (scanIntervalSeconds > 0) {
			Scanner scanner = new Scanner(PathKit.getRootClassPath(), scanIntervalSeconds) {
				public void onChange() {
					try {
						System.err.println("\n加载改动 ......");
						webApp.stop();
						setClassLoader();
						webApp.start();
						System.err.println("加载成功 (^_^)");
					} catch (Exception e) {
						System.err.println("Error reconfiguring/restarting webapp after change in watched files");
						LogKit.error(e.getMessage(), e);
					}
				}
			};
			System.out.println("启动代码热加载监听器，扫描间隔： " + scanIntervalSeconds + " 秒");
			scanner.start();
		}
		
		try {
			System.out.println("启动 Jetty Server " + Server.getVersion() + "，端口: " + port);
			server.start();
			System.out.println("启动完成，欢迎使用 JWinner (^_^)");
			server.join();
		} catch (Exception e) {
			LogKit.error(e.getMessage(), e);
			System.exit(100);
		}
		return;
	}
	
	protected void setClassLoader() {
		webApp.setClassLoader(EFile.getClassLoader());
//		try {
//			String classPath = getClassPath();
//			LocalServerClassLoader loader = new LocalServerClassLoader(webApp, classPath);
//			webApp.setClassLoader(loader);
//		} catch (IOException e) {
//			LogKit.error(e.getMessage(), e);
//		}
	}
	
	protected String getClassPath() {
		return System.getProperty("java.class.path");
	}
	
	protected void deleteSessionData() {
		try {
			FileKit.delete(new File(getStoreDir()));
		}
		catch (Exception e) {
			LogKit.logNothing(e);
		}
	}
	
	protected String getStoreDir() {
		String storeDir = PathKit.getWebRootPath() + "/../../session_data" + context;
		if ("\\".equals(File.separator)) {
			storeDir = storeDir.replaceAll("/", "\\\\");
		}
		return storeDir;
	}
	
	protected void persistSession() {
		String storeDir = getStoreDir();
		FileSessionDataStoreFactory fsdsf = new FileSessionDataStoreFactory();
		fsdsf.setStoreDir(new File(storeDir));
		server.addBean(fsdsf);
	}
	
	protected static boolean available(int port) {
		if (port <= 0) throw new IllegalArgumentException("无效的启动端口: " + port);
		
		ServerSocket ss = null;
		DatagramSocket ds = null;
		try {
			ss = new ServerSocket(port);
			ss.setReuseAddress(true);
			ds = new DatagramSocket(port);
			ds.setReuseAddress(true);
			return true;
		} catch (IOException e) {
			LogKit.logNothing(e);
		} finally {
			if (ds != null) {
				ds.close();
			}
			
			if (ss != null) {
				try {
					ss.close();
				} catch (IOException e) {
					// should not be thrown, just detect port available.
					LogKit.logNothing(e);
				}
			}
		}
		return false;
	}
	
	/**
	 * 需要在 Server 启动前执行更多设置时，请覆盖该方法
	 */
	protected void setBaseResources() {
		try {
			List<String> resources = new ArrayList<String>();
			resources.add(webAppDir);
			Enumeration<URL> urls = EFile.getClassLoader().getResources("META-INF/resources");
			while (urls.hasMoreElements()) {
				URL url = urls.nextElement();
                if (url != null) {
                	String type = url.getProtocol();
                	if("file".equals(type)){
                		resources.add(url.getPath());
                	}else if("jar".equals(type)){
                		String jarPath = "jar:" + url.getPath();
                		resources.add(jarPath);
                	}
                }
			}
			String[] resourcesArr = resources.toArray(new String[resources.size()]);
			webApp.setBaseResource(new ResourceCollection(resourcesArr));
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

    /**
     * 需要启用SSL时，请覆盖该方法
     * @return
     */
    protected void setSslContext(){
		/*
		SslContextFactory scf = new SslContextFactory();
        scf.setKeyStorePath("*.pfx文件路径");
        scf.setKeyStoreType("PKCS12");
        scf.setKeyStorePassword("密码");
        scf.setNeedClientAuth(false);
        
        ServerConnector https = new ServerConnector(server, scf);
        https.setPort(443);
		https.setIdleTimeout(30000);
		https.setAcceptQueueSize(100);
        server.addConnector(https);
        */
    }

}
