package itez.kit.queue;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.jfinal.kit.Kv;
import itez.kit.ELog;
import itez.kit.log.ELogBase;
import itez.kit.log.ExceptionUtil;

/**
 * 队列工厂
 */
public class QueueFactory {

	private final ELogBase log = ELog.log(getClass());
	
	//消费者实现逻辑
	private IQueueHandle handle;

	//待处理队列
	private final BlockingQueue<Kv> queue;
	//消费者线程池
	private final ExecutorService customerPool;

	//队列容量
	private final int maxQueueSize;
	//消费者最大线程数
	private final int threadPoolSize;

	//工厂状态
	private boolean factoryState = false;
	//是否为长期执行模式（长期执行模式需要手动调用stop()方法停止工厂；否则当队列为空时，自动停止工厂）
	private boolean longTerm = false;
	
	/**
	 * 构造新工厂
	 * 
	 * @param handle 消费者实现类
	 * @param maxQueueSize 队列最大值
	 * @param threadPoolSize 线程数
	 */
	QueueFactory(IQueueHandle handle, int maxQueueSize, int threadPoolSize, boolean longTerm){
		if(handle == null) throw new RuntimeException("未发现有效的QueueHandle");
		this.handle = handle;
		this.maxQueueSize = maxQueueSize;
		this.threadPoolSize = threadPoolSize;
		this.longTerm = longTerm;
		queue = new LinkedBlockingQueue<Kv>(maxQueueSize);
		customerPool = Executors.newFixedThreadPool(threadPoolSize);
	}
	
	/**
	 * 启动工厂
	 */
	public void start(){
		if(factoryState) return;
		for(int i=0; i<threadPoolSize; i++){
			QueueThread thread = new QueueThread();
			customerPool.execute(thread);
		}
		factoryState = true;
	}
	
	/**
	 * 待当前任务处理完之后停止工厂
	 */
	public void stop(){
		stop(false);
	}
	
	/**
	 * 立即停止工厂
	 */
	public void stopNow(){
		stop(true);
	}
	
	/**
	 * 停止工厂
	 * 
	 * @param force 是否强制停止，即使还有未处理完成的任务
	 */
	private void stop(boolean force){
		if(!factoryState) return;
		if(force){
			int size = customerPool.shutdownNow().size();
			log.info("线程池已停止，未处理的任务数：{}", size);
		}else{
			customerPool.shutdown();
			try {
				customerPool.awaitTermination(3, TimeUnit.SECONDS);
			} catch (InterruptedException e) {
				//e.printStackTrace();
			}
		}
		factoryState = false;
	}
	
	/**
	 * 加入任务
	 * @param kv
	 */
	public void offer(Kv params){
		if (queue.size() >= maxQueueSize) throw new RuntimeException("队列已满");
		if(!queue.offer(params)) throw new RuntimeException("插入队列失败");
	}
	
	/**
	 * 检出任务
	 * @return
	 */
	public Kv poll(){
		return (Kv)queue.poll();
	}
	
	/**
	 * 检出任务
	 * @return
	 */
	public Kv poll(long timeout, TimeUnit unit) throws InterruptedException{
		return (Kv)queue.poll(timeout, unit);
	}
	
	/**
	 * 返回队列中的任务数
	 * @return
	 */
	public int getQueueSize(){
		return queue.size();
	}

	/**
	 * 返回消费者执行类
	 * @return
	 */
	public IQueueHandle getHandle() {
		return handle;
	}

	/**
	 * 返回队列最大值
	 * @return
	 */
	public int getMaxQueueSize() {
		return maxQueueSize;
	}

	/**
	 * 返回最大线程数
	 * @return
	 */
	public int getThreadPoolSize() {
		return threadPoolSize;
	}

	/**
	 * 返回工厂状态
	 * @return
	 */
	public boolean getFactoryState() {
		return factoryState;
	}

	/**
	 * 返回工厂是否为长期模式
	 * @return
	 */
	public boolean getLongTerm() {
		return longTerm;
	}

	/**
	 * 消费者线程类
	 */
	class QueueThread implements Runnable {

		private static final long THREAD_SLEEP = 1000;// * 60;
		private AtomicInteger count = new AtomicInteger(0);
		
		@Override
		public void run() {
			log.info("消费者线程启动：{}", Thread.currentThread().getId());
			try {
				while (getFactoryState()) {
					Kv parmas = poll();
					if (parmas != null) {
						log.info("消费者线程激活：{}，消费总数：{}，队列长度：{}", Thread.currentThread().getId(), count.incrementAndGet(), getQueueSize());
						getHandle().handle(parmas);
					}else if(!getLongTerm()){
						log.info("消费者线程停止：{}，消费总数：{}，队列长度：{}", Thread.currentThread().getId(), count.get(), getQueueSize());
						if(getFactoryState()) getHandle().callback();
						stop();
					}else{
						log.info("消费者线程休眠：{}，消费总数：{}，队列长度：{}", Thread.currentThread().getId(), count.get(), getQueueSize());
						Thread.sleep(THREAD_SLEEP);
					}
				}
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				log.error("消费者线程异常退出：{}", ExceptionUtil.getMessage(e));
			}
		}
		
	}
	
}
