package itez.kit.poi;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.OfficeXmlFileException;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import com.google.common.collect.Maps;

import itez.kit.ELog;

public class EPoi {
	
	public static Excel load(File file) throws IOException{
		return load(file, null);
	}
	
	public static Excel load(File file, boolean onlyFirst) throws IOException{
		return load(file, 1);
	}
	
	public static Excel load(File file, Integer limit) throws IOException{
		Excel excel = null;
		try {
			excel = loadLow(file, limit);
		} catch (OfficeXmlFileException e) {
			excel = loadPlus(file, limit);
		}
		return excel;
	}

	@SuppressWarnings("resource")
	private static Excel loadLow(File file, Integer limit) throws IOException, OfficeXmlFileException{
		InputStream input = new FileInputStream(file);
	    HSSFWorkbook workbook = new HSSFWorkbook(input);
	    Excel excel = new Excel();
	    Integer sheetCount = workbook.getNumberOfSheets();
	    excel.setSheetCount(sheetCount);
	    for(int i=0; i<sheetCount; i++){
	    	ESheet sheet = loadSheetLow(workbook.getSheetAt(i), limit);
	    	sheet.setIndex(i);
	    	excel.putSheet(sheet);
	    }
	    return excel;
	}
	
	private static ESheet loadSheetLow(HSSFSheet xsheet, Integer limit){
		ESheet sheet = new ESheet();
		int lastRowNum = xsheet.getLastRowNum();
		int lastColNum = 0;
		int rowsCount = 0;
		for(int i=0; i<=lastRowNum; i++){
			if(null != limit && i >= limit) break;
			HSSFRow xrow = xsheet.getRow(i);
			if(xrow == null) continue;
			rowsCount++;
			ERow row = null;
			try {
				row = loadRowLow(xrow);
				row.setIndex(i);
				sheet.putRow(row);
			} catch (Exception e) {
				ELog.error("解析EXCEL出错：行{}", i + 1);
				throw new RuntimeException(e);
			}
			if(rowsCount == 1){
				sheet.setFirstRow(row);
				lastColNum = xrow.getLastCellNum();
			}
		}
		sheet.setCaption(xsheet.getSheetName());
		sheet.setRowCount(rowsCount);
		sheet.setCellCount(lastColNum);
		return sheet;
	}
	
	private static ERow loadRowLow(HSSFRow xrow){
		ERow row = new ERow();
		int lastColNum = xrow.getLastCellNum();
		for(int i=0; i<lastColNum; i++){
			try {
				ECell cell = loadCellLow(xrow.getCell(i));
				cell.setRowIndex(xrow.getRowNum());
				cell.setCellIndex(i);
				row.putCell(cell);
			} catch (Exception e) {
				ELog.error("解析EXCEL出错：列{}", i + 1);
				throw new RuntimeException(e);
			}
		}
		return row;
	}
	
	private static ECell loadCellLow(HSSFCell xcell){
		ECell cell = new ECell();
		Map<String, Object> values = Maps.newHashMap();
		Object value = null;
		CellType cellType = xcell == null ? CellType.STRING : xcell.getCellType();
		switch (cellType) {
		case NUMERIC:
			cell.setType(cellType.name());
			value = xcell == null ? null : xcell.getNumericCellValue();
			break;
		case ERROR:
			cell.setType(CellType.STRING.name());
			value = "";
			break;
		default:
			cell.setType(CellType.STRING.name());
			value = xcell == null ? null : xcell.getStringCellValue();
			break;
		}
		values.put(cell.getType(), value);
		cell.setValue(values);
		return cell;
	}

	@SuppressWarnings("resource")
	private static Excel loadPlus(File file, Integer limit) throws IOException{
		InputStream input = new FileInputStream(file);
		XSSFWorkbook workbook = new XSSFWorkbook(input);
	    Excel excel = new Excel();
	    Integer sheetCount = workbook.getNumberOfSheets();
	    excel.setSheetCount(sheetCount);
	    for(int i=0; i<sheetCount; i++){
	    	ESheet sheet = loadSheetPlus(workbook.getSheetAt(i), limit);
	    	sheet.setIndex(i);
	    	excel.putSheet(sheet);
	    }
	    return excel;
	}
	
	private static ESheet loadSheetPlus(XSSFSheet xsheet, Integer limit){
		ESheet sheet = new ESheet();
		int lastRowNum = xsheet.getLastRowNum();
		int lastColNum = 0;
		int rowsCount = 0;
		for(int i=0; i<=lastRowNum; i++){
			if(null != limit && i >= limit) break;
			XSSFRow xrow = xsheet.getRow(i);
			if(xrow == null) continue;
			rowsCount++;
			ERow row = null;
			try {
				row = loadRowPlus(xrow);
				row.setIndex(i);
				sheet.putRow(row);
			} catch (Exception e) {
				ELog.error("解析EXCEL出错：行{}", i + 1);
				throw new RuntimeException(e);
			}
			if(rowsCount == 1){
				sheet.setFirstRow(row);
				lastColNum = xrow.getLastCellNum();
			}
		}
		sheet.setCaption(xsheet.getSheetName());
		sheet.setRowCount(rowsCount);
		sheet.setCellCount(lastColNum);
		return sheet;
	}
	
	private static ERow loadRowPlus(XSSFRow xrow){
		ERow row = new ERow();
		int lastColNum = xrow.getLastCellNum();
		for(int i=0; i<lastColNum; i++){
			try {
				ECell cell = loadCellPlus(xrow.getCell(i));
				cell.setRowIndex(xrow.getRowNum());
				cell.setCellIndex(i);
				row.putCell(cell);
			} catch (Exception e) {
				ELog.error("解析EXCEL出错：列{}", i + 1);
				throw new RuntimeException(e);
			}
		}
		return row;
	}
	
	private static ECell loadCellPlus(XSSFCell xcell){
		ECell cell = new ECell();
		Map<String, Object> values = Maps.newHashMap();
		Object value = null;
		CellType cellType = xcell == null ? CellType.STRING : xcell.getCellTypeEnum();
		switch (cellType) {
		case NUMERIC:
			cell.setType(cellType.name());
			value = xcell == null ? null : xcell.getNumericCellValue();
			break;
		case ERROR:
			cell.setType(CellType.STRING.name());
			value = "";
			break;
		default:
			cell.setType(CellType.STRING.name());
			value = xcell == null ? null : xcell.getStringCellValue();
			break;
		}
		values.put(cell.getType(), value);
		cell.setValue(values);
		return cell;
	}
	
}
