package itez.kit.mqtt;

import java.util.Map;

import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.paho.client.mqttv3.MqttPersistenceException;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;

import com.google.common.collect.Maps;

import itez.kit.ELog;
import itez.kit.EStr;
import itez.kit.log.ELogBase;

public class MqttUtil {

	/**
	 * 连接配置
	 */
	private String host;
	private String userName;
	private String passWord;
	private String clientId;
	
	/**
	 * 心跳配置
	 * 服务端在1.5个时长未收到客户端的心跳请求，则断开该客户端的连接
	 */
	private int keepAliveInterval = 60;			//心跳周期（秒）

	/**
	 * 是否自动清理离线的Session信息
	 * 1）cleanSession = true：
	 * 		该客户端上线，并订阅了某主题，那么该主题会一直存在，即使客户端离线，该主题也仍然会记忆在服务器内存。
	 * 		当客户端离线又上线时，仍然会接收到离线期间别人发来的publish消息。类似即时通讯软件可接收离线消息。
	 * 		除非客户端主动取消订阅主题，否则主题一直存在。
	 * 2）cleanSession = false：
	 * 		该客户端上线，并订阅了某主题，那么该主题会随着客户端离线而删除。
	 * 		当客户端离线又上线时，接收不到离线期间别人发来的publish消息。
	 */
	private boolean cleanSession = true;
	
	/**
	 * 发布消息默认配置：Qos（质量）
	 * 当发布消息未指定Qos时，将使用此处的默认配置
	 * 
	 * 0 - most once; 1 - least once; 2 - exactly once
	 */
	private int publishQos = 1;
	
	/**
	 * 发布消息默认配置：Retain（保持）
	 * 当发布消息未指定Retain时，将使用此处的默认配置
	 * 
	 * 当发布消息（PUBLISH）时，如果将RETAIN标志位设置为true，那么MQTT服务器会将最近收到的一条RETAIN标志位为true的消息保存在服务器端（内存或文件）。
	 * 特别注意：MQTT服务器只会为每一个Topic保存最近收到的一条RETAIN标志位为true的消息！也就是说，如果MQTT服务器上已经为某个Topic保存了一条Retained消息，当客户端再次发布一条新的Retained消息，那么服务器上原来的那条消息会被覆盖！
	 * 每当MQTT客户端连接到MQTT服务器并订阅了某个topic，如果该topic下有Retained消息，那么MQTT服务器会立即向客户端推送该条Retained消息。
	 * 
	 * 如果想让MQTT服务器为某个Topic保留消息，只需要在发布消息的时候指定RETAIN标志位为true即可。
	 * 如果客户端想让MQTT服务器删除某个Topic下保存的Retained消息，唯一的方法是向MQTT服务器发布一条RETAIN标志位为true的空消息。
	 * 注：空消息即为发布消息（PUBLISH）的时候，Payload中设置0个字节的内容。
	 */
	private boolean publishRetained = false;
	
	/**
	 * 遗愿配置
	 * 当客户端出现异常断开连接时，将由服务端代替该客户端向其他订阅者推送该遗愿消息
	 */
	private String lastWillTopic = "LASTWILL";	//遗愿主题
	private String lastWillPayload = "LOST";	//遗愿内容
	private int lastWillQos = 1;				//遗愿质量：0 - most once; 1 - least once; 2 - exactly once
	private Boolean lastWillRetained = false;	//遗愿保持：0 - 仅为当前订阅者推送此消息; 1 - 永久保存，不受服务器重启影响，为当前及以后的新订阅者推送此消息
	
	private MqttClient client;
	private MqttConnectOptions options;
	private MemoryPersistence memoryPersistence = new MemoryPersistence();
	
	private final static ELogBase log = ELog.log(MqttUtil.class);

	private final static Map<String, MqttUtil> mqtts = Maps.newConcurrentMap();
	
	public MqttUtil(String host, String clientId){
		this.host = host;
		this.clientId = clientId;
		mqtts.put(clientId, this);
	}
	
	/**
	 * 构造方法
	 * @param host
	 * @param userName
	 * @param passWord
	 * @param clientId
	 */
	public MqttUtil(String host, String userName, String passWord, String clientId) {
		this.host = host;
		this.userName = userName;
		this.passWord = passWord;
		this.clientId = clientId;
		mqtts.put(clientId, this);
	}
	
	/**
	 * 将Mqtt对象实例加入静态哈希表，实现对象重用
	 * @param clientId
	 * @return
	 */
	public static void setMqtt(MqttUtil mqtt){
		mqtts.put(mqtt.getClientId(), mqtt);
	}
	
	/**
	 * 返回指定的Mqtt对象实例
	 * @param clientId
	 * @return
	 */
	public static MqttUtil getMqtt(String clientId){
		return mqtts.get(clientId);
	}
	
	/**
	 * 连接MQTT服务器
	 * @param callback 继承MqttCallback的回调对象
	 * @return
	 * @throws MqttException
	 */
	public void connect(MqttCallback callback){
		if(null != client){
			if(!client.isConnected()){
				try {
					client.reconnect();
				} catch (Exception e) {
					log.error("MQTT重新连接时发生错误");
					e.printStackTrace();
				}
			}
		}else{
			options = new MqttConnectOptions();
			options.setCleanSession(cleanSession);
			options.setKeepAliveInterval(keepAliveInterval);
			options.setWill(lastWillTopic, lastWillPayload.getBytes(), lastWillQos, lastWillRetained);
			if(EStr.notEmpty(userName)) options.setUserName(userName);
			if(EStr.notEmpty(passWord)) options.setPassword(passWord.toCharArray());
			try {
				client = new MqttClient(host, clientId, memoryPersistence);
				if(null != callback) client.setCallback(callback);
				client.connect(options);
			} catch (Exception e) {
				log.error("MQTT连接时发生错误");
				e.printStackTrace();
			}
		}
	}
	
	/**
	 * 断开连接
	 * @return
	 */
	public void disconnect(){
		try {
			memoryPersistence.close();
		} catch (Exception e) {
			log.error("MQTT存储持久化关闭时发生错误");
			e.printStackTrace();
		}
		if(client == null) return;
		if(!client.isConnected()) return;
		try {
			client.disconnect();
			client.close();
		} catch (Exception e) {
			log.error("MQTT断开连接时发生错误");
			e.printStackTrace();
		}
	}
	
	/**
	 * 订阅主题（默认QOS：1，即至少成功接收到一次）
	 * @param topic
	 */
	public void subscribe(String topic){
		subscribe(topic, 1);
	}
	
	/**
	 * 订阅主题
	 * @param topic
	 * @param qos
	 */
	public void subscribe(String topic, int qos){
		subscribe(new String[]{ topic }, new int[]{ qos });
	}
	
	/**
	 * 订阅主题
	 * @param topic
	 * @param qos
	 */
	public void subscribe(String[] topic, int[] qos){
		try {
			client.subscribe(topic, qos);
		} catch (MqttException e) {
			log.error("订阅主题时发生错误");
			e.printStackTrace();
		}
	}
	
	/**
	 * 取消订阅主题
	 * @param topic
	 */
	public void unsubscribe(String topic){
		unsubscribe(new String[]{ topic });
	}
	
	/**
	 * 取消订阅主题
	 * @param topic
	 */
	public void unsubscribe(String[] topic){
		try {
			client.unsubscribe(topic);
		} catch (MqttException e) {
			log.error("取消订阅主题时发生错误");
			e.printStackTrace();
		}
	}
	
	/**
	 * 发布信息
	 * @param topic
	 * @param content
	 */
	public MqttMessage publish(String topic, String content){
		return publish(topic, content, publishQos, publishRetained);
	}
	
	/**
	 * 发布信息
	 * @param topic
	 * @param content
	 */
	public MqttMessage publish(String topic, String content, int qos){
		return publish(topic, content, qos, publishRetained);
	}
	
	/**
	 * 发布信息
	 * @param topic
	 * @param content
	 * @param qos
	 */
	public MqttMessage publish(String topic, String content, int qos, boolean retained){
		MqttMessage msg = new MqttMessage();
		msg.setPayload(EStr.ifEmpty(content, "").getBytes());
		msg.setQos(qos);
		msg.setRetained(retained);
		return publish(topic, msg);
	}
	
	/**
	 * 发布信息
	 * @param topic
	 * @param msg
	 */
	public MqttMessage publish(String topic, MqttMessage msg){
		try {
			client.publish(topic, msg);
		} catch (MqttPersistenceException e) {
			log.error("发布信息时发生错误");
			e.printStackTrace();
		} catch (MqttException e) {
			log.error("发布信息时发生错误");
			e.printStackTrace();
		}
		return msg;
	}

	/**
	 * 返回客户端对象
	 * @return
	 */
	public MqttClient getClient() {
		return client;
	}

	public String getHost() {
		return host;
	}

	public void setHost(String host) {
		this.host = host;
	}

	public String getUserName() {
		return userName;
	}

	public void setUserName(String userName) {
		this.userName = userName;
	}

	public String getClientId() {
		return clientId;
	}

	public void setClientId(String clientId) {
		this.clientId = clientId;
	}

	public int getPublishQos() {
		return publishQos;
	}

	public void setPublishQos(int publishQos) {
		this.publishQos = publishQos;
	}

	public boolean isPublishRetained() {
		return publishRetained;
	}

	public void setPublishRetained(boolean publishRetained) {
		this.publishRetained = publishRetained;
	}

	public int getKeepAliveInterval() {
		return keepAliveInterval;
	}

	public void setKeepAliveInterval(int keepAliveInterval) {
		this.keepAliveInterval = keepAliveInterval;
	}

	public boolean isCleanSession() {
		return cleanSession;
	}

	public void setCleanSession(boolean cleanSession) {
		this.cleanSession = cleanSession;
	}

	public String getLastWillTopic() {
		return lastWillTopic;
	}

	public void setLastWillTopic(String lastWillTopic) {
		this.lastWillTopic = lastWillTopic;
	}

	public String getLastWillPayload() {
		return lastWillPayload;
	}

	public void setLastWillPayload(String lastWillPayload) {
		this.lastWillPayload = lastWillPayload;
	}

	public int getLastWillQos() {
		return lastWillQos;
	}

	public void setLastWillQos(int lastWillQos) {
		this.lastWillQos = lastWillQos;
	}

	public Boolean getLastWillRetained() {
		return lastWillRetained;
	}

	public void setLastWillRetained(Boolean lastWillRetained) {
		this.lastWillRetained = lastWillRetained;
	}

	public MemoryPersistence getMemoryPersistence() {
		return memoryPersistence;
	}

	public void setPassWord(String passWord) {
		this.passWord = passWord;
	}
	
}
