package itez.kit.cache.ehredis;

import java.util.Set;

import com.jfinal.plugin.ehcache.IDataLoader;

import itez.kit.ELog;
import itez.kit.EUid;
import itez.kit.cache.ICache;
import itez.kit.cache.EhcacheImpl;
import itez.kit.cache.RedisImpl;
import itez.kit.log.ELogBase;
import itez.kit.serializer.ESerializerFactory;
import itez.kit.serializer.ISerializer;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import net.sf.ehcache.event.CacheEventListener;
import redis.clients.jedis.BinaryJedisPubSub;

public class EhredisImpl implements ICache, CacheEventListener {

	private static final ELogBase log = ELog.log(EhredisImpl.class);
	private EhcacheImpl ehcache;
	private RedisImpl redis;

    private String channel = "jwinner_ehredis_channel";
	private String clientId;
	private ISerializer serializer;
	
	public EhredisImpl(){
		clientId = EUid.generator();
		serializer = ESerializerFactory.me.getSerializer();
		
		ehcache = new EhcacheImpl();
		ehcache.setCacheEventListener(this);
		
		redis = new RedisImpl();
		redis.subscribe(new BinaryJedisPubSub() {
            @Override
            public void onMessage(byte[] channel, byte[] message) {
            	EhredisImpl.this.onMessage((String) serializer.deserialize(channel), serializer.deserialize(message));
            }
        }, serializer.serialize(channel));
	}

	@Override
	public boolean has(String cacheName, Object key) {
		if(ehcache.has(cacheName, key)){
			return true;
		}else if(redis.has(cacheName, key)){
			ehcache.put(cacheName, key, redis.get(cacheName, key), redis.getTtl(cacheName, key));
			return true;
		}else{
			return false;
		}
	}
	
	@Override
	public <T> T get(String cacheName, Object key) {
		if(ehcache.has(cacheName, key)){
			return ehcache.get(cacheName, key);
		}else if(redis.has(cacheName, key)){
			ehcache.put(cacheName, key, redis.get(cacheName, key), redis.getTtl(cacheName, key));
			return ehcache.get(cacheName, key);
		}else{
			return null;
		}
	}

	@Override
	public void put(String cacheName, Object key, Object value) {
		ehcache.put(cacheName, key, value);
		redis.put(cacheName, key, value);
		publishMessage(EhredisMessage.ACTION.PUT, cacheName, key);
	}

	@Override
	public void put(String cacheName, Object key, Object value, Integer liveSeconds) {
		ehcache.put(cacheName, key, value, liveSeconds);
		redis.put(cacheName, key, value, liveSeconds);
		publishMessage(EhredisMessage.ACTION.PUT, cacheName, key);
	}

	@Override
	public void remove(String cacheName, Object key) {
		ehcache.remove(cacheName, key);
		redis.remove(cacheName, key);
		publishMessage(EhredisMessage.ACTION.REMOVE, cacheName, key);
	}

	@Override
	public void removeAll(String cacheName) {
		ehcache.removeAll(cacheName);
		redis.removeAll(cacheName);
		publishMessage(EhredisMessage.ACTION.REMOVEALL, cacheName, null);
	}

	@Override
	public <T> T get(String cacheName, Object key, IDataLoader dataLoader) {
		return null;
	}

	@Override
	public <T> T get(String cacheName, Object key, IDataLoader dataLoader, Integer liveSeconds) {
		return null;
	}

	@Override
	public Integer getTtl(String cacheName, Object key) {
		if(ehcache.has(cacheName, key)){
			return ehcache.getTtl(cacheName, key);
		}else if(redis.has(cacheName, key)){
			ehcache.put(cacheName, key, redis.get(cacheName, key), redis.getTtl(cacheName, key));
			return ehcache.getTtl(cacheName, key);
		}else{
			return null;
		}
	}

	@Override
	public void setTtl(String cacheName, Object key, Integer seconds) {
		ehcache.setTtl(cacheName, key, seconds);
		redis.setTtl(cacheName, key, seconds);
	}

	@Override
	public Set<String> getKeys(String cacheName) {
		Set<String> keysInEhcache = ehcache.getKeys(cacheName);
		Set<String> keysInRedis = redis.getKeys(cacheName);
		keysInEhcache.addAll(keysInRedis);
		return keysInEhcache;
	}
	
	/**
	 * 向redis发布同步消息
	 * @param action
	 * @param cacheName
	 * @param key
	 */
    private void publishMessage(EhredisMessage.ACTION action, String cacheName, Object key) {
        EhredisMessage message = new EhredisMessage(clientId, action, cacheName, key);
        redis.publish(serializer.serialize(channel), serializer.serialize(message));
    }

	/**
	 * 接收来自redis的消息
	 * @param channel
	 * @param obj
	 */
    public void onMessage(String channel, Object obj) {
        EhredisMessage msg = (EhredisMessage) obj;
        if (clientId.equals(msg.getClientId())) return;
        log.info("执行ehcache数据同步，操作类型：{}，Cache：{}，Key：{}", msg.getAction(), msg.getCacheName(), msg.getKey());
        switch (msg.getAction()) {
            case PUT:
            	ehcache.remove(msg.getCacheName(), msg.getKey());
                break;
            case REMOVE:
            	ehcache.remove(msg.getCacheName(), msg.getKey());
                break;
            case REMOVEALL:
            	ehcache.removeAll(msg.getCacheName());
                break;
        }
    }

	/**
	 * 接收来自ehcache的消息：删除项目
	 */
	@Override
	public void notifyElementRemoved(Ehcache cache, Element element) throws CacheException {
	}

	/**
	 * 接收来自ehcache的消息：添加项目
	 */
	@Override
	public void notifyElementPut(Ehcache cache, Element element) throws CacheException {
	}

	/**
	 * 接收来自ehcache的消息：更新项目
	 */
	@Override
	public void notifyElementUpdated(Ehcache cache, Element element) throws CacheException {
	}

	/**
	 * 接收来自ehcache的消息：超期项目
	 */
	@Override
	public void notifyElementExpired(Ehcache cache, Element element) {
	}

	/**
	 * 接收来自ehcache的消息：回收项目
	 */
	@Override
	public void notifyElementEvicted(Ehcache cache, Element element) {
	}

	/**
	 * 接收来自ehcache的消息：删除全部
	 */
	@Override
	public void notifyRemoveAll(Ehcache cache) {
	}

	@Override
	public void dispose() {
	}

    @Override
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

}
