package itez.kit.cache;

import java.util.Arrays;
import java.util.Set;

import com.jfinal.plugin.ehcache.IDataLoader;
import com.jfinal.plugin.redis.Cache;
import com.jfinal.plugin.redis.IKeyNamingPolicy;
import com.jfinal.plugin.redis.serializer.FstSerializer;
import com.jfinal.plugin.redis.serializer.ISerializer;

import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.log.ELogBase;
import redis.clients.jedis.BinaryJedisPubSub;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.JedisPubSub;
import redis.clients.jedis.Protocol;
import redis.clients.jedis.exceptions.JedisConnectionException;

public class RedisImpl implements ICache {

	private static final ELogBase log = ELog.log(RedisImpl.class);
	private Cache cache;
	private JedisPoolConfig config;
	private JedisPool jedisPool;

	public RedisImpl(){
		this.config = new JedisPoolConfig();
		this.init();
	}

	public RedisImpl(JedisPoolConfig jedisPoolConfig){
		this.config = jedisPoolConfig;
		this.init();
	}
	
	private void init(){
		String host = EProp.RedisHost; //redis服务地址
		String pass = EProp.RedisPass; //redis服务密码
		int port = EProp.RedisPort; //redis服务端口
		int timeout = Protocol.DEFAULT_TIMEOUT; //redis服务链接超时
		jedisPool = new JedisPool(config, host, port, timeout, pass);
		ISerializer serializer = FstSerializer.me; //序列化服务
		IKeyNamingPolicy keyNamingPolicy = IKeyNamingPolicy.defaultKeyNamingPolicy; //默认的Key生成方案实现
		cache = new Cache("redis", jedisPool, serializer, keyNamingPolicy);
		log.info("创建默认RedisCache：redis");
	}

	private String buildKey(String cacheName, Object key) {
		return String.format("%s>%s", cacheName, key);
    }

	@Override
	public boolean has(String cacheName, Object key) {
		return cache.exists(buildKey(cacheName, key));
	}
	
	@Override
	public <T> T get(String cacheName, Object key) {
		return cache.get(buildKey(cacheName, key));
	}

	@Override
	public void put(String cacheName, Object key, Object value) {
		put(cacheName, key, value, 0);
	}

	@Override
	public void remove(String cacheName, Object key) {
		cache.del(buildKey(cacheName, key));
	}

	@Override
	public void removeAll(String cacheName) {
        Object[] keys = getKeys(cacheName).toArray();
        cache.del(keys);
	}

	@Override
	public void put(String cacheName, Object key, Object value, Integer liveSeconds) {
		if(value == null) return;
		if(liveSeconds > 0){
			cache.setex(buildKey(cacheName, key), liveSeconds, value);
		}else{
			cache.set(buildKey(cacheName, key), value);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> T get(String cacheName, Object key, IDataLoader dataLoader) {
        Object data = get(cacheName, key);
        if (data == null) {
            data = dataLoader.load();
            put(cacheName, key, data);
        }
        return (T) data;
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> T get(String cacheName, Object key, IDataLoader dataLoader, Integer liveSeconds) {
        if (liveSeconds <= 0) {
            return get(cacheName, key, dataLoader);
        }
        Object data = get(cacheName, key);
        if (data == null) {
            data = dataLoader.load();
            put(cacheName, key, data, liveSeconds);
        }
        return (T) data;
	}

	@Override
	public Integer getTtl(String cacheName, Object key) {
        Long ttl = cache.ttl(buildKey(cacheName, key));
        return ttl != null ? ttl.intValue() : null;
	}

	@Override
	public void setTtl(String cacheName, Object key, Integer seconds) {
		cache.expire(buildKey(cacheName, key), seconds);
	}

	@Override
	public Set<String> getKeys(String cacheName) {
		return cache.keys(cacheName + ">*");
	}
	
	public JedisPoolConfig getJedisPoolConfig() {
		return config;
	}

    /**
     * 发布
     *
     * @param channel
     * @param message
     */
    public void publish(String channel, String message){
        Jedis jedis = getJedis();
        try {
            jedis.publish(channel, message);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 发布
     *
     * @param channel
     * @param message
     */
    public void publish(byte[] channel, byte[] message){
        Jedis jedis = getJedis();
        try {
            jedis.publish(channel, message);
        } finally {
            returnResource(jedis);
        }
    }

    /**
     * 订阅
     *
     * @param listener
     * @param channels
     */
    public void subscribe(JedisPubSub listener, final String... channels){
    	new Thread(() -> {
            while (true) {
                Jedis jedis = getJedis();
                try {
                    jedis.subscribe(listener, channels);
                    log.warn("订阅完成，断开链接与 redis channels 的链接: {}", Arrays.toString(channels));
                    break;
                } catch (JedisConnectionException e) {
                	log.error("无法链接到 redis channels, 1秒后重试。", e);
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException ie) {
                        break;
                    }
                } finally {
                    returnResource(jedis);
                }
            }
    	}).start();
    }

    /**
     * 订阅
     *
     * @param binaryListener
     * @param channels
     */
    public void subscribe(BinaryJedisPubSub binaryListener, final byte[]... channels){
    	new Thread(() -> {
            while (true) {
                Jedis jedis = getJedis();
                try {
                    jedis.subscribe(binaryListener, channels);
                    log.warn("订阅完成，断开链接与 redis channels 的链接: {}", Arrays.toString(channels));
                    break;
                } catch (JedisConnectionException e) {
                	log.error("无法链接到 redis channels, 1秒后重试。", e);
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException ie) {
                        break;
                    }
                } finally {
                    returnResource(jedis);
                }
            }
    	}).start();
    }

    public Jedis getJedis() {
    	return cache.getJedis();
    }

    public void returnResource(Jedis jedis) {
        cache.close(jedis);
    }

}
