package itez.kit;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.codec.binary.Base64;

/**
 * 正则表达式工具类
 * 
 * @category 正则表达式工具类
 * @author netwild
 *
 */
public class ERegex {

	/**
	 * @category 编码格式
	 */
	public static final String ENCODING = "UTF-8";	
	/**
	 * @category 数字：匹配正负整数及小数
	 */
	public static final String PATTERN_NUM_ALL = "^-?\\d+(\\.\\d+)?$";
	/**
	 * @category 数字：匹配正负整数
	 */
	public static final String PATTERN_NUM_INT = "^-?\\d+$";
	/**
	 * @category 数字：匹配正整数
	 */
	public static final String PATTERN_NUM_INT1 = "^\\d+$";
	/**
	 * @category Email
	 */
	public static final String PATTERN_EMAIL = "^\\w+((-\\w+)|(\\.\\w+))*\\@[A-Za-z0-9]+((\\.|-)[A-Za-z0-9]+)*\\.[A-Za-z0-9]+$";
	/**
	 * @category IP地址：只匹配IPv4
	 */
	public static final String PATTERN_IP_ADDRESS = "^\\d{1,3}(\\.\\d{1,3}){3}$";
	/**
	 * @category 固定电话：匹配 024-12345678 或者 0419-1234567
	 */
	public static final String PATTERN_FIXED_PHONE = "^\\d{3}-\\d{8} ?\\d{4}-\\d{7}$";
	/**
	 * @category 手机号码
	 */
	public static final String PATTERN_MOBILE_PHONE = "^1[0-9]{10}$";
	/**
	 * @category 登录名称：英文字母开头，可以包含英文字符、数字、下划线、英文句号，长度3~15
	 */
	public static final String PATTERN_LOGIN_NAME = "^[a-zA-Z][a-zA-Z0-9_.]{2,15}$";
	/**
	 * @category QQ号码：从10000开始
	 */
	public static final String PATTERN_QQ = "^[1-9][0-9]{4,}$";
	/**
	 * @category 邮政编码：6位数字
	 */
	public static final String PATTERN_POST_CODE = "^[1-9]\\d{5}(?!\\d)$";
	/**
	 * @category 身份证号码：18位，包括最后一位X校验码
	 */
	public static final String PATTERN_ID_CARD = "^\\d{6}(18|19|([23]\\d))\\d{2}((0[1-9])|(10|11|12))(([0-2][1-9])|10|20|30|31)\\d{3}[0-9Xx]$";
	/**
	 * @category 变量命名规范：字母或者下划线开头
	 */
	public static final String PATTERN_VARIABLE = "^[A-Za-z_]+\\w+$";
	/**
	 * @category 中文字符
	 */
	public static final String PATTERN_CHINESE = "[\\u4e00-\\u9fa5]";
	/**
	 * @category 双字节（包括中文）
	 */
	public static final String PATTERN_DOUBLE_BYTE = "[^\\x00-\\xff]";
	
	/**
	 * 验证字符串str中是否含有与表达式pattern匹配的部分
	 * 
	 * @category 验证字符串str中是否含有与表达式pattern匹配的部分
	 * @param str
	 * @param pattern
	 * @return
	 */
	public static boolean has(String str, String pattern){
		Pattern keywordPattern = Pattern.compile(pattern);
		Matcher matcher = keywordPattern.matcher(str);
		return matcher.find();
	}
	/**
	 * 验证字符串str是否与表达式pattern完全匹配
	 * 
	 * @category 验证字符串str是否与表达式pattern完全匹配
	 * @param str
	 * @param pattern
	 * @return
	 */
	public static boolean check(String str, String pattern){
		Pattern keywordPattern = Pattern.compile(pattern);
		Matcher matcher = keywordPattern.matcher(str);
		return matcher.matches();
	}
	
	/**
	 * 从字符串中查找与表达式pattern匹配的部分
	 * 
	 * @category 从字符串中查找与表达式pattern匹配的部分
	 * @param str
	 * @param pattern
	 * @return
	 */
	public static String find(String str, String pattern){
		Pattern keywordPattern = Pattern.compile(pattern);
		Matcher matcher = keywordPattern.matcher(str);
		if(matcher.find()){
			return matcher.group();
		}
		return null;
	}
	
	/**
	 * 从字符串中查找与表达式pattern匹配的部分中的第一个子字符串
	 * 要求pattern中必须含有小括号“（）”
	 * @category 从字符串中查找与表达式pattern匹配的部分中的第一个子字符串
	 * @param str
	 * @param pattern
	 * @return
	 */
	public static String findFirst(String str, String pattern){
		Pattern keywordPattern = Pattern.compile(pattern);
		Matcher matcher = keywordPattern.matcher(str);
		if(matcher.find() && matcher.groupCount() > 0){
			return matcher.group(1);
		}
		return null;
	}

	/**
	 * 从字符串中查找与表达式pattern匹配的部分中的所有子字符串
	 * 要求pattern中必须含有小括号“（）”
	 * @category 从字符串中查找与表达式pattern匹配的部分中的所有子字符串
	 * @param str
	 * @param pattern
	 * @return
	 */
	public static String[] findAll(String str, String pattern){
		Pattern keywordPattern = Pattern.compile(pattern);
		Matcher matcher = keywordPattern.matcher(str);
		List<String> group = new ArrayList<String>();
		while(matcher.find()){
			int size = matcher.groupCount();
			for(int i=1; i<=size; i++) group.add(matcher.group(i));
		}
		return group.toArray(new String[group.size()]);
	}
	
	/**
	 * 验证模式：数字
	 * 
	 * @category 验证模式：数字
	 * @param num
	 */
	public static boolean checkNumber(String num){
		return true;
	}

	/**
	 * 验证模式：IP地址
	 * 
	 * @category 验证模式：电子信箱地址
	 * @param email
	 * @param pattern
	 * @return
	 */
	public static boolean checkIP(String ip){
		return check(ip, PATTERN_IP_ADDRESS);
	}

	/**
	 * 验证模式：电子信箱地址
	 * 
	 * @category 验证模式：电子信箱地址
	 * @param email
	 * @param pattern
	 * @return
	 */
	public static boolean checkEmail(String email){
		return has(email, PATTERN_EMAIL);
	}

	/**
	 * 验证模式：手机号码
	 * 
	 * @category 验证模式：手机号码
	 * @param phone
	 * @param pattern
	 * @return
	 */
	public static boolean checkMobilePhone(String phone){
		return has(phone, PATTERN_MOBILE_PHONE);
	}

	/**
	 * 验证模式：登录名称
	 * 
	 * @category 验证模式：登录名称
	 * @param loginName
	 * @param pattern
	 * @return
	 */
	public static boolean checkLoginName(String loginName){
		return has(loginName, PATTERN_LOGIN_NAME);
	}

	/**
	 * 验证模式：变量命名规范
	 * 
	 * @category 变量命名规范
	 * @param str
	 * @param pattern
	 * @return
	 */
	public static boolean checkVar(String var){
		return has(var, PATTERN_VARIABLE);
	}
	
	/**
	 * <p>
	 * 检测身份证号码（18位）
	 * </p>
	 * 
	 * @param idenCode
	 * @return
	 */
	public static boolean checkIdenCode(String idenCode){
		if(EStr.isEmpty(idenCode)) return false;
		if(idenCode.length() != 18) return false;
		if(!idenCode.matches(PATTERN_ID_CARD)) return false;

	    // 加权因子
	    int[] W = { 7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2 };
	    char[] idenArr = idenCode.toCharArray();
	    int sum = 0;
	    for (int i = 0; i < W.length; i++) {
	        sum += Integer.parseInt(String.valueOf(idenArr[i])) * W[i];
	    }
	    // 校验位是X，则表示10
	    if (idenArr[17] == 'X' || idenArr[17] == 'x') {
	        sum += 10;
	    } else {
	        sum += Integer.parseInt(String.valueOf(idenArr[17]));
	    }
	    // 如果除11模1，则校验通过
	    return sum % 11 == 1;
	}

	/**
	 * Url Base64编码
	 * 
	 * @category Url Base64编码
	 * @param data  待编码数据
	 * @return String 编码数据
	 * @throws Exception
	 */
	public static String encode(String data) throws Exception {
		byte[] b = Base64.encodeBase64URLSafe(data.getBytes(ENCODING));
		return new String(b, ENCODING);
	}

	/**
	 * Url Base64解码
	 * 
	 * @param data 待解码数据
	 * @return String 解码数据
	 * @throws Exception
	 */
	public static String decode(String data) throws Exception {
		byte[] b = Base64.decodeBase64(data.getBytes(ENCODING));
		return new String(b, ENCODING);
	}

	/**
	 * URL编码（utf-8）
	 * 
	 * @category URL编码（utf-8）
	 * @param source
	 * @return
	 */
	public static String urlEncode(String source) {
		String result = source;
		try {
			result = java.net.URLEncoder.encode(source, ENCODING);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return result;
	}

	/**
	 * 首字母转小写
	 * @param s
	 * @return
	 */
	public static String toLowerCaseFirstOne(String s) {
        if(Character.isLowerCase(s.charAt(0))){
        	return s;
        } else {
        	return (new StringBuilder()).append(Character.toLowerCase(s.charAt(0))).append(s.substring(1)).toString();
        }
    }
	
	/**
	 * 首字母转大写
	 * @param s
	 * @return
	 */
    public static String toUpperCaseFirstOne(String s) {
        if(Character.isUpperCase(s.charAt(0))){
        	return s;
        } else {
        	return (new StringBuilder()).append(Character.toUpperCase(s.charAt(0))).append(s.substring(1)).toString();
        }
    }
	
	/**
	 * @param args
	 */
    /*
	*/
	public static void main(String[] args) {
		String str = "aa123aaaaaaa456aaaabbbb000weaea999sss";
		String ptn = "[a-z]{2}(\\d+)";
		String ret = find(str, ptn);
		System.out.println(ret);
		String retf = findFirst(str, ptn);
		System.out.println(retf);
		String[] rets = findAll(str, ptn);
		if(rets != null) System.out.println(Arrays.toString(rets));
	}
	
}
