package itez.core.wrapper.handler;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.jfinal.handler.Handler;
import com.jfinal.kit.Ret;

import itez.core.runtime.EContext;
import itez.core.runtime.service.Ioc;
import itez.core.runtime.service.common.IComp;
import itez.core.runtime.service.common.ICompService;
import itez.core.runtime.service.common.ISiteDomain;
import itez.core.runtime.session.EAttr;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.ERegex;
import itez.kit.EStr;
import itez.kit.EWeb;
import itez.kit.log.ELogBase;

/**
 * 二级域名解析
 * 支持两种域名模式：
 * 1）前缀：（http://[domain].xxxxx.com/...），要求域名开通泛解析
 * 2）后缀：（http://www.xxxxx.com/[domain]/...），无需域名泛解析
 * 
 * @author netwild
 *
 */
public class EDomainHandler extends Handler {

	private final static ELogBase log = ELog.log(EDomainHandler.class);
	
	//默认的顶级域名（为空时根据请求地址自动获取；否则以该顶级域名生成各种链接地址串）
	//private final static String DomainName = EProp.DomainName;
	//是否开启泛域名解析（true：开启，false：关闭）
	private final static boolean DomainUsed = EProp.DomainHandle;
	//域名解析方式（前缀：prefix；后缀：postfix，其中前缀模式需服务器支持泛域名解析）
	private final static String DomainMode = EProp.DomainMode.toUpperCase();
	//默认域名（当泛域名解析关闭，或未匹配到域名时，将使用默认域名）
	private final static String DomainDefault = EProp.DomainDefault;
	
	/**
	 * 域名解析模式枚举
	 * PREFIX：前缀（http://[domain].xxxxx.com/...），要求域名开通泛解析
	 * POSTFIX：后缀（http://www.xxxxx.com/[domain]/...），无需域名泛解析
	 */
	private static enum DomainModeDict { PREFIX, POSTFIX }

	private final static String Ret_Key_Domain = "domain";
	private final static String Ret_Key_Target = "target";
	private final static String Ret_Key_Virtual = "virtual";
	
	private final ICompService compSer = Ioc.getComp();
	private final ISiteDomain siteDomainSer = Ioc.get(ISiteDomain.class);
	
	@Override
	public void handle(String target, HttpServletRequest request, HttpServletResponse response, boolean[] isHandled) {

		//静态资源直接跳出
		if (target.indexOf('.') != -1) return;
		
		//设置网页头信息，目的：禁止缓存
		/*
		request.setAttribute("decorator", "none");
		response.setHeader("Cache-Control","no-cache"); //HTTP 1.1
		response.setHeader("Pragma","no-cache"); //HTTP 1.0
		response.setDateHeader ("Expires", 0);
		*/
		
		//获得默认公司
		String domainDef = EProp.DomainDefault;
		IComp compDef = compSer.findByDomain(domainDef);
		if(compDef == null){
			String errMsg = String.format("初始化错误：默认域名[%s]所指向的单位不存在！", domainDef);
			EWeb.outPage(response, errMsg);
			log.error(errMsg);
			isHandled[0] = true;
			return;
		}
		
		//解析二级域名
		String domain, redirct;
		Boolean virtual = false;
		domain = request.getHeader(Ret_Key_Domain);
		if(EStr.notEmpty(domain)){ //首先尝试从Header中获取子域，一般用于APP环境
			redirct = target;
		}else{ //否则尝试从访问地址中获取子域，一般用于WEB环境
			Ret ret = formatDomain(target, request);
			domain = ret.getStr(Ret_Key_Domain);
			redirct = ret.getStr(Ret_Key_Target);
			virtual = ret.getBoolean(Ret_Key_Virtual);
		}
		
		//根据domain返回公司信息
		IComp comp = domain.equals(domainDef) ? compDef : compSer.findByDomain(domain);
		if(comp == null){
			String fromAddr = EWeb.getRealIp(request);
			String errMsg = String.format("请求地址[%s]中，域名[%s]所指向的单位不存在！来自：%s", EWeb.getFullUrl(request), domain, fromAddr);
			EWeb.outPage(response, errMsg);
			log.error(errMsg);
			isHandled[0] = true;
			return;
		}
		
		try {
			//业务环境参数
			EAttr attr = new EAttr(request, compDef, comp, virtual);
			//将业务环境参数存入线程
			EContext.bind(attr);
			//访问资源
			next.handle(redirct, request, response, isHandled);
		}finally {
			//销毁业务环境线程
			EContext.releaseAttr();
		}
	}

	/**
	 * 动态解析二级域名
	 * 
	 * @category 动态解析二级域名
	 * @param target
	 * @param request
	 * @return
	 */
	private Ret formatDomain(String target, HttpServletRequest request){
		Boolean virtual = false;
		String domain = DomainDefault;
		DomainModeDict domainMode = DomainModeDict.valueOf(DomainMode);
				
		if(!DomainUsed){
			return Ret.create(Ret_Key_Domain, domain).set(Ret_Key_Target, target).set(Ret_Key_Virtual, virtual);
		}else{
			String serverName = request.getServerName();
			
			/**
			 * 分析当前的主机头信息，如果是IP地址或者计算机名，只能采用后缀模式
			 */
			if(ERegex.checkIP(serverName) || serverName.indexOf(".") < 1) domainMode = DomainModeDict.POSTFIX;
			
			if(domainMode == DomainModeDict.PREFIX){
				
				/**
				 * 前缀模式（格式：http://[domain].xxxxx.com/...）
				 */
				
				String sysDomain = siteDomainSer.getBind(serverName);
				if(EStr.notEmpty(sysDomain)){ //获取到绑定的域名信息
					domain = sysDomain;
				}else{ //未找到域名绑定信息，使用默认二级域名
					String str = ERegex.findFirst(serverName, "^([^.]+)\\.");
					if(EStr.isEmpty(str)){
						log.warn("不是有效的域名地址：{}，域名模式：{}", serverName, domainMode);
					}else{
						domain = str.toLowerCase();
						
						/**
						 * 处理虚拟后缀模式
						 * 当全局配置使用前缀模式时，也可以通过后缀模式访问，主要为了解决通过“https://www”这种方式来实现子域解析
						 * 由于证书有限，一般只能开通www主机的https服务，那在前缀模式下，其他子域将无法通过https访问
						 * 因此使用：http(s)://www.xxxx.com/___domain/xxx 方式也可进行子域解析，注意要在子域前添加三个下划线
						 */
						if(domain.equals("www")){
							String sec = ERegex.findFirst(target, "^\\/___([^/]+)");
							if(EStr.notEmpty(sec)){
								domain = sec.toLowerCase();
								target = target.replaceFirst("\\/___" + domain + "(\\/?)", "$1");
								virtual = true;
							}
						}
					}
				}
			}else{
				
				/**
				 * 后缀模式（格式：http://www.xxxxx.com/[domain]/...）
				 */
				
				String str = ERegex.findFirst(target, "^\\/([^/]+)");
				if(EStr.isEmpty(str)){
					log.warn("不是有效的域名地址：{}，域名模式：{}", serverName, domainMode);
				}else{
					domain = str.toLowerCase();
					/**
					 * 后缀模式时，需要将[domain]部分从url中移除，才能匹配到Controller的路由
					 */
					target = target.replaceFirst("\\/" + domain + "(\\/?)", "$1");
				}
			}
			return Ret.create(Ret_Key_Domain, domain).set(Ret_Key_Target, target).set(Ret_Key_Virtual, virtual);
		}
	}

}
