package itez.core.util.grouping;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.beust.jcommander.internal.Lists;
import com.beust.jcommander.internal.Maps;
import com.beust.jcommander.internal.Sets;
import com.jfinal.kit.Kv;
import com.jfinal.plugin.activerecord.Record;

import itez.core.util.grouping.GroupingField.Order;
import itez.core.util.grouping.GroupingField.Type;
import itez.kit.ENum;

public abstract class GroupingPlanBase {

	protected List<Record> data;								//待分组数据
	protected Integer totalCount;								//数组总条数
	protected Double avgCount;									//分组平均数据条数
	protected Map<String, GroupingField> fields;				//字段配置（K：字段名称，V：字段配置对象）
	protected Integer countWeight;								//数量权重
	protected List<GroupingItem> groups;						//分组结果
	protected Set<String> fieldNames;							//字段列表
	protected List<GroupingField> fieldsSort;					//按权重排序的字段列表（权重越高越优先）
	
	/**
	 * 字段区间统计
	 * 
	 * Key：字段名（fieldName）
	 * Kv：字段内各区间的数量统计（例如：性别字段“sex”去重之后，有“男”和“女”两种取值（区间），则该Kv内需要分别累计两个区间各自的数量）
	 * 备注：区间统计只适用于字符型的字段，数值型字段只在Kv中累计“sum”即可。
	 */
	private Map<String, Kv> fieldEnumStis;
	
	/**
	 * 字段区间平均数统计
	 * 
	 * Key：字段名（fieldName）
	 * Kv：字段内各区间的平均数统计（例如：性别字段“sex”去重之后，有“男”和“女”两种取值（区间），则该Kv内需要分别累计两个区间各自的平均数）
	 * 备注：区间统计只适用于字符型的字段，数值型字段只在Kv中累计“avg”即可。
	 */
	private Map<String, Kv> fieldEnumAvg;
	
	public GroupingPlanBase() {
		data = Lists.newArrayList();
		totalCount = 0;
		fields = Maps.newHashMap();
		groups = Lists.newArrayList();
		countWeight = 1;
		fieldNames = Sets.newHashSet();
		fieldsSort = Lists.newArrayList();
		fieldEnumStis = Maps.newHashMap();
		fieldEnumAvg = Maps.newHashMap();
	}
	
	/**
	 * 返回待分组数据
	 * @return
	 */
	public List<Record> getData() {
		return data;
	}

	/**
	 * 设置待分组数据
	 * @param data
	 */
	public void setData(List<Record> data) {
		this.data = data;
		this.totalCount = data.size();
	}
	
	/**
	 * 添加字段配置
	 * 
	 * @param field
	 * @return
	 */
	public GroupingPlanBase putField(GroupingField field){
		fields.put(field.getName(), field);
		fieldEnumStis.put(field.getName(), Kv.create());
		fieldEnumAvg.put(field.getName(), Kv.create());
		fieldNames.add(field.getName());
		return this;
	}
	
	public GroupingPlanBase createGroup(){
		GroupingItem group = new GroupingItem(this, groups.size());
		groups.add(group);
		return this;
	}

	/**
	 * 返回字段配置
	 * 
	 * @return
	 */
	public Map<String, GroupingField> getFields() {
		return fields;
	}
	
	/**
	 * 累加字段区间数量（字符型）
	 * @param field
	 * @param enumv
	 */
	public void addFieldEnumStis(String field, String enumv){
		Kv enums = fieldEnumStis.get(field);
		if(enums.get(enumv) == null) enums.set(enumv, 0);
		enums.set(enumv, enums.getInt(enumv) + 1);
	}
	
	/**
	 * 累加字段区间数量（数值型）
	 * @param field
	 * @param fieldValue
	 */
	public void addFieldEnumStis(String field, Double fieldValue){
		Kv enums = fieldEnumStis.get(field);
		if(enums.get("sum") == null) enums.set("sum", 0d);
		Double v = enums.getAs("sum");
		enums.set("sum", ENum.add(v, fieldValue));
	}
	
	/**
	 * 将字段按照权重进行排序
	 */
	private void sortFieldByWeight() {
		for(String fieldName : fieldNames) fieldsSort.add(fields.get(fieldName));
		fieldsSort = fieldsSort.stream().sorted((a, b) -> {
			return -a.getWeight().compareTo(b.getWeight());
		}).collect(Collectors.toList());
	}
	
	/**
	 * 将数据进行排序
	 */
	protected void sortData() {
		sortFieldByWeight();
		data = data.stream().sorted(this::comparator).collect(Collectors.toList());
	}
	private int comparator(Record rec1, Record rec2) {
		for(GroupingField fieldCfg : fieldsSort){
			GroupingField.Type type = fieldCfg.getType();
			GroupingField.Order order = fieldCfg.getOrder();
			String fieldName = fieldCfg.getName();
			int compare = 0;
			if(type == Type.NUMBER){
				Double v1 = rec1.getDouble(fieldName);
				Double v2 = rec2.getDouble(fieldName);
				compare = v1.compareTo(v2);
			}else{
				String v1 = rec1.getStr(fieldName);
				String v2 = rec2.getStr(fieldName);
				compare = v1.compareTo(v2);
			}
			if(compare != 0){
				if(order == Order.ASC) return compare;
				else if(order == Order.DESC) return -compare;
			}
		}
		return 0;
	}
	
	/**
	 * 统计分组差异
	 * @return
	 */
	public GroupingPlanBase groupingStis(){
		calcEnumAvg();
		for(GroupingItem group : groups){
			group.calcEnumDiff();
		}
		return this;
	}
	
	/**
	 * 计算各字段区间平均数
	 */
	@SuppressWarnings("unchecked")
	private void calcEnumAvg(){
		Double groupCount = Double.valueOf(getGroupCount());
		avgCount = ENum.div(Double.valueOf(totalCount), groupCount);
		for(String fieldName : fieldNames){
			Kv stis = fieldEnumStis.get(fieldName);
			GroupingField.Type type = fields.get(fieldName).getType();
			if(type == Type.NUMBER){
				Double sum = stis.getAs("sum");
				Double avg = ENum.div(sum, groupCount);
				fieldEnumAvg.put(fieldName, Kv.by("avg", avg));
			}else{
				Kv avgs = Kv.create();
				Set<String> enums = stis.keySet();
				for(Object enumv : enums){
					Integer enumCnt = stis.getInt(enumv);
					avgs.set(enumv, ENum.div(Double.valueOf(enumCnt), groupCount));
				}
				fieldEnumAvg.put(fieldName, avgs);
			}
		}
	}
	
	/**
	 * 返回指定字段在各分组的差异率统计
	 * @param fieldName
	 * @return
	 */
	public List<Kv> getFieldGroupDiff(String fieldName){
		List<Kv> ret = Lists.newArrayList();
		for(GroupingItem group : groups){
			Map<String, Kv> groupDiff = group.getFieldDiffStis();
			ret.add(groupDiff.get(fieldName));
		}
		return ret;
	}
	
	/**
	 * 返回指定字段的总差异率统计
	 * @param fieldName
	 * @return
	 */
	public Kv getFieldDiff(String fieldName){
		Kv ret = Kv.create().set("fieldDiff", 0d).set("rateVal", 0d).set("ratePer", 0d);
		for(GroupingItem group : groups){
			Map<String, Kv> groupDiff = group.getFieldDiffStis();
			Kv fieldDiff = groupDiff.get(fieldName);
			Double v1 = ret.getAs("fieldDiff");
			Double v2 = ret.getAs("rateVal");
			Double v3 = ret.getAs("ratePer");
			Double d1 = fieldDiff.getAs("fieldDiff");
			Double d2 = fieldDiff.getAs("rateVal");
			Double d3 = fieldDiff.getAs("ratePer");
			ret.set("fieldDiff", ENum.add(v1, d1));
			ret.set("rateVal", ENum.add(v2, d2));
			ret.set("ratePer", ENum.add(v3, d3));
		}
		return ret;
	}
	
	/**
	 * 返回指定分组的各字段差异率统计
	 * @param fieldName
	 * @return
	 */
	public Map<String, Kv> getGroupFieldDiff(Integer groupIndex){
		return groups.get(groupIndex).getFieldDiffStis();
	}
	
	/**
	 * 返回指定分组的总差异率统计
	 * @param fieldName
	 * @return
	 */
	public Kv getGroupDiff(Integer groupIndex){
		Kv ret = Kv.create().set("fieldDiff", 0d).set("rateVal", 0d).set("ratePer", 0d);
		Map<String, Kv> groupDiff = groups.get(groupIndex).getFieldDiffStis();
		for(String fieldName : fieldNames){
			Kv fieldDiff = groupDiff.get(fieldName);
			Double v1 = ret.getAs("fieldDiff");
			Double v2 = ret.getAs("rateVal");
			Double v3 = ret.getAs("ratePer");
			Double d1 = fieldDiff.getAs("fieldDiff");
			Double d2 = fieldDiff.getAs("rateVal");
			Double d3 = fieldDiff.getAs("ratePer");
			ret.set("fieldDiff", ENum.add(v1, d1));
			ret.set("rateVal", ENum.add(v2, d2));
			ret.set("ratePer", ENum.add(v3, d3));
		}
		return ret;
	}
	
	/**
	 * 返回总差异率
	 * @return
	 */
	public Kv getDiff(){
		Kv ret = Kv.create().set("fieldDiff", 0d).set("rateVal", 0d).set("ratePer", 0d);
		for(String fieldName : fieldNames){
			Kv fieldDiff = getFieldDiff(fieldName);
			Double v1 = ret.getAs("fieldDiff");
			Double v2 = ret.getAs("rateVal");
			Double v3 = ret.getAs("ratePer");
			Double d1 = fieldDiff.getAs("fieldDiff");
			Double d2 = fieldDiff.getAs("rateVal");
			Double d3 = fieldDiff.getAs("ratePer");
			ret.set("fieldDiff", ENum.add(v1, d1));
			ret.set("rateVal", ENum.add(v2, d2));
			ret.set("ratePer", ENum.add(v3, d3));
		}
		return ret;
	}
	
	/**
	 * 返回数量权重
	 * 
	 * @return
	 */
	public Integer getCountWeight() {
		return countWeight;
	}

	/**
	 * 设置数量权重
	 * 
	 * @param countWeight
	 * @return
	 */
	public GroupingPlanBase setCountWeight(Integer countWeight) {
		this.countWeight = countWeight;
		return this;
	}

	/**
	 * 返回分组结果
	 * 
	 * @return
	 */
	public List<GroupingItem> getGroups() {
		return groups;
	}

	/**
	 * 返回分组个数
	 * @return
	 */
	public Integer getGroupCount(){
		return groups.size();
	}
	
	/**
	 * 返回数据总条数
	 * @return
	 */
	public Integer getTotalCount(){
		return totalCount;
	}
	
	/**
	 * 执行分组
	 */
	public abstract GroupingPlanBase groupingDo();

	public Map<String, Kv> getFieldEnumAvg() {
		return fieldEnumAvg;
	}

	public Double getAvgCount() {
		return avgCount;
	}

	public Set<String> getFieldNames() {
		return fieldNames;
	}

	public List<GroupingField> getFieldSort() {
		return fieldsSort;
	}
	
}
