package itez.core.util.drawing;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.List;
import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;

import com.beust.jcommander.internal.Lists;
import com.jfinal.render.Render;
import com.jfinal.render.RenderException;

import itez.kit.ERegex;
import itez.kit.EStr;

/**
 * 画板
 * @author netwild
 */
public class Board extends Render {
	
	public static enum Paper { A4h, A4v }

	private String name;
	private Color backColor;
	private String backImage;
	private Integer width;
	private Integer height;
	private List<BoardElement> elements = Lists.newArrayList();
	
	private BufferedImage outImg;
	private Graphics2D outG;
	private BufferedImage bgImg = null;
	private Integer centerX = 0;
	private Integer centerY = 0;
	
	public Board(String name, Paper paper){
		Integer[] lens = getPaperLengs(paper);
		init(name, null, null, lens[0], lens[1]);
	}
	
	public Board(String name, Color backColor, Paper paper){
		Integer[] lens = getPaperLengs(paper);
		init(name, backColor, null, lens[0], lens[1]);
	}
	
	public Board(String name, Color backColor, String backImage, Paper paper){
		Integer[] lens = getPaperLengs(paper);
		init(name, backColor, backImage, lens[0], lens[1]);
	}
	
	public Board(String name, Integer width, Integer height){
		init(name, null, null, width, height);
	}
	
	public Board(String name, Color backColor, Integer width, Integer height){
		init(name, backColor, null, width, height);
	}
	
	public Board(String name, Color backColor, String backImage){
		init(name, backColor, backImage, null, null);
	}
	
	public Board(String name, Color backColor, String backImage, Integer width, Integer height){
		init(name, backColor, backImage, width, height);
	}
	
	public Board(String name, String backImage){
		init(name, null, backImage, null, null);
	}
	
	public Board(String name, String backImage, Integer width, Integer height){
		init(name, null, backImage, null, null);
	}
	
	/**
	 * 初始化画板
	 * 
	 * @param name 文件名（不包括扩展名）
	 * @param backColor 背景颜色（支持rgba格式）
	 * @param backImage 背景图片路径（支持png/jpg/gif等格式）
	 * @param width 画板宽度（未传入时，使用背景图宽度；否则背景图按该宽度拉伸）
	 * @param height 画板高度（未传入时，使用背景图高度；否则背景图按该高度拉伸）
	 */
	protected void init(String name, Color backColor, String backImage, Integer width, Integer height){
		this.name = name;
		this.backColor = backColor;
		this.backImage = backImage;
		
		if(backImage != null){
			bgImg = getImage(backImage);
			if(bgImg == null) error("背景图片加载失败：" + backImage);
			if(width == null) width = bgImg.getWidth();
			if(height == null) height = bgImg.getHeight();
		}

		this.width = width;
		this.height = height;
		centerX = width / 2;
		centerY = height / 2;
		outImg = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
		outG = outImg.createGraphics();
		
        //呈现质量和总时间/质量折衷的控制
		outG.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
		//插值模式
		outG.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
        //图形抗锯齿
		outG.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        //字体抗锯齿
		outG.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
	}
	
	/**
	 * 添加画板元素
	 * 
	 * @param element
	 * @return
	 */
	public Board addElement(BoardElement element){
		elements.add(element);
		return this;
	}
	
	/**
	 * 绘制画板
	 */
	public Board drawing(){        
		//绘制背景色
		if(backColor != null){
			outG.setColor(backColor);
			outG.fillRect(0, 0, width, height);
		}
		//绘制背景图
		if(bgImg != null) outG.drawImage(bgImg, 0, 0, width, height, imageobserver);
		//绘制画板元素
		elements.forEach(e -> e.drawing(this, outG));
		outG.dispose();
		return this;
	}
	
	/**
	 * 导出JPG图片
	 * @param path
	 */
	public void exportJpg(String path){
		export(path, "jpg");
	}
	
	/**
	 * 导出PNG图片
	 * @param path
	 */
	public void exportPng(String path){
		export(path, "png");
	}
	
	protected Integer[] getPaperLengs(Paper paper){
		switch (paper) {
		case A4h:
			return new Integer[]{ 1650, 1180 };
		case A4v:
			return new Integer[]{ 1180, 1650 };
		default:
			return new Integer[]{ 1650, 1180 };
		}
	}
	
	protected void export(String path, String extName){
		if(EStr.isEmpty(path)) path = "/";
		if(!path.endsWith("/") && !path.endsWith("\\")) path = path.concat("/");
		String filePath = path.concat(name).concat(".").concat(extName);
        File file = new File(filePath);
        if (!file.getParentFile().exists()) file.getParentFile().mkdirs();
        try (OutputStream os = new FileOutputStream(file)) {
            ImageIO.write(outImg, extName.toUpperCase(), os);
        } catch (FileNotFoundException e) {
			error("图片输出目录不存在！");
		} catch (IOException e) {
			error("图片输出失败！");
		}
	}

	@Override
	public void render() {
        response.setHeader("Content-Disposition", getFilename());
        response.setHeader("Pragma","no-cache");
        response.setHeader("Cache-Control","no-cache");
        response.setDateHeader("Expires", 0);
        response.setContentType("image/jpeg");
        ServletOutputStream sos = null;
        try {
            sos = response.getOutputStream();
            ImageIO.write(outImg, "jpeg", sos);
        } catch (IOException e) {
            if (getDevMode()) {
                throw new RenderException(e);
            }
        } catch (Exception e) {
            throw new RenderException(e);
        } finally {
            if (sos != null) {
                try {sos.close();} catch (IOException e) {e.printStackTrace();}
            }
        }
	}
 
    protected BufferedImage getImage(String imgPath) {
        try {
        	if(ERegex.check(imgPath, "^(http|https|HTTP|HTTPS)\\:")){
        		return ImageIO.read(new URL(imgPath));
        	}else{
                return ImageIO.read(new FileInputStream(imgPath));
        	}
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }
    
    protected void error(String msg) {
		throw new RuntimeException(msg);
	}
 
    protected String getFilename() {
        try {
            String agent = request.getHeader("USER-AGENT");
            if (agent.toLowerCase().indexOf("firefox") > 0) {
                name = new String(name.getBytes("utf-8"), "iso-8859-1");
            } else {
            	name = URLEncoder.encode(name, "UTF-8");
            }
        } catch (UnsupportedEncodingException e) {
            error(e.getMessage());
        }
        return "attachment; filename=" + name + ".jpg";
    }
 
    protected ImageObserver imageobserver = new ImageObserver() {
        @Override
        public boolean imageUpdate(Image img, int infoflags, int x, int y, int width, int height) {
            return true;
        }
    };
    
    public String getName() {
		return name;
	}

	public Color getBackColor() {
		return backColor;
	}

	public String getBackImage() {
		return backImage;
	}

	public Integer getWidth() {
		return width;
	}

	public Integer getHeight() {
		return height;
	}

	public List<BoardElement> getElements() {
		return elements;
	}

	public Integer getCenterX() {
		return centerX;
	}

	public Integer getCenterY() {
		return centerY;
	}
	
	public static Color parseFromRgb(String rgb){
		if(EStr.isEmpty(rgb)) return null;
		int f = rgb.indexOf("(");
		int e = rgb.indexOf(")");
		rgb = rgb.substring(f + 1, e).replace(" ", "");
		int[] rgbs = Arrays.stream(rgb.split(",")).mapToInt(Integer::valueOf).toArray();
		if(rgbs.length == 3){
			return new Color(rgbs[0], rgbs[1], rgbs[2]);
		}else{
			return new Color(rgbs[0], rgbs[1], rgbs[2], rgbs[3]);
		}
	}
	
	public static void main(String[] args) {
//		Board board = new Board("test", new Color(255, 255, 255), "D:/data/bg.png");
//		board.addElement(new ImageElement("D:/data/logo.png", 40, 80, 40, 40));
//		board.addElement(new TextElement("对方小外前两场的上球速度打辽宁直懵逼，第四节前都没找到解决办法，上了纪委后直接掐死，这表现相当可以了对方小外前两场的上球速度打辽宁直懵逼，第四节前都没找到解决办法，上了纪委后直接掐死，这表现相当可以了对方小外前两场的上球速度打辽宁直懵逼，第四节前都没找到解决办法，上了纪委后直接掐死，这表现相当可以了666", 100, 150, 400, 200).setFontSize(12));
//		board.drawing().exportPng("D:/data");
//		Color color = parseFromRgb("rgb(0, 0, 0, 255)");
//		System.out.println(color);
	}
	
}
