package itez.core.runtime.session;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.concurrent.TimeUnit;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionContext;

import itez.core.runtime.EContext;
import itez.core.runtime.service.Ioc;
import itez.core.runtime.service.common.IUser;
import itez.kit.ECookie;
import itez.kit.EStr;
import itez.kit.EUid;

@SuppressWarnings("deprecation")
public abstract class ESessionBase implements HttpSession {

	protected String token;
	protected String id;
	protected long creationTime;
	protected volatile long lastAccessedTime;
	protected int maxInactiveInterval;
	protected ServletContext servletContext;
	protected volatile boolean invalid;
	
	protected Integer SessionTimeOutMinutes = TokenManager.SessionTimeOutMinutes;
	protected String SessionAttrName = TokenManager.SessionAttrName;
	protected String SessionCacheName = TokenManager.SessionCacheName;

	protected final static String ATTR_LOGINED = "logined";
	protected final static String ATTR_UID = "userId";
    
	//客户端类型：Plat（WEB平台方式）、APP（APP、SPA客户端方式）
	public static enum ClientType { PLAT, APP }
	protected ClientType clientType = ClientType.PLAT;
	
	/**
	 * 获取cookie的存储key
	 * @return
	 */
	protected String getCookieKey(){
		return TokenManager.me.getCookieKey();
	}
	
	/**
	 * 获取cookie的存储key
	 * @return
	 */
	protected String getCookieKey(String name){
		return TokenManager.me.getCookieKey(name);
	}
	
	/**
	 * 生成或者获取SessionId
	 * @return
	 */
	protected String generateSessionId(){
		token = TokenManager.me.getToken();
		String sessionId = TokenManager.me.parseToken(token);
		
		if(EStr.isEmpty(sessionId)){
	        sessionId = EUid.generator();
	        token = TokenManager.me.createToken(sessionId);
	        ECookie.setCookie(EContext.getResponse(), getCookieKey(), token, -1, TimeUnit.MINUTES);
		}
		
        EContext.setRequestAttr(SessionAttrName, token);
		return sessionId;
	}
	
	public String getToken(){
		return token;
	}
	
	/**
	 * 获取客户端类型
	 * @return
	 */
	public ClientType getType() {
		return clientType;
	}

	/**
	 * 设置客户端类型
	 * @param type
	 */
	public void setType(ClientType type) {
		this.clientType = type;
	}

	/**
	 * getAttribute泛型
	 * @param name
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public <T> T getAttr(String name){
		Object val = getAttribute(name);
		if(val == null) return null;
		return (T) val;
	}

	/**
	 * 设置登录状态
	 * @param logined
	 */
	public void setLogined(Boolean logined){
		setAttribute(ATTR_LOGINED, logined);
	}
	
	/**
	 * 返回登录状态
	 * @return
	 */
	public Boolean getLogined(){
		Boolean ret = getAttr(ATTR_LOGINED);
		if(ret == null) ret = false;
		return ret;
	}

	/**
	 * 设置用户ID
	 * @param uid
	 */
	public void setUid(String uid){
		setAttribute(ATTR_UID, uid);
	}

	/**
	 * 返回用户ID
	 * @param uid
	 */
	public String getUid(){
		return getAttr(ATTR_UID);
	}
	
	/**
	 * 返回用户对象实例
	 * @return
	 */
	public IUser getUser(){
		String uid = getUid();
		if(uid == null) return null;
		IUser user = Ioc.getUser().findById(uid);
		return user;
	}
	
	/**
	 * 更新Session
	 */
	public void refresh(){
		//由子类具体实现
	}

	@Override
	public HttpSessionContext getSessionContext() {
		throw new RuntimeException("getSessionContext method not finished.");
	}

	@Override
	public String[] getValueNames() {
		Enumeration<String> names = getAttributeNames();
		ArrayList<String> namel = Collections.list(names);
        return namel.toArray(new String[0]);
	}

	@Override
	public Object getValue(String name) {
		return getAttribute(name);
	}
	
	@Override
	public void putValue(String name, Object value) {
		setAttribute(name, value);
	}
	
	@Override
	public void removeValue(String name) {
		removeAttribute(name);
	}

	@Override
	public boolean isNew() {
		return Boolean.TRUE;
	}

	@Override
	public String getId() {
		return id;
	}
	
	@Override
	public long getCreationTime() {
		return creationTime;
	}

	@Override
	public long getLastAccessedTime() {
		return lastAccessedTime;
	}

	@Override
	public ServletContext getServletContext() {
		return servletContext;
	}

	@Override
	public void setMaxInactiveInterval(int interval) {
		maxInactiveInterval = interval;
	}

	@Override
	public int getMaxInactiveInterval() {
		return maxInactiveInterval;
	}

}
