package itez.core.runtime.session;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.servlet.http.HttpServletRequest;

import itez.core.runtime.modules.ModuleBase;
import itez.core.runtime.modules.ModuleManager;
import itez.core.runtime.modules.ModuleManager.ControllerConfig;
import itez.core.runtime.service.common.IComp;
import itez.core.wrapper.controller.EController;
import itez.kit.EJson;
import itez.kit.EProp;
import itez.kit.ERegex;
import itez.kit.EStr;
import itez.kit.EWeb;
import itez.kit.fileup.EFileKit;

public class EAttr {

	private boolean devMode; //是否调试模式
	
	private String web; //应用根地址，主要用于访问本地的静态资源
	private String src; //带域名格式的根地址，主要用于访问本地的服务端资源

	private IComp compDef; //默认公司信息
	private IComp comp; //当前会话的公司信息
	private String domain; //当前的二级域名（未启用二级域名解析，或未匹配到域名时，将使用默认域名）
	private Boolean virtual; //虚拟后缀模式

	private Boolean commonUseCDN; //是否启用CDN
	private String commonResLibPath; //公共静态资源访问地址（HTTP），可设置为七牛等云存储访问域名
	private String commonPicLibPath; //公共图片库访问地址（HTTP），可设置为七牛等云存储访问域名
	
	private String indexCtrl; //默认控制器
	private String indexUrl; //首页地址
	private String loginUrl; //登录页面地址
	private String errorUrl; //错误页面地址
	
	private String tempLayoutPath; //公共布局文件本地路径
	private String tempCommonPath; //公共视图文件本地路径

	private Integer page = 1; //当前页码
	
	private String clientIp = null; //客户端IP地址
	private String agent = null; //客户端user-agent信息
	private String browser = null; //浏览器名称
	private String browserVer = null; //浏览器版本
	private Boolean isAjax = false; //是否Ajax访问
	private Boolean isMobile = false; //是否为移动环境访问
	private Boolean isWx = false; //是否为微信内访问
	private Boolean isWxa = false; //是否为微信小程序内访问
	private String wxaAppId = null; //小程序的AppId

	private String moduleUrl = null; //当前模块完整路径
	private String moduleCode = null; //当前模块Code
	private String modulePath = null; //当前模块Path
	private String moduleName = null; //当前模块名称
	private String moduleView = null; //当前模块资源文件夹名称
	private String moduleAsset = null; //模块资源文件访问地址（HTTP）
	private String moduleTemp = null; //模块视图文件访问地址（HTTP）
	private String ctrl = null; //当前控制器的访问地址，不包含action，便于视图端快速组成向当前控制器交互的请求地址
	private String requestUrl = null; //本次请求的完整地址
	private String requestReferer = null; //本次请求的来源地址
	private String fileStoreUrl = null; //文件存储服务根路径
	
	private Map<String, Object> items = new ConcurrentHashMap<String, Object>(); //其他参数

	/**
	 * 构造方法
	 * @param request
	 * @param compDef
	 * @param comp
	 */
	public EAttr(HttpServletRequest request, IComp compDef, IComp comp, Boolean virtual){
		
		/**
		 * 初始化默认单位及当前单位
		 */
		this.compDef = compDef;
		this.comp = comp;
		this.virtual = virtual;
		
		/**
		 * 初始化基本参数
		 */
		devMode = EProp.DevMode;
		web = EWeb.getContextPath(request);
		domain = comp.getDomain();
		requestUrl = EWeb.getFullUrl(request);
		requestReferer = EWeb.getReferer(request);
		isAjax = EWeb.isAjax(request);
		
		/**
		 * 处理公共资源库及公共图片库的访问路径
		 */
		String httpPrefix = "^http(s?)\\:\\/\\/";
		commonUseCDN = EProp.CommonUseCDN;
		commonResLibPath = ERegex.has(EProp.CommonResLibPath, httpPrefix) ? EProp.CommonResLibPath : web + EProp.CommonResLibPath;
		commonPicLibPath = ERegex.has(EProp.CommonPicLibPath, httpPrefix) ? EProp.CommonPicLibPath : web + EProp.CommonPicLibPath;

		/**
		 * 初始化后台服务端资源根地址
		 */
		src = web;
		if(EProp.DomainHandle){
			if(EProp.DomainMode.equalsIgnoreCase("postfix")) src += "/" + domain;
			if(virtual) src += "/___" + domain;
		}
		
		/**
		 * 初始化最常用的地址参数
		 */
		ModuleBase defaultModule = ModuleManager.me.getDefaultModule();
		String defaultModulePath = defaultModule.getModulePath();
		if(EStr.notEmpty(defaultModulePath) && !defaultModulePath.startsWith("/")) defaultModulePath = "/" + defaultModulePath;
		String compIndex = comp.get("index", defaultModulePath);
		String compLogin = comp.get("login", defaultModulePath.concat("/login"));
		
		indexCtrl = String.format("%s%s", src, defaultModulePath); //默认控制器
		indexUrl = String.format("%s%s", src, compIndex); //默认页地址
		loginUrl = String.format("%s%s", src, compLogin); //登录页页地址
		errorUrl = String.format("%s%s/error", src, defaultModulePath); //错误页地址
		
		/**
		 * 初始化文件存储根路径
		 */
		fileStoreUrl = String.join("/", EFileKit.getFileStoreUrl(), domain);
		
		/**
		 * 初始化最常用的模板位置参数
		 */
		tempLayoutPath = EProp.TempLayoutPath;
		tempCommonPath = EProp.TempCommonPath;
		

		/**
		 * 设置默认模块
		 */
		moduleCode = defaultModule.getModuleCode();
		modulePath = defaultModule.getModulePath();
		moduleView = defaultModule.getModuleView();
		moduleName = defaultModule.getModuleName();
		moduleAsset = String.format("%s%s/asset", web, moduleView);
		moduleTemp = String.format("%s%s/temp", web, moduleView);
		moduleUrl = String.format("%s%s", src, modulePath);
		
		/**
		 * 初始化客户端环境参数
		 */
		clientIp = EWeb.getIpAddr(request);
		agent = request.getHeader("user-agent");
		initAgent();
		
		/**
		 * 根据requestReferer判断请求来源是否为小程序
		 */
		String wxaServer = "https://servicewechat.com/";
		if(EStr.notEmpty(requestReferer) && requestReferer.startsWith(wxaServer)){
			isWxa = true;
			wxaAppId = ERegex.findFirst(requestReferer.substring(wxaServer.length()), "^([^/]+)\\/");
		}
	}
	
	/**
	 * 切换单位，主要用于小程序端，默认都是访问www，但可根据referer中的appId重新设置单位信息
	 * @param comp
	 * @return
	 */
	public EAttr changeComp(IComp comp){
		this.domain = comp.getDomain();
		this.comp = comp;
		return this;
	}

	/**
	 * 设置Session的当前请求信息
	 * @param controller 当前请求的控制器
	 * @return
	 */
	public EAttr Caller(EController controller){
		//获得控制器类
		String clazz = controller.getClass().getName();
		//获得控制器注册信息
		ControllerConfig ctrlCfg = ModuleManager.me.getController(clazz);
		//获得控制器所在模块信息
		ModuleBase module = ModuleManager.me.getModuleByPkg(ctrlCfg.getPack());
		//初始化当前访问的模块参数
		moduleCode = module.getModuleCode();
		modulePath = module.getModulePath();
		moduleView = module.getModuleView();
		moduleName = module.getModuleName();
		moduleAsset = String.format("%s%s/asset", web, moduleView);
		moduleTemp = String.format("%s%s/temp", web, moduleView);
		moduleUrl = String.format("%s%s", src, modulePath);
		//获得控制器Key
		String ctrlKey = ctrlCfg.getKey();
		//初始化当前控制器的访问地址，便于在模板中编写指向本控制器的链接地址
		ctrl = String.format("%s%s%s", src, modulePath, ctrlKey.equals("/") ? "" : ctrlKey);
		return this;
	}

	/**
	 * 获取当前的客户端环境
	 */
	protected void initAgent(){
		if(agent == null) return;
		String str = agent.toLowerCase();
		if(ERegex.has(str, "(ucbrowser)")){ //UC
			browser = "UC";
			browserVer = ERegex.findFirst(str, "ucbrowser[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(360se|360cse|360ee)")){ //360
			browser = "360";
			browserVer = "";
		}else if(ERegex.has(str, "(micromessenger)")){ //微信
			browser = "微信";
			browserVer = ERegex.findFirst(str, "micromessenger[ \\/]([\\w.]+)");
			isWx = true;
		}else if(ERegex.has(str, "(qqbrowser)")){ //QQ
			browser = "QQ";
			browserVer = ERegex.findFirst(str, "qqbrowser[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(edge)")){ //edge
			browser = "EDGE";
			browserVer = ERegex.findFirst(str, "edge[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(msie)")){ //IE 6~10
			browser = "IE";
			browserVer = ERegex.findFirst(str, "msie[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(trident\\/[\\w.]+\\; rv)")){ //IE 11
			browser = "IE";
			browserVer = ERegex.findFirst(str, "rv[ \\/\\:]([\\w.]+)");
		}else if(ERegex.has(str, "(firefox)")){ //Firefox
			browser = "Firefox";
			browserVer = ERegex.findFirst(str, "firefox[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(chrome)")){ //Chrome
			browser = "Chrome";
			browserVer = ERegex.findFirst(str, "chrome[ \\/]([\\w.]+)");
		}else if(ERegex.has(str, "(safari)")){ //Safari
			browser = "Safari";
			browserVer = ERegex.findFirst(str, "safari[ \\/]([\\w.]+)");
		}else{
			browser = "未知";
			browserVer = "";
		}
		if(ERegex.has(str, "(mobile|phone|pod|pad)")) isMobile = true;
	}
	
	/**
	 * 检测终端是否满足要求
	 * @return
	 */
	public Boolean checkAgent(){
		if(isMobile) return true;
		if(browser.equals("IE")){
			Integer ver = Integer.parseInt(browserVer.split("\\.")[0]);
			if(ver >= 11) return true;
			else return false;
		}else if(browser.equals("Firefox") || browser.equals("Chrome") || browser.equals("Safari") || browser.equals("EDGE")){
			return true;
		}
		return false;
	}
	
	/**
	 * 转换为JSON
	 * @return
	 */
	public String getJson(){
		return EJson.toJsonFilter(this, "devMode", "web", "src", "compDef", "comp", "domain", "commonUseCDN", "commonResLibPath", "commonPicLibPath", "indexCtrl", "indexUrl", "loginUrl", "errorUrl", "page", "ctrl", "moduleCode", "moduleUrl", "modulePath", "moduleName", "moduleAsset", "moduleTemp", "isWx", "isMobile", "requestUrl", "requestReferer", "fileStoreUrl");
	}
	
	/**
	 * 添加其他参数
	 * @param key
	 * @param value
	 */
	public void setItem(String key, Object value){
		items.put(key, value);
	}
	
	/**
	 * 返回其他参数值
	 * @param key
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public <T> T getItem(String key){
		Object value = items.get(key);
		if(value == null) return null;
		return (T)value;
	}
	
	public boolean getDevMode() {
		return devMode;
	}

	public String getWeb() {
		return web;
	}

	public String getSrc() {
		return src;
	}

	public IComp getCompDef() {
		return compDef;
	}

	public IComp getComp() {
		return comp;
	}

	public String getDomain() {
		return domain;
	}

	public Boolean getCommonUseCDN() {
		return commonUseCDN;
	}

	public String getCommonResLibPath() {
		return commonResLibPath;
	}

	public String getCommonPicLibPath() {
		return commonPicLibPath;
	}

	public String getIndexCtrl() {
		return indexCtrl;
	}

	public String getIndexUrl() {
		return indexUrl;
	}

	public String getLoginUrl() {
		return loginUrl;
	}

	public String getErrorUrl() {
		return errorUrl;
	}

	public String getTempLayoutPath() {
		return tempLayoutPath;
	}

	public String getTempCommonPath() {
		return tempCommonPath;
	}

	public Integer getPage() {
		return page;
	}

	public String getClientIp() {
		return clientIp;
	}

	public String getAgent() {
		return agent;
	}

	public String getBrowser() {
		return browser;
	}

	public String getBrowserVer() {
		return browserVer;
	}

	public boolean getIsAjax() {
		return isAjax;
	}

	public boolean getIsMobile() {
		return isMobile;
	}

	public boolean getIsWx() {
		return isWx;
	}

	public boolean getIsWxa() {
		return isWxa;
	}

	public String getWxaAppId() {
		return wxaAppId;
	}

	public String getModuleUrl() {
		return moduleUrl;
	}

	public String getModuleCode() {
		return moduleCode;
	}

	public String getModulePath() {
		return modulePath;
	}

	public String getModuleName() {
		return moduleName;
	}

	public String getModuleView() {
		return moduleView;
	}

	public String getModuleAsset() {
		return moduleAsset;
	}

	public String getModuleTemp() {
		return moduleTemp;
	}

	public String getCtrl() {
		return ctrl;
	}

	public String getRequestUrl() {
		return requestUrl;
	}

	public String getRequestReferer() {
		return requestReferer;
	}

	public Boolean getVirtual() {
		return virtual;
	}

	public String getFileStoreUrl() {
		return fileStoreUrl;
	}
	
}
