package itez.core.runtime.service.interceptor.cache;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import com.jfinal.aop.Invocation;

import itez.core.runtime.EContext;
import itez.core.runtime.cache.Cache;
import itez.core.util.ECacheKit;
import itez.kit.EClass;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.ERet;
import itez.kit.EStr;
import itez.kit.log.ELogBase;
import itez.kit.El;

public class ServiceCache {

	private final static ELogBase log = ELog.log(ServiceCache.class);
	
	public static enum CacheType { ABLE, PUT, DEL, DELALL }
	
	private Method method; //被拦截的方法体对象
	private Object[] args; //执行被拦截的方法体时，所传入的参数
	private boolean hasReturn = false;
	private boolean hasCache = false;
	private boolean hasInited = false;
	
	private String defaultCacheName;
	private String defaultCacheKey;
	private String defaultCacheKeys;
	private Object methodReturn;

	private ERet context;
	private List<ServiceCacheItem> items = new ArrayList<>();
	
	public ServiceCache(Invocation inv) {
		method = inv.getMethod();
		args = inv.getArgs();
		getCacheAnno();
	}
	
	private void getCacheAnno(){
		Annotation[] annos = method.getAnnotations();
		if(annos.length > 0){
			Arrays.stream(annos).forEach(anno -> {
				if(anno instanceof Cache.able && hasReturn){ //缓存可用
					initDefaultInfo();
					Cache.able temp = (Cache.able)anno;
					String cacheName = buildCacheName(temp.cache());
					String cacheKey = buildCacheKey(temp.key());
					items.add(new ServiceCacheItem(CacheType.ABLE, cacheName, cacheKey));
				}else if(anno instanceof Cache.del){ //单条缓存移除注解
					initDefaultInfo();
					Cache.del temp = (Cache.del)anno;
					String cacheName = buildCacheName(temp.cache());
					String cacheKey = buildCacheKey(temp.key());
					String cacheKeys = buildCacheKeys(temp.keys());
					items.add(new ServiceCacheItem(CacheType.DEL, cacheName, cacheKey, cacheKeys));
				}else if(anno instanceof Cache.delAll){ //单条缓存清空注解
					initDefaultInfo();
					Cache.delAll temp = (Cache.delAll)anno;
					String cacheName = buildCacheName(temp.cache());
					items.add(new ServiceCacheItem(CacheType.DELALL, cacheName));
				}else if(anno instanceof Cache.put && hasReturn){ //插入缓存
					initDefaultInfo();
					Cache.put temp = (Cache.put)anno;
					String cacheName = buildCacheName(temp.cache());
					String cacheKey = buildCacheKey(temp.key());
					items.add(new ServiceCacheItem(CacheType.PUT, cacheName, cacheKey));
				}else if(anno instanceof Cache.dels){ //多条缓存移除注解
					initDefaultInfo();
					Cache.dels dels = (Cache.dels)anno;
					Cache.del[] delarr = dels.value();
					Arrays.stream(delarr).forEach(temp -> {
						String cacheName = buildCacheName(temp.cache());
						String cacheKey = buildCacheKey(temp.key());
						String cacheKeys = buildCacheKeys(temp.keys());
						items.add(new ServiceCacheItem(CacheType.DEL, cacheName, cacheKey, cacheKeys));
					});
				}else if(anno instanceof Cache.delAlls){ //多条缓存清空注解
					initDefaultInfo();
					Cache.delAlls dels = (Cache.delAlls)anno;
					Cache.delAll[] delarr = dels.value();
					Arrays.stream(delarr).forEach(temp -> {
						String cacheName = buildCacheName(temp.cache());
						items.add(new ServiceCacheItem(CacheType.DELALL, cacheName));
					});
				}
			});
		}
	}
	
	@SuppressWarnings("unchecked")
	private void initDefaultInfo(){
		if(hasInited) return;
		hasCache = true;
		hasReturn = !method.getReturnType().getName().equals("void");
		
		context = ERet.create(); //实参列表
		final Parameter[] params = method.getParameters(); //形参列表
		for(int i=0, len=params.length; i<len; i++){
			context.put(params[i].getName(), args[i]);
		}
		context.put("attr", EContext.getAttr());
		
		String argsKey = Arrays.stream(args).map(arg -> arg == null ? "NULL" : arg.hashCode() + "").collect(Collectors.joining(","));
		defaultCacheName = EClass.getClassFullPath(method); //EClass.getMethodFullPath(method);
		defaultCacheKey = String.format("%s:%s", EClass.getMethodFullPath(method), argsKey);
	}
	
	private String buildCacheName(String cacheName){
		return EStr.isEmpty(cacheName) ? defaultCacheName : cacheName;
	}
	
	private String buildCacheKey(String cacheKeyExpr){
		if(EStr.isEmpty(cacheKeyExpr)){
			return defaultCacheKey;
		}else{
			String key = El.exec(cacheKeyExpr, context);
			if(EStr.isEmpty(key)) log.error("解析缓存KEY表达式返回空值：{}，位置：{}", cacheKeyExpr, EClass.getMethodFullPath(method));
			return key;
		}
	}
	
	private String buildCacheKeys(String cacheKeysExpr){
		if(EStr.isEmpty(cacheKeysExpr)){
			return defaultCacheKeys;
		}else{
			String key = El.exec(cacheKeysExpr, context);
			if(EStr.isEmpty(key)) log.error("解析缓存KEY表达式返回空值：{}，位置：{}", cacheKeysExpr, EClass.getMethodFullPath(method));
			return key;
		}
	}
	
	public Object getCacheAble(){
		for(ServiceCacheItem item : items){
			if(item.getCacheType() != CacheType.ABLE) continue;
			Object val = ECacheKit.get(item);
			if(null != val){
				if(!EProp.DevMode) log.debug("命中业务缓存数据：{} - {}", item.getCacheName(), item.getCacheKey());
				return val;
			}
//			if(ECacheKit.has(item)){
//				if(!EProp.DevMode) log.debug("命中业务缓存数据：{} - {}", item.getCacheName(), item.getCacheKey());
//				return ECacheKit.get(item);
//			}
		}
		return null;
	}
	
	public void setMethodReturn(Object methodReturn) {
		this.methodReturn = methodReturn;
		items.stream().forEach(item -> {
			item.setCacheValue(this.methodReturn);
		});
	}
	
	public void putCacheAble(){
		items.stream().forEach(item -> {
			if(item.getCacheType() == CacheType.ABLE && null != item.getCacheValue()){
				ECacheKit.put(item);
				if(!EProp.DevMode) log.debug("添加新的业务缓存数据：{} - {}", item.getCacheName(), item.getCacheKey());
			}else if(item.getCacheType() == CacheType.DEL){
				if(EStr.isEmpty(item.getCacheKeys())){
					ECacheKit.remove(item);
					if(!EProp.DevMode) log.debug("删除业务缓存数据：{} - {}", item.getCacheName(), item.getCacheKey());
				}else{
					String[] keys = item.getCacheKeys().split(",");
					Arrays.stream(keys).forEach(key -> {
						ECacheKit.remove(new ServiceCacheItem(item.getCacheType(), item.getCacheName(), key.trim()));
					});
					if(!EProp.DevMode) log.debug("删除业务缓存数据：{} - {}", item.getCacheName(), item.getCacheKeys());
				}
			}else if(item.getCacheType() == CacheType.DELALL){
				ECacheKit.removeAll(item);
				if(!EProp.DevMode) log.debug("清空业务缓存数据：{}", item.getCacheName());
			}else if(item.getCacheType() == CacheType.PUT){
				ECacheKit.put(item);
				if(!EProp.DevMode) log.debug("添加新的业务缓存数据：{} - {}", item.getCacheName(), item.getCacheKey());
			}
		});
	}

	public boolean hasReturn() {
		return hasReturn;
	}

	public boolean hasCache() {
		return hasCache;
	}
	
}
