package itez.core.runtime.modules;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Lists;
import com.jfinal.config.Routes;

import itez.core.launch.JW;
import itez.core.wrapper.controller.ControllerDefine;
import itez.core.wrapper.controller.EController;
import itez.core.wrapper.routes.ERoutes;
import itez.kit.EClass;
import itez.kit.EClassSearch;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.EStr;
import itez.kit.log.ELogBase;

public class ModuleManager {

	private final static ELogBase log = ELog.log(ModuleManager.class);
	private ModuleManager(){}
	public final static ModuleManager me = new ModuleManager();

	//控制器字典（控制器类：控制器属性）
	private Map<String, ControllerConfig> ControllerDict = new HashMap<String, ControllerConfig>();
	//模块字典（模块别名：模块信息）
	private Map<String, ModuleBase> ModuleCodeDict = new HashMap<String, ModuleBase>();
	//模块包字典（包路径：模块别名）
	private Map<String, String> ModulePackDict = new HashMap<String, String>();
	//模块控制器字典（模块别名：该模块下的控制器类列表）
	private Map<String, List<String>> ModuleControllersDict = new HashMap<String, List<String>>();
	//默认模块别名，将以该模块为基础，寻找登录、首页等基本Action
	private String defaultModuleCode = "plat";

	/**
	 * 自动部署控制器
	 * @param routes
	 */
	public void deployControllers(Routes routes){
		if(!JW.getLicenseVali()) {
			log.error("当前JWinner版本未通过授权！");
			return;
		}
		this.defaultModuleCode = EProp.DefaultModuleCode;
		Set<Class<?>> controllerList = EClassSearch.getClassList(JW.ControllerPackage, true, EController.class, ControllerDefine.class);
		for(Class<?> clazz : controllerList){
			ControllerConfig ctrlCfg = new ControllerConfig(clazz);
			String pack = clazz.getPackage().getName();
			ModuleBase moduleCfg = findModuleConfig(pack);
			if(moduleCfg == null) continue;
			String moduleCode = moduleCfg.getModuleCode();
			if(EStr.notEmpty(moduleCfg.getModulePath()) && ctrlCfg.key.equals("/")) ctrlCfg.key = "";
			ModulePackDict.put(pack, moduleCfg.getModuleCode());
			ControllerDict.put(ctrlCfg.className, ctrlCfg);
			if(!ModuleControllersDict.containsKey(moduleCode)) ModuleControllersDict.put(moduleCode, new ArrayList<String>());
			ModuleControllersDict.get(moduleCode).add(ctrlCfg.className);
		}
		deployRoutes(routes);
		log.info("控制器及路由配置完成！");
	}

	/**
	 * 查找指定控制器的模块配置类
	 * @param pack
	 * @return
	 */
	public ModuleBase findModuleConfig(String pack){
		if(ModulePackDict.containsKey(pack)){
			String moduleCode = ModulePackDict.get(pack);
			return ModuleCodeDict.get(moduleCode);
		}else{
			Set<Class<?>> moduleClazzs = EClassSearch.getClassList(pack, false, ModuleBase.class, null);
			if(moduleClazzs.size() == 0){
				int lastPoint = pack.lastIndexOf(".");
				if(lastPoint > -1) return findModuleConfig(pack.substring(0, lastPoint));
				else return null;
			}else{
				ModuleBase moduleCfg = null;
				for(Class<?> clazz : moduleClazzs){
					moduleCfg = (ModuleBase) EClass.newInstance(clazz);
					String moduleCode = moduleCfg.getModuleCode();
					String modulePath = moduleCfg.getModulePath();
					String moduleView = moduleCfg.getModuleView();
					
					if(modulePath == null) modulePath = "";
					else if(EStr.notEmpty(modulePath) && !modulePath.startsWith("/")) modulePath = "/" + modulePath;
					if(moduleView == null) moduleView = "";
					else if(EStr.notEmpty(moduleView) && !moduleView.startsWith("/")) moduleView = "/" + moduleView;
					
					moduleCfg.setModulePath(modulePath);
					moduleCfg.setModuleView(moduleView);
					
					ModuleCodeDict.put(moduleCode, moduleCfg);
					ModulePackDict.put(pack, moduleCode);
				}
				return moduleCfg;
			}
		}
	}
	
	/**
	 * 注册控制器路由
	 * @param routes
	 * @param controllerClazz
	 * @param cmap
	 */
	private void deployRoutes(Routes routes){
		ModuleControllersDict.forEach((moduleCode, controllerClazz) -> {
			List<String> controllers = ModuleControllersDict.get(moduleCode);
			List<ControllerConfig> ctrlCfgs = Lists.newArrayList();
			for(String clazz : controllers) ctrlCfgs.add(ControllerDict.get(clazz));
			ERoutes moduleRoutes = new ERoutes(getModule(moduleCode), ctrlCfgs);
			routes.add(moduleRoutes);
		});
	}
	
	/**
	 * 根据控制器类返回详细信息
	 * @param controllerClazz
	 * @return
	 */
	public ControllerConfig getController(String controllerClazz){
		return ControllerDict.get(controllerClazz);
	}
	
	/**
	 * 返回当前有效的模块列表
	 * @return
	 */
	public Map<String, ModuleBase> getModules(){
		return ModuleCodeDict;
	}
	
	/**
	 * 根据模块别名，返回模块信息
	 * @param pkg
	 * @return
	 */
	public ModuleBase getModule(String moduleCode){
		return ModuleCodeDict.get(moduleCode);
	}
	
	/**
	 * 根据控制器类所在包路径，返回所属的模块信息
	 * @param pkg
	 * @return
	 */
	public ModuleBase getModuleByPkg(String pkg){
		String moduleCode = ModulePackDict.get(pkg);
		ModuleBase module = ModuleCodeDict.get(moduleCode);
		return module;
	}

	/**
	 * 获取默认的模块
	 * @return
	 */
	public ModuleBase getDefaultModule() {
		return getModule(defaultModuleCode);
	}

	/**
	 * 设置默认模块Code
	 * @param defaultModuleCode
	 */
	public void setDefaultModuleCode(String defaultModuleCode) {
		this.defaultModuleCode = defaultModuleCode;
	}
	
	public class ControllerConfig{

		String className;
		String pack;
		String key;
		String view;
		String summary;
		
		ControllerConfig(Class<?> clazz){
			ControllerDefine controllerAnno = (ControllerDefine)clazz.getAnnotation(ControllerDefine.class);
			this.className = clazz.getName();
			this.pack = clazz.getPackage().getName();
			this.key = (String)EStr.ifNull(controllerAnno.key(), "");
			this.view = (String)EStr.ifNull(controllerAnno.view(), "");
			this.summary = (String)EStr.ifNull(controllerAnno.summary(), key);
			
			if(EStr.notEmpty(key) && !key.startsWith("/")) key = "/" + key;
			if(EStr.notEmpty(view) && !view.startsWith("/")) view = "/" + view;
		}
		
		public String getClassName() {
			return className;
		}

		public String getPack() {
			return pack;
		}

		public String getKey() {
			return key;
		}

		public String getView() {
			return view;
		}

		public String getSummary() {
			return summary;
		}
	}
	
}
