package itez.ai.baidu.ocr.entity;

import java.util.List;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;

public class Result {

	private String json;
	
	/**
	 * 唯一的log id，用于问题定位
	 */
	private String log_id;
	
	/**
	 * 识别结果数，表示words_result的元素个数
	 */
	private Integer words_result_num;
	
	/**
	 * 定位和识别结果数组
	 */
	private List<WordsResult> words_result = Lists.newArrayList();
	
	/**
	 * 识别结果中每一行的置信度值，包含average：行置信度平均值，variance：行置信度方差，min：行置信度最小值
	 */
	private Probability probability;
	
	/**
	 * 图像方向，当detect_direction=true时存在。
	 * 
	 * 	-1:未定义
	 * 	 0:正向
	 * 	 1: 逆时针90度
	 * 	 2:逆时针180度
	 * 	 3:逆时针270度
	 */
	private Integer direction;
	
	/**
	 * 语言，当detect_language=true时存在。
	 * -1:未定义，
	 *  0:英文，
	 *  1: 日文，
	 *  2:韩文，
	 *  3:中文
	 */
	private Integer language;

	/**
	 * 错误代码
	 */
	private String error_code;
	
	/**
	 * 错误描述
	 */
	private String error_msg;
	
	public Result(String json){
		this.json = json;
		JSONObject object = JSON.parseObject(json);
		log_id = object.getString("log_id");
		words_result_num = object.getInteger("words_result_num");
		direction = object.getInteger("direction");
		language = object.getInteger("language");
		error_code = object.getString("error_code");
		error_msg = object.getString("error_msg");
		JSONObject proba = object.getJSONObject("probability");
		if(null != proba) probability = new Probability(proba);
		JSONArray arr = object.getJSONArray("words_result");
		if(null != arr){
			arr.stream().forEach(o -> {
				JSONObject obj = (JSONObject)o;
				words_result.add(new WordsResult(obj));
			});
		}
	}
	
	public String getJson() {
		return json;
	}
	
	public String getLog_id() {
		return log_id;
	}

	public Integer getWords_result_num() {
		return words_result_num;
	}

	public List<WordsResult> getWords_result() {
		return words_result;
	}

	public Probability getProbability() {
		return probability;
	}

	public Integer getDirection() {
		return direction;
	}

	public Integer getLanguage() {
		return language;
	}

	public String getError_code() {
		return error_code;
	}

	public String getError_msg() {
		return error_msg;
	}

	/**
	 * 字符结果
	 */
	class WordsResult {
		private String words;						//识别结果字符串
		private Location location;					//位置数组（坐标0点为左上角）
		private List<Chars> chars;					//单字符结果。当recognize_granularity=small时存在
		private Vertexes vertexes_location;			//当前为四个顶点: 左上，右上，右下，左下。当vertexes_location=true时存在
		public WordsResult(JSONObject jsonObj){
			this.words = jsonObj.getString("words");
			JSONObject loca = jsonObj.getJSONObject("location");
			if(null != loca) this.location = new Location(loca);
			JSONObject vert = jsonObj.getJSONObject("vertexes_location");
			if(null != vert) this.vertexes_location = new Vertexes(vert);
			JSONArray arr = jsonObj.getJSONArray("chars");
			if(null != arr){
				arr.stream().forEach(o -> {
					JSONObject obj = (JSONObject)o;
					chars.add(new Chars(obj));
				});
			}
		}
		public String getWords(){
			return words;
		}
		public Location getLocation() {
			return location;
		}
		public List<Chars> getChars() {
			return chars;
		}
		public Vertexes getVertexes_location() {
			return vertexes_location;
		}
	}
	
	/**
	 * 表格结果
	 */
	class TableResult {
		private TableResultData result_data;	//识别结果
		private int percent;					//表格识别进度（百分比）
		private String request_id;				//该图片对应请求的request_id
		private Integer ret_code;				//识别状态，1：任务未开始，2：进行中,3:已完成
		private String ret_msg;					//识别状态信息，任务未开始，进行中,已完成
		public TableResult(JSONObject jsonObj){
			JSONObject data = jsonObj.getJSONObject("result_data");
			if(null != data) this.result_data = new TableResultData(data);
			this.percent = jsonObj.getInteger("percent");
			this.request_id = jsonObj.getString("request_id");
			this.ret_code = jsonObj.getInteger("ret_code");
			this.ret_msg = jsonObj.getString("ret_msg");
		}
		public TableResultData getResult_data() {
			return result_data;
		}
		public int getPercent() {
			return percent;
		}
		public String getRequest_id() {
			return request_id;
		}
		public Integer getRet_code() {
			return ret_code;
		}
		public String getRet_msg() {
			return ret_msg;
		}
	}
	
	/**
	 * 表格识别内容
	 */
	class TableResultData {
		private String file_url;		//当request_type为excel时返回
		private Integer form_num;		//表格数量（可能一张图片中包含多个表格）
		private String forms;			//表格内容信息的列表
		public TableResultData(JSONObject jsonObj){
			this.file_url = jsonObj.getString("file_url");
			this.form_num = jsonObj.getInteger("form_num");
			this.forms = jsonObj.getJSONObject("forms").toJSONString();
		}
		public String getFile_url() {
			return file_url;
		}
		public Integer getForm_num() {
			return form_num;
		}
		public String getForms() {
			return forms;
		}
	}
	
	/**
	 * iOCR自定义模板返回数据结构
	 */
	class IOCRData {
		private String templateSign;	//图片分类结果对应的模板id或指定使用的模版id
		private Float scores;			//分类置信度，如果指定templateSign，则该值为1
		private Boolean isStructured;	//是否结构化成功，true为成功，false为失败
		private String logId;			//调用的日志id
		private List<IOCRDataRet> ret = Lists.newArrayList();	//识别出来的字段数组
		public IOCRData(JSONObject jsonObj){
			templateSign = jsonObj.getString("templateSign");
			scores = jsonObj.getFloat("scores");
			isStructured = jsonObj.getBoolean("isStructured");
			logId = jsonObj.getString("logId");
			JSONArray arr = jsonObj.getJSONArray("ret");
			if(null != arr){
				arr.stream().forEach(o -> {
					JSONObject obj = (JSONObject)o;
					ret.add(new IOCRDataRet(obj));
				});
			}
		}
		public String getTemplateSign() {
			return templateSign;
		}
		public Float getScores() {
			return scores;
		}
		public Boolean getIsStructured() {
			return isStructured;
		}
		public String getLogId() {
			return logId;
		}
		public List<IOCRDataRet> getRet() {
			return ret;
		}
	}
	
	/**
	 * iOCR自定义模板返回数据项
	 */
	class IOCRDataRet {
		private String word;				//识别的字符串或单字
		private String word_name;			//isStructured 为 true 时存在，表示字段的名字
		private Location location;			//字段在原图上对应的矩形框
		private Probability probability;	//字段的置信度，包括最大，最小和方差
		public IOCRDataRet(JSONObject jsonObj){
			this.word = jsonObj.getString("word");
			this.word_name = jsonObj.getString("word_name");
			JSONObject loca = jsonObj.getJSONObject("location");
			if(null != loca) location = new Location(loca);
			JSONObject prob = jsonObj.getJSONObject("probability");
			if(null != prob) probability = new Probability(prob);
		}
		public String getWord() {
			return word;
		}
		public String getWord_name() {
			return word_name;
		}
		public Location getLocation() {
			return location;
		}
		public Probability getProbability() {
			return probability;
		}
	}
	
	/**
	 * 四个顶点: 左上，右上，右下，左下
	 */
	class Vertexes {
		private Integer x;
		private Integer y;
		public Vertexes(JSONObject jsonObj){
			this.x = jsonObj.getInteger("x");
			this.y = jsonObj.getInteger("y");
		}
		public Integer getX() {
			return x;
		}
		public Integer getY() {
			return y;
		}
	}

	/**
	 * 字符位置（坐标0点为左上角）
	 */
	class Location {
		private Integer left;
		private Integer top;
		private Integer width;
		private Integer height;
		public Location(JSONObject jsonObj){
			this.left = jsonObj.getInteger("left");
			this.top = jsonObj.getInteger("top");
			this.width = jsonObj.getInteger("width");
			this.height = jsonObj.getInteger("height");
		}
		public Integer getLeft() {
			return left;
		}
		public Integer getTop() {
			return top;
		}
		public Integer getWidth() {
			return width;
		}
		public Integer getHeight() {
			return height;
		}
	}
	
	/**
	 * 单字符识别结果
	 */
	class Chars {
		private String chr;
		private Location location;
		public Chars(JSONObject jsonObj){
			this.chr = jsonObj.getString("char");
			JSONObject loca = jsonObj.getJSONObject("location");
			if(null != loca) this.location = new Location(loca);
		}
		public String getChr() {
			return chr;
		}
		public Location getLocation() {
			return location;
		}
	}
	
	/**
	 * 置信度
	 * @author Administrator
	 *
	 */
	class Probability {
		private Float average;	//行置信度平均值
		private Float min;		//行置信度最小值
		private Float variance;	//行置信度方差
		public Probability(JSONObject jsonObj){
			this.average = jsonObj.getFloat("average");
			this.min = jsonObj.getFloat("min");
			this.variance = jsonObj.getFloat("variance");
		}
		public Float getAverage() {
			return average;
		}
		public Float getMin() {
			return min;
		}
		public Float getVariance() {
			return variance;
		}
	}
	
}
