package itez.plat.base.queue;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;

import com.google.inject.Singleton;

import itez.core.runtime.service.EService;
import itez.core.runtime.service.Define;
import itez.core.runtime.service.queue.IQueueHandleService;
import itez.core.runtime.service.queue.IQueueService;
import itez.core.runtime.service.queue.QueueHandleService;
import itez.core.runtime.session.ERequestBean;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.log.ELogBase;

@Define
@Singleton
public class QueueProcessService extends EService implements IQueueService {

	private final ELogBase log = ELog.log(getClass());
	private final ExecutorService customerPool;
	private final BlockingQueue<ERequestBean> queue;
	private final Boolean queueUsed;
	private final Integer maxQueueSize;
	private final Integer threadPoolSize;
	private Boolean threadPoolInit = false;
	private Boolean customerRunningFlag = false;
	private Boolean producerRunningFlag = false;

	/**
	 * 构造队列及消费者线程池
	 */
	public QueueProcessService(){
		this.queueUsed = EProp.QueueUsed;
		this.maxQueueSize = EProp.QueueMaxSize;
		this.threadPoolSize = EProp.QueueCustomerThreadSize;
		if(queueUsed){
			queue = new LinkedBlockingQueue<ERequestBean>(maxQueueSize);
			customerPool = Executors.newFixedThreadPool(threadPoolSize);
			log.info("已创建队列，最大数量：{}", maxQueueSize);
			log.info("已创建消费者线程池，最大数量：{}", threadPoolSize);
		}else{
			queue = null;
			customerPool = null;
		}
	}
	
	private void initThreadPool(){
		threadPoolInit = true;
		for(int i=0; i<threadPoolSize; i++){
			IQueueHandleService handle = new QueueHandleService();
			customerPool.execute(handle);
		}
		customerPool.shutdown();
	}
	
	/**
	 * 将请求添加到队列
	 * @param data
	 * @throws Exception
	 */
	@Override
	public void offer(ERequestBean data) throws Exception{
		if (queue == null) throw new Exception("还未创建队列");
		if (queue.size() >= maxQueueSize) throw new Exception("队列已满");
		if(queue.offer(data)){
			producerRunningFlag = true;
			if(!threadPoolInit) initThreadPool();
		}else{
			throw new Exception("插入队列失败");
		}
	}
	
	/**
	 * 停止生产
	 */
	@Override
	public void stopProduce(){
		producerRunningFlag = false;
		customerPool.shutdown();
	}

	@Override
	public Boolean getQueueUsed() {
		return queueUsed;
	}

	@Override
	public int getQueueSize(){
		return queue == null ? 0 : queue.size();
	}

	@Override
	public BlockingQueue<ERequestBean> getQueue() {
		return queue;
	}

	@Override
	public Boolean getCustomerRunningFlag() {
		return customerRunningFlag;
	}

	@Override
	public void setCustomerRunningFlag(Boolean customerRunningFlag) {
		this.customerRunningFlag = customerRunningFlag;
	}

	@Override
	public Boolean getProducerRunningFlag() {
		return producerRunningFlag;
	}

	@Override
	public void setProducerRunningFlag(Boolean producerRunningFlag) {
		this.producerRunningFlag = producerRunningFlag;
	}
	
}
