package itez.plat.base.controller;

import com.google.inject.Inject;
import com.jfinal.aop.Before;
import com.jfinal.aop.Clear;
import com.jfinal.ext.interceptor.NotAction;
import com.jfinal.upload.UploadFile;

import itez.core.runtime.auth.AuthRequire;
import itez.core.runtime.auth.AuthCode;
import itez.core.runtime.service.common.ICompService;
import itez.core.runtime.service.common.IUser;
import itez.core.runtime.service.common.IUserService;
import itez.core.runtime.session.ESessionBase;
import itez.core.runtime.session.TokenManager;
import itez.core.wrapper.controller.ControllerDefine;
import itez.core.wrapper.controller.EController;
import itez.kit.EProp;
import itez.kit.ERet;
import itez.kit.EStr;
import itez.kit.El;
import itez.kit.fileup.EFileKit;
import itez.plat.base.model.Comp;
import itez.plat.base.util.ConfigKit;

/**
 * 平台默认控制器
 * 只实现最通用的几个入口Action
 * 
 * @author netwild
 *
 */
@ControllerDefine(key = "/", summary = "平台首页", view = "/")
public class IndexController extends EController {

	@Inject
	ICompService compService;

	@Inject
	IUserService userService;
	
	/**
	 * 平台首页
	 */
	@AuthRequire.Logined
	public void index(){
		Comp comp = (Comp)attr().getComp();
		//如果当前单位设置了自定义首页，则直接跳转
		if(EStr.notEmpty(comp.getIndex())){
			redirect(attr().getIndexUrl());
			return;
		}
		String tempPath = "";
		String selfTempExpr = comp.getSelfTempExp();
		if(EStr.notEmpty(selfTempExpr)){
			ERet context = ERet.create("attr", attr()).set("user", session().getUser()).set("comp", comp);
			tempPath = El.me.exec(selfTempExpr, ERet.create("attr", context));
		}
		if(EStr.isEmpty(tempPath)) tempPath = "index.html";
		if(comp.getSelfTemp() == 1) tempPath = String.format("%s/%s", comp.getDomain(), tempPath);
		render(tempPath);
	}
	
	/**
	 * 测试页01
	 */
	@AuthRequire.Logined
	public void test01(){
		render("test/01.html");
	}
	
	/**
	 * 测试页02
	 */
	@AuthRequire.Logined
	public void test02(){
		render("test/02.html");
	}
	
	/**
	 * 平台登录页
	 */
	@Clear
	public void login(){
		String from = getPara("from"); //登入前的页面，用于登录成功后返回
		if(EStr.notEmpty(from)) setAttr("previousPageUrl", from);
		render(EProp.TempLoginPath);
	}

	/**
	 * 验证码
	 */
	@Clear
	public void valicode(){
		renderCaptcha();
	}
	
	/**
	 * 登录逻辑
	 * @return
	 */
	@Before(NotAction.class)
	private ERet loginLogic(ESessionBase.ClientType type){
		if(type == ESessionBase.ClientType.PLAT && ConfigKit.getBool("CaptchaAble") && !validateCaptcha("valicode")){
			return ERet.fail("验证码错误");
		}
		String loginName = getPara("loginName");
		String loginPass = getPara("loginPass");
		if(EStr.isEmpty(loginName) || EStr.isEmpty(loginPass)){
			return ERet.fail("用户名及密码不允许为空");
		}
		ERet ret = ERet.create();
		IUser user = userService.findByLogin(attr().getDomain(), loginName, loginPass);
		if(null == user){
			return ret.setFail().setMsg("用户名或密码错误");
		}
		if(user.getUsed() != 1){
			return ret.setFail().setMsg("用户已被注销");
		}
		ret.setOk().set("user", user);
		return ret;
	}
	
	/**
	 * 平台登录验证（Web方式）
	 */
	@Clear
	public void login4plat(){
		String previousPageUrl = getPara("previousPageUrl");
		ERet ret = loginLogic(ESessionBase.ClientType.PLAT);
		if(ret.isFail()){
			setFlashMsg(ret.getMsg());
			setFlashAttr("previousPageUrl", previousPageUrl);
			redirect(attr().getLoginUrl());
		}else if(ret.isOk()){
			IUser user = ret.getAs("user");
			session().setLogined(true);
			session().setUid(user.getId());
			String url = EStr.ifEmpty(previousPageUrl, attr().getIndexUrl());
			redirect(url);
		}
	}
	
	/**
	 * 平台登录验证（App方式）
	 */
	@Clear
	public void login4app(){
		ERet ret = loginLogic(ESessionBase.ClientType.APP);
		if(ret.isOk()){
			IUser user = ret.getAs("user");
			String tokenCode = TokenManager.me.createToken(user.getId()).toString();
			ret.set(TokenManager.TokenParamName, tokenCode);
		}
		renderJson(ret);
	}
	
	/**
	 * 注销身份
	 */
	@Clear
	public void logout(){
		session().setLogined(false);
		session().setUid(null);
		setFlashMsg("身份已注销，请重新登录！");
		redirect(attr().getLoginUrl());
	}
	
	/**
	 * 错误页
	 */
	@Clear
	public void error(){
		AuthCode authCode = AuthCode.valueOf(getPara("authCode"));
		String errMsg = getPara("errMsg");
		String helpUrl = getPara("helpUrl");
		if(EStr.isEmpty(errMsg)) renderErr(authCode);
		else if(EStr.isEmpty(helpUrl)) renderErrMsg(authCode, errMsg);
		else renderErrHelp(authCode, errMsg, helpUrl);
		return;
	}
	
	/**
	 * 自定义编辑器上传
	 */
	@Clear
	public void ueditorUpload(){
		String action = getPara("action", "config");
		/**
		 * 百度编辑器加载出按钮图标前 会将所有控件的路径 先通过config.json
		 * 文件加载出来(包括上传图片路径，视频路径等路径都是通过config.json 文件读取的)
		 * 所以某些控件点击不了 是因为 config.json文件没有找到 或者是文件里面的路径有问题
		 */
		if("config".equals(action)){
			render("/base/asset/js/ueditorConfig.json");
		}else{
			UploadFile upfile = getFile("upfile");
			if(upfile != null){
				String fileName = upfile.getFileName();
				String url = EFileKit.upload(upfile.getFile());
				ERet ret = ERet.create();
				ret.set("state", "SUCCESS");
				ret.set("url", url);
				ret.set("title", fileName);
				ret.set("original", upfile.getOriginalFileName());
				ret.set("type", fileName.split("\\.")[1]);
				ret.set("size", upfile.getFile().length());
				renderJson(ret);
			}
		}
	}

}
