package cn.cloudself.query.util

import cn.cloudself.query.config.QueryProConfig
import java.math.BigDecimal
import java.text.SimpleDateFormat
import java.util.Date

object SqlLog {
    fun <P: Any?, R> withQuery(logger: Log, sql: String, params: Array<P>, query: () -> R): R {
        val result = try {
            query()
        } catch (e: Exception) {
            doLog(logger, sql, params, e.javaClass.simpleName)
            throw e
        }
        doLog(logger, sql, params, result)
        return result
    }

    private fun <P: Any?> doLog(logger: Log, sql: String, params: Array<P>, result: Any?) {
        val printLog = QueryProConfig.final.printLog()
        val level = QueryProConfig.final.printLogLevel()

        if (!printLog && level == LogLevel.DEBUG) {
            return
        }

        val printLargeElementWholly = QueryProConfig.final.printLargeElementWholly()

        val (resultSize, firstResult) = when (result) {
            is List<*> -> result.size to result.getOrNull(0)
            is Collection<*> -> result.size to null
            is Array<*> -> result.size to result.getOrNull(0)
            else -> 1 to result
        }

        if (!printLog) {
            logger.debug("{0}\n{1}", getCallInfo(), sql)
            logger.debug(params)
            if (resultSize > 256 && !printLargeElementWholly) {
                logger.debug("very large result, type: {0}, size: {1}, first result(maybe): {2}", result?.javaClass?.name, resultSize, firstResult)
            } else {
                logger.debug(result)
            }
            return
        }

        val printCallByInfo = QueryProConfig.final.printCallByInfo()
        val printResult = QueryProConfig.final.printResult()

        val isBatch = params.isNotEmpty() && (params[0] is Array<*>)
        val autoReplaceParams = !isBatch && sql.count { it == '?' } == params.size
        val paramsReplacedSql = if (autoReplaceParams) {
            var i = 0
            sql.trim().replace(Regex("\\?")) {
                when (val param = params[i++]) {
                    is String -> "'$param'"
                    is Date -> "'${SimpleDateFormat("yyyy-MM-dd hh:mm:ss").format(param)}'"
                    is Boolean, is Number, null -> param.toString()
                    else -> "'$param'"
                }
            }
        } else {
            sql.trim()
        }

        logger.level(
            level,
            """{0}⭣⭣⭣
                |--------------------------------
                |{1}
                |{2}
                |result(size: {3}):
                |{4}
                |--------------------------------
            """.trimMargin(),

            if (printCallByInfo) getCallInfo() else "",

            if(paramsReplacedSql.startsWith("/*BATCH MODE ") && paramsReplacedSql.length > 256 && !printLargeElementWholly)
                paramsReplacedSql.substring(0, 255) + "..."
            else
                paramsReplacedSql,

            if (autoReplaceParams) ""
            else if (!isBatch) params.contentToString()
            else if (params.size > 32 && !printLargeElementWholly) "very large params, size: ${params.size}, first: ${params[0]}\n"
            else params.joinToString("\n") { (it as Array<*>).contentToString() },

            resultSize,

            if (resultSize > 128 && !printLargeElementWholly) "very large result, type: ${result?.javaClass?.name}, first: $firstResult" else result
        )

        if (!printResult && QueryProConfig.final.printLogLevel() != LogLevel.DEBUG) {
            logger.debug("result: {0}", result)
        }
    }
}