package cn.cloudself.query.util

import cn.cloudself.query.config.QueryProConfig
import cn.cloudself.query.exception.UnSupportException
import javax.persistence.Column
import javax.persistence.Id
import javax.persistence.Table

object ClassParser {
    private const val keywordsSuffix = "_column"
    @JvmStatic
    fun parse(clazz: Class<*>): ParsedClass {
        return caches.getOrPut(clazz) {
            parseClass(clazz)
        }
    }

    private fun parseClass(clazz: Class<*>): ParsedClass {
        val columns = mutableMapOf<String, ParsedColumn>()
        var idColumn: String? = null
        var idColumnMay: String? = null
        var idColumnType: Class<*>? = null

        val tableAnnotation: Table? = try {
            clazz.getAnnotation(Table::class.java)
        } catch (e: Exception) {
            null
        }
        val dbNameForTable = tableAnnotation?.name ?: to_snake_case(clazz.name)

        var classOrSuperClass: Class<*>? = clazz
        while (classOrSuperClass != null) {
            for (field in classOrSuperClass.declaredFields) {
                val fieldName = field.name
                if (QueryProConfig.final.shouldIgnoreFields().contains(fieldName)) {
                    continue
                }

                val idAnnotation: Id? = field.getAnnotation(Id::class.java)
                val columnAnnotation: Column? = field.getAnnotation(Column::class.java)
                val dbName = columnAnnotation?.name ?: to_snake_case(fieldName)
                if (idAnnotation != null) {
                    if (idColumn != null) {
                        throw UnSupportException("不支持联合主键")
                    }
                    idColumn = dbName
                    idColumnType = field.type
                }
                if (dbName == "id") {
                    idColumnMay = dbName
                    idColumnType = field.type
                }

                val setterMethodName = "set${Character.toUpperCase(fieldName[0])}${fieldName.substring(1)}"
                val getterMethodName = "get${Character.toUpperCase(fieldName[0])}${fieldName.substring(1)}"

                val setter = try {
                    clazz.getMethod(setterMethodName, field.type)
                } catch (e: Exception) {
                    null
                } ?: if (fieldName.endsWith(keywordsSuffix))
                    try {
                        clazz.getMethod(setterMethodName.substring(0, setterMethodName.length - 7), field.type)
                    } catch (e: Exception) {
                        null
                    }
                else
                    null
                val getter = try {
                    clazz.getDeclaredMethod(getterMethodName)
                } catch (e: Exception) {
                    null
                } ?: try {
                    clazz.getMethod(getterMethodName)
                } catch (e: Exception) {
                    null
                } ?: if (fieldName.endsWith(keywordsSuffix))
                    try {
                        clazz.getDeclaredMethod(getterMethodName.substring(0, getterMethodName.length - 7))
                    } catch (e: Exception) {
                        null
                    }
                else
                    null

                columns[dbName] = ParsedColumn(
                    javaName = fieldName,
                    javaType = field.type,
                    setter = { o, v ->
                        if (setter != null) {
                            setter.invoke(o, v)
                        } else {
                            field.trySet(o, v).getOrElse { throw UnSupportException("无法访问私有且无setter的属性 {0}", fieldName) }
                        }
                    },
                    getter = { o ->
                        if (o is Map<*, *>) {
                            if (o.contains(fieldName)) {
                                return@ParsedColumn o[fieldName]
                            }
                            return@ParsedColumn o[dbName]
                        }
                        return@ParsedColumn field.tryGet(o).getOrElse {
                            if (getter == null) {
                                throw UnSupportException("无法访问私有且无getter的属性 {0}", fieldName)
                            }
                            return@ParsedColumn getter.invoke(o)
                        }
                    },
                    dbName = dbName,
                )
            }
            classOrSuperClass = classOrSuperClass.superclass
        }

        return ParsedClass(dbNameForTable, columns, idColumn ?: idColumnMay, idColumnType)
    }

    private val caches = mutableMapOf<Class<*>, ParsedClass>()

    data class ParsedColumn(
        val javaName: String,
        val javaType: Class<*>,
        val setter: (o: Any?, v: Any?) -> Unit,
        val getter: (o: Any?) -> Any?,
        val dbName: String,
    )

    data class ParsedClass(
        val dbName: String,
        val columns: Map<String, ParsedColumn>,
        val idColumn: String?,
        val idColumnType: Class<*>?,
    ) {
        fun getColumnDbFieldName(fieldName: String) = columns[fieldName]
        fun getColumnByJavaPropertyName(propertyName: String) = columns.values.find { it.javaName == propertyName }
    }
}
