package cn.cloudself.query.util;

import cn.cloudself.query.config.QueryProConfig;
import cn.cloudself.query.exception.UnSupportException;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.InvocationTargetException;
import java.util.*;

/**
 * Bean代理，
 * 支持生成三种类型的数据:
 * <li>Map</li>
 * <li>基本对象</li>
 * <li>JavaBean</li>
 * <i>BeanProxy</i>的主要目的是虚拟出一个对象.
 * <br/>
 *
 * 使用{@link BeanProxy}.{@link BeanProxy#fromClass(Class)}或{@link BeanProxy}.{@link BeanProxy#fromBean}构造该对象
 * 使用{@link BeanProxy}.{@link BeanProxy#newInstance()}创建临时对象{@link BeanInstance}
 * 使用{@link BeanInstance}.{@link BeanInstance#setProperty}设置目标对象的属性
 * 使用{@link BeanInstance}.{@link BeanInstance#getPropertyType}获取目标某字段的类型
 * 使用{@link BeanInstance}.{@link BeanInstance#toBean}将临时对象转为目标对象
 */
public abstract class BeanProxy<B> {
    private static final Map<Class<?>, BeanProxy<?>> beanProxyCaches = new HashMap<>();
    private static final Log logger = LogFactory.getLog(BeanProxy.class);

    @NotNull
    public static <B> BeanProxy<B> fromClass(@NotNull final Class<B> clazz) {
        final BeanProxy<?> cachedBeanProxy = beanProxyCaches.get(clazz);
        if (cachedBeanProxy != null) {
            //noinspection unchecked
            return (BeanProxy<B>) cachedBeanProxy;
        }

        final BeanProxy<B> beanProxy;
        if (Map.class.isAssignableFrom(clazz)) {
            beanProxy = new MapProxy<>(clazz);
        } else if (QueryProConfig.computed.supportedColumnType().stream().anyMatch(it -> it.isAssignableFrom(clazz))) {
            beanProxy = new BasicTypeProxy<>(clazz);
        } else {
            beanProxy = new JavaBeanProxy<>(clazz);
        }

        beanProxyCaches.put(clazz, beanProxy);
        return beanProxy;
    }

    /**
     * 构造一个BeanInstance
     *
     * @param bean 只能为`Java Bean`, 不能为`Map`, 如需为`Map`, 参考[fromBean(R, Class<Any>)]
     */
    public static <B> BeanInstance<B> fromBean(B bean) {
        //noinspection unchecked
        final Class<B> clazz = (Class<B>) bean.getClass();
        return new BeanInstance<>(bean, fromClass(clazz));
    }

    /**
     * 构造一个BeanInstance
     *
     * @param bean 可以为Map或者JavaBean,
     * 为Map时，BeanInstance.getParsedClass无法调用，如需要可以手动调用`BeanProxy.fromClass(refer).getParsedClass()`获取
     *
     * @param refer 该Map参考的`Bean Class`. (目前仅`getPropertyType`方法使用到, V1.1.4)
     */
    public static <B> BeanInstance<B> fromBean(B bean, Class<?> refer) {
        //noinspection unchecked
        final Class<B> clazz = (Class<B>) bean.getClass();
        return new BeanInstance<B>(bean, fromClass(clazz), refer);
    }

    public static class BeanInstance<B> {
        private final B instance;
        private final BeanProxy<B> proxy;
        private final Class<?> clazz;

        public BeanInstance(@NotNull B instance, @NotNull BeanProxy<B> proxy) {
            this.instance = instance;
            this.proxy = proxy;
            this.clazz = null;
        }

        public BeanInstance(@NotNull B instance, @NotNull BeanProxy<B> proxy, @Nullable Class<?> clazz) {
            this.instance = instance;
            this.proxy = proxy;
            this.clazz = clazz;
        }

        /**
         * 设置一个属性，该方法不会自动转换value的类型。
         * <br/>
         * 如代理对象的`id`是`Long`或者`Int`类型，现在有一个`BigDecimal`类型的值，如何将这个`BigDecimal`设置到`id`上呢？
         * <br/>
         * 参考如下代码：
         * <code><pre>
         *     BeanInstance&lt;Object> instance = BeanProxy.fromBean(bean);
         *     String field_name = "id";
         *
         *     Object typedValue = new ObjectMapper().convertValue(bigDecimalId, instance.getPropertyType(field_name));
         *     instance.setProperty(field_name, typedValue);
         * </pre></code>
         * ```java
         * ```
         */
        @NotNull
        public BeanInstance<B> setProperty(@NotNull String db_field_name, @Nullable Object value) {
            proxy.setProp(instance, db_field_name, value);
            return this;
        }

        /**
         * get
         */
        @Nullable
        public Object getProperty(@NotNull String db_field_name) {
            return proxy.getProp(instance, db_field_name);
        }

        /**
         * 获取某属性的类型
         */
        @Nullable
        public Class<?> getPropertyType(@NotNull String db_field_name) {
            if (clazz != null) {
                if (clazz != proxy. clazz) {
                    final ClassParser.ParsedColumn column = ClassParser.parse(clazz).getColumnDbFieldName(db_field_name);
                    if (column == null) {
                        return null;
                    }
                    return column.getJavaType();
                }
            }
            if (instance instanceof Map) {
                final Object val = ((Map<?, ?>) instance).get(db_field_name);
                if (val == null) {
                    return null;
                }
                return val.getClass();
            }
            return proxy.getPropertyType(db_field_name);
        }

        /**
         * 获取bean
         */
        @Nullable
        public B toBean() {
            return proxy.toRes(instance);
        }

        /**
         * 获取bean的类型
         */
        public Class<B> getBeanType(){
            return proxy.clazz;
        }

        public ClassParser.ParsedClass getParsedClass() {
            return proxy.getParsedClass();
        }

        /**
         * 以Map.Entry的形式迭代。
         */
        @NotNull
        public Iterator<Map.Entry<String, Object>> iterator() {
            if (instance instanceof Map) {
                final Map<?, ?> mapInstance = (Map<?, ?>) instance;
                final Iterator<? extends Map.Entry<?, ?>> iterator = mapInstance.entrySet().iterator();
                return new Iterator<Map.Entry<String, Object>>() {
                    @Override
                    public boolean hasNext() {
                        return iterator.hasNext();
                    }

                    @Override
                    public Map.Entry<String, Object> next() {
                        final Map.Entry<?, ?> next = iterator.next();
                        final Object key = next.getKey();
                        if (key instanceof String) {
                            return new AbstractMap.SimpleEntry<>((String) key, next.getValue());
                        }
                        throw new UnSupportException("不支持非Map<String, ?>类型的Map");
                    }
                };
            }
            final Map<String, ClassParser.ParsedColumn> columns = getParsedClass().getColumns();
            final Iterator<Map.Entry<String, ClassParser.ParsedColumn>> iterator = columns.entrySet().iterator();
            return new Iterator<Map.Entry<String, Object>>() {
                @Override
                public boolean hasNext() {
                    return iterator.hasNext();
                }

                @Override
                public Map.Entry<String, Object> next() {
                    final Map.Entry<String, ClassParser.ParsedColumn> next = iterator.next();
                    return new AbstractMap.SimpleEntry<>(next.getKey(), next.getValue().getGetter().invoke(instance));
                }
            };
        }
    }

    private static class MapProxy<B> extends ProxyTemplate<B, Map<String, Object>> {
        protected MapProxy(@NotNull Class<B> clazz) {
            super(clazz);
        }

        @NotNull
        @Override
        protected Map<String, Object> createInstance() {
            if (LinkedHashMap.class.isAssignableFrom(clazz)) {
                return new LinkedHashMap<String, Object>();
            }
            if (HashMap.class.isAssignableFrom(clazz)) {
                return new HashMap<String, Object>();
            }
            throw new UnSupportException("不支持LinkedHashMap与HashMap<String, Object>以外的Map");
        }

        @NotNull
        @Override
        protected ClassParser.ParsedClass getParsedClass() {
            throw new UnSupportException("不支持Map转properties");
        }

        @Override
        protected void setProperty(@NotNull Map<String, Object> obj, @NotNull String prop, @Nullable Object val) {
            obj.put(prop, val);
        }

        @Nullable
        @Override
        protected Object getProperty(@NotNull Map<String, Object> obj, @NotNull String prop) {
            return obj.get(prop);
        }

        @Nullable
        @Override
        protected Class<?> getPropertyType(@NotNull String prop) {
            return null;
        }

        @Nullable
        @Override
        protected B toResult(@NotNull Map<String, Object> obj) {
            //noinspection unchecked
            return (B) obj;
        }
    }

    private static class BasicTypeProxy<B> extends ProxyTemplate<B, Ref<B>> {

        protected BasicTypeProxy(@NotNull Class<B> clazz) {
            super(clazz);
        }

        @NotNull
        @Override
        protected Ref<B> createInstance() {
            return new Ref<>(null);
        }

        @NotNull
        @Override
        protected ClassParser.ParsedClass getParsedClass() {
            throw new UnSupportException("不支持基本类型转properties");
        }

        @Override
        protected void setProperty(@NotNull Ref<B> obj, @NotNull String prop, @Nullable Object val) {
            //noinspection unchecked
            obj.setValue((B) val);
        }

        @Nullable
        @Override
        protected Object getProperty(@NotNull Ref<B> obj, @NotNull String prop) {
            return obj.getValue();
        }

        @Nullable
        @Override
        protected Class<?> getPropertyType(@NotNull String prop) {
            return clazz;
        }

        @Nullable
        @Override
        protected B toResult(@NotNull Ref<B> obj) {
            return obj.getValue();
        }
    }

    private static class JavaBeanProxy<B> extends ProxyTemplate<B, B> {
        private final ClassParser.ParsedClass parsedClass;
        private final Map<String, ClassParser.ParsedColumn> columns;

        protected JavaBeanProxy(@NotNull Class<B> clazz) {
            super(clazz);
            parsedClass = ClassParser.parse(clazz);
            columns = parsedClass.getColumns();
        }

        @NotNull
        @Override
        protected B createInstance() {
            try {
                return clazz.getDeclaredConstructor().newInstance();
            } catch (InstantiationException | IllegalAccessException | InvocationTargetException |
                     NoSuchMethodException e) {
                throw new UnSupportException(e, "{0} 没有找到无参构造函数，该类是一个JavaBean吗, " +
                        "对于Kotlin，需使用kotlin-maven-noarg生成默认的无参构造函数" +
                        "或在生成工具中配置QueryProFileMaker.disableKtNoArgMode()来生成默认的构造函数", clazz);
            }
        }

        @NotNull
        @Override
        protected ClassParser.ParsedClass getParsedClass() {
            return parsedClass;
        }

        @Override
        protected void setProperty(@NotNull B obj, @NotNull String prop, @Nullable Object val) {
            final ClassParser.ParsedColumn column = columns.get(prop);
            if (column == null) {
                logger.warn("给定的类型${clazz.simpleName}中不存在字段$p, 该字段的赋值已跳过。");
                return;
            }
            column.getSetter().invoke(obj, val);
        }

        @Nullable
        @Override
        protected Object getProperty(@NotNull B obj, @NotNull String prop) {
            final ClassParser.ParsedColumn column = columns.get(prop);
            if (column == null) {
                logger.warn("没有找到该字段 $p");
                return null;
            }
            return column.getGetter().invoke(obj);
        }

        @Nullable
        @Override
        protected Class<?> getPropertyType(@NotNull String prop) {
            return columns.get(prop).getJavaType();
        }

        @Nullable
        @Override
        protected B toResult(@NotNull B obj) {
            return obj;
        }
    }


    protected final Class<B> clazz;
    public abstract BeanInstance<B> newInstance();
    /**
     * 获取属性列表
     */
    @NotNull
    protected abstract ClassParser.ParsedClass getParsedClass();
    /**
     * 设置属性
     */
    protected abstract void setProp(@NotNull Object obj, @NotNull String prop, @Nullable Object val);
    /**
     * 获取属性
     */
    @Nullable
    protected abstract Object getProp(@NotNull Object obj, @NotNull String prop);
    /**
     * 获取某属性的类型
     * 当clazz为Map类型时，返回null
     */
    @Nullable
    protected abstract Class<?> getPropertyType(@NotNull String prop);
    /**
     * 转为目标对象
     */
    @Nullable
    protected abstract B toRes(@NotNull Object obj);

    protected BeanProxy(Class<B> clazz) {
        this.clazz = clazz;
    }

    private static abstract class ProxyTemplate<B, O> extends BeanProxy<B> {

        /**
         * 创建目标对象或临时对象
         */
        @NotNull
        protected abstract O createInstance();

        /**
         * 设置属性
         */
        protected abstract void setProperty(@NotNull O obj, @NotNull String prop, @Nullable Object val);

        /**
         * 获取属性
         */
        @Nullable
        protected abstract Object getProperty(@NotNull O obj, @NotNull String prop);

        /**
         * 转为目标对象
         */
        @Nullable
        protected abstract B toResult(@NotNull O obj);

        @Override
        public BeanInstance<B> newInstance() {
            //noinspection unchecked
            return new BeanInstance<>((B) createInstance(), this);
        }

        @Override
        protected void setProp(@NotNull Object obj, @NotNull String prop, @Nullable Object val) {
            //noinspection unchecked
            setProperty((O) obj, prop, val);
        }

        @Nullable
        @Override
        protected Object getProp(@NotNull Object obj, @NotNull String prop) {
            //noinspection unchecked
            return getProperty((O) obj, prop);
        }

        @Nullable
        @Override
        protected B toRes(@NotNull Object obj) {
            //noinspection unchecked
            return toResult((O) obj);
        }

        protected ProxyTemplate(@NotNull Class<B> clazz) {
            super(clazz);
        }
    }
}
