package cn.cloudself.query.resolver

import cn.cloudself.query.config.QueryProConfig
import cn.cloudself.query.config.SqlAndParams

/**
 * 工具类，目的是根据需要`insert`的对象以及目标数据库表信息生成对应的`SQL`语句(JDBC风格，参数占位符为`?`)
 */
object ToSqlByInsertObjects {
    fun toSql(objs: Collection<Any>, table: String, columnCollection: Collection<QSR.Column>): List<SqlAndParams> {
        val dbType = QueryProConfig.final.dbType()
        val maxParameterSize = QueryProConfig.final.maxParameterSize()
        val isMsSql = DbType.MSSQL == dbType
        val leftQuota = if (isMsSql) '[' else '`'
        val rightQuota = if (isMsSql) ']' else '`'
        val sqlAndParamsList = mutableListOf<SqlAndParams>()
        // 对象中是否存在ID，存在ID的对象和不存在ID的对象，不能放在同一条`INSERT`语句中插入，否则JDBC会返回错误的ID
        val hasId = Array(objs.size) { false }
        // 有序列
        val columns = columnCollection.toList()
        // 所有对象中全为NULL的字段
        val allNullColumnMap = mutableMapOf(*columns.map { it.column to true }.toTypedArray())
        // 按照columns顺序排列的值列表 列表
        val paramsArr = objs.mapIndexed { i, obj ->
            columns.map { col ->
                val value = col.getter(obj)
                if (col.isId && value != null) {
                    hasId[i] = true
                }
                if (value != null) {
                    allNullColumnMap[col.column] = false
                }
                value
            }
        }

        val sqlBuilder = StringBuilder("INSERT INTO ")
        if (!isMsSql) {
            sqlBuilder.append(leftQuota)
        }
        sqlBuilder.append(table)
        if (!isMsSql) {
            sqlBuilder.append(rightQuota)
        }
        sqlBuilder.append(" (")
        var firstAppend = false
        for (col in columns) {
            val column = col.column
            if (allNullColumnMap[column] == true) {
                continue
            }
            if (firstAppend) {
                sqlBuilder.append(", ")
            } else {
                firstAppend = true
            }
            sqlBuilder.append(leftQuota, column, rightQuota)
        }
        sqlBuilder.append(") VALUES ")
        val sqlPrefix = sqlBuilder.toString()

        val maxPageSize = 1000

        var parameterCount = 0
        var curParams = mutableListOf<Any?>()
        var firstRow = true
        var firstRowIndex = 0
        var batchMode = false
        var lastHasId = false
        var fromRowNumber = 1
        for ((i, params) in paramsArr.withIndex()) {
            if (
                (maxParameterSize != null && (parameterCount + params.size + 1) >= maxParameterSize) ||
                (sqlBuilder.length > 1000 * 500) ||
                (i != 0 && i % maxPageSize == 0) ||
                (hasId[i] && !lastHasId && !firstRow) ||
                (lastHasId && !hasId[i])
            ) { // ~0.5M if all ascii char
                batchMode = true
                sqlAndParamsList.add(SqlAndParams("/*BATCH MODE $fromRowNumber to $i*/ $sqlBuilder", curParams.toTypedArray()))
                sqlBuilder.clear()
                sqlBuilder.append(sqlPrefix)
                curParams = mutableListOf()
                firstRow = true
                firstRowIndex = i
                fromRowNumber = i
                parameterCount = 0
            }
            lastHasId = hasId[i]
            sqlBuilder.append(if (firstRow) { firstRow = false; "(" } else ", (")
            var firstCol = true
            for ((j, param) in params.withIndex()) {
                val col = columns[j]
                val useDefault = col.hasDefault == true && param == null
                if (allNullColumnMap[col.column] == true) {
                    continue
                }
                if (!useDefault) {
                    parameterCount++;
                    curParams.add(param)
                }
                sqlBuilder.append(if (firstCol) {
                    firstCol = false;
                    if (useDefault) "DEFAULT" else "?"
                } else
                    if (useDefault) ", DEFAULT" else ", ?"
                )
            }
            sqlBuilder.append(')')
        }

        if (!firstRow) {
            val sql = if (batchMode) "/*BATCH MODE ${firstRowIndex + 1} to ${paramsArr.size}*/ $sqlBuilder" else sqlBuilder.toString()
            val params = curParams.toTypedArray()
            sqlAndParamsList.add(SqlAndParams(sql, params))
        }

        return sqlAndParamsList
    }
}