package cn.cloudself.query.resolver

import cn.cloudself.query.*
import cn.cloudself.query.config.HashMapStore
import cn.cloudself.query.config.Lifecycle
import cn.cloudself.query.config.QueryProConfig
import cn.cloudself.query.config.SqlAndParams
import cn.cloudself.query.psi.Const
import cn.cloudself.query.psi.structure.*
import cn.cloudself.query.util.ClassParser
import cn.cloudself.query.util.LogFactory
import cn.cloudself.query.util.Result

/**
 * `QueryStructure`解析器
 *
 * 该类是`resolver`包最终对外暴露的类
 *
 * 该类的主要作用是为IQueryStructureResolver引入生命周期的支持
 */
object Resolver {
    private val logger = LogFactory.getLog(Resolver::class.java)
    @JvmStatic
    fun create(store: () -> HashMapStore): UseWithStore {
        return UseWithStore(store)
    }
    @JvmStatic
    fun create(clazz: Class<Any>, queryStructure: QueryStructure, payload: QueryPayload): UseWithQueryStructure {
        return UseWithQueryStructure(clazz, queryStructure, payload)
    }

    private fun beforeRunSql(sqlAndParams: SqlAndParams): Result<SqlAndParams, Throwable> {
        var transformedSqlAndParams = sqlAndParams

        val transformers = (QueryProConfig.final.lifecycle() as Lifecycle.Internal).beforeRunSqlTransformers
        for (transformer in transformers) {
            transformedSqlAndParams = transformer.transform(transformedSqlAndParams).getOrElse {
                logger.warn("beforeRunSql钩子阻止了本次操作" as Any, it)
                return Result.err(it)
            }
        }

        return Result.ok(transformedSqlAndParams)
    }

    private fun <R> afterRunSql(result: R): Result<R, Throwable> {
        val transformers = (QueryProConfig.final.lifecycle() as Lifecycle.Internal).afterRunSqlTransformers

        var transformedResult: Any? = result
        for (transformer in transformers) {
            transformedResult = transformer.transform(result).getOrElse {
                logger.warn("afterRunSql钩子阻止了本次操作" as Any, it)
                return Result.err(it)
            }
        }

        @Suppress("UNCHECKED_CAST")
        return Result.ok(transformedResult as R)
    }

    class UseWithStore(private val store: (() -> HashMapStore)) {
        fun <R, EXE: IQueryStructureResolver.PreparedSql<R>> use(resolve: IQueryStructureResolver.() -> EXE): R {
            val config = store().toMap()
            return QueryProConfig.code.use(config) {
                val executor = resolve(QueryProConfig.final.queryStructureResolver())
                executor.init()
                for ((sqlAndParams, run) in executor.lines()) {
                    run(beforeRunSql(sqlAndParams).getOrElse { throw it })
                }
                val result = executor.collect()
                afterRunSql(result).getOrElse { throw it }
            }
        }
    }

    class UseWithQueryStructure(
        private val clazz: Class<Any>,
        private val queryStructure: QueryStructure,
        private val payload: QueryPayload,
    ) {
        fun <R, SQL: IQueryStructureResolver.PreparedSql<R>> use(resolve: IQueryStructureResolver.(QueryStructure) -> SQL): R {
            val config = payload.configs.toMap()
            return QueryProConfig.code.use(config) {
                preRun()
                val transformedQueryStructure = beforeExecLifecycleCallback(queryStructure).getOrElse { throw it }
                val executor = resolve(QueryProConfig.final.queryStructureResolver(), transformedQueryStructure)
                executor.init()
                for ((sqlAndParams, run) in executor.lines()) {
                    run(beforeRunSql(sqlAndParams).getOrElse { throw it })
                }
                var result = executor.collect()
                result = afterRunSql(result).getOrElse { throw it }
                afterExecLifecycleCallback(result).getOrElse { throw it }
            }
        }

        private fun preRun() {
            if (QueryProConfig.final.logicDelete()) {
                val logicDeleteField = QueryProConfig.final.logicDeleteField()
                val hasDeletedField = ClassParser.parse(payload.clazz).columns[logicDeleteField] != null
                if (hasDeletedField) {
                    if (queryStructure.action == QueryStructureAction.DELETE) {
                        val update = Update(data = mutableMapOf(logicDeleteField to true), override = false)
                        queryStructure.action = QueryStructureAction.UPDATE
                        queryStructure.update = update
                    } else {
                        val mainTable = queryStructure.from.alias ?: queryStructure.from.main
                        val hasOrClause = queryStructure.where.find { it.operator == Const.OR } != null
                        val noDeletedWhereClause = WhereClause(Field(mainTable, logicDeleteField), Const.EQUAL, false)
                        if (hasOrClause) {
                            queryStructure.where =
                                    listOf(WhereClause(operator = Const.OPEN_PAR)) +
                                    queryStructure.where +
                                    WhereClause(operator = Const.CLOSE_PAR) +
                                    noDeletedWhereClause
                        } else {
                            queryStructure.where = queryStructure.where + noDeletedWhereClause
                        }
                    }
                }
            }
        }

        private fun beforeExecLifecycleCallback(queryStructure: QueryStructure): Result<QueryStructure, Throwable> {
            val printDebugLog = QueryProConfig.final.printDebugLog()
            val printQueryStructureWholly = printDebugLog && (
                    (queryStructure.insert?.data?.size ?: 0) <= 32 || QueryProConfig.final.printLargeElementWholly()
            )
            if (printDebugLog) {
                if (printQueryStructureWholly) {
                    logger.debug("query structure is transforming. {0}", queryStructure)
                } else {
                    val insert = queryStructure.insert
                    val data = insert?.data
                    val simpleQueryStructure = if (insert != null && data != null) {
                        queryStructure.copy(insert = insert.copy(data = listOf(data.first())))
                    } else {
                        queryStructure
                    }
                    logger.debug(
                        "query structure is transforming. but the insert elements is so large, only first element is print: {0}",
                        simpleQueryStructure
                    )
                }
            }

            var transformedQueryStructure = queryStructure

            val transformers = (QueryProConfig.final.lifecycle() as Lifecycle.Internal).beforeExecTransformers
            for (transformer in transformers) {
                transformedQueryStructure = transformer.transform(clazz, transformedQueryStructure, payload).getOrElse {
                    logger.warn("beforeExec钩子阻止了本次操作" as Any, it)
                    return Result.err(it)
                }
            }

            if (printDebugLog) {
                if (printQueryStructureWholly) {
                    logger.debug("query structure transformed. {0}", transformedQueryStructure)
                } else {
                    val insert = transformedQueryStructure.insert
                    val data = insert?.data
                    val simpleQueryStructure = if (insert != null && data != null) {
                        transformedQueryStructure.copy(insert = insert.copy(data = listOf(data.first())))
                    } else {
                        transformedQueryStructure
                    }
                    logger.debug(
                        "query structure transformed. but the insert elements is so large, only first element is print: {0}",
                        simpleQueryStructure
                    )
                }
            }

            return Result.ok(transformedQueryStructure)
        }

        private fun <R> afterExecLifecycleCallback(result: R): Result<R, Throwable> {
            @Suppress("KotlinConstantConditions") var transformedResult: Any = result ?: return Result.ok(result)

            val transformers = (QueryProConfig.final.lifecycle() as Lifecycle.Internal).afterExecTransformers
            for (transformer in transformers) {
                transformedResult = transformer.transform(clazz, transformedResult, queryStructure, payload).getOrElse {
                    logger.warn("afterExec钩子阻止了本次操作" as Any, it)
                    return Result.err(it)
                }
            }

            @Suppress("UNCHECKED_CAST")
            return Result.ok(transformedResult as R)
        }
    }
}