package cn.cloudself.query.resolver

import cn.cloudself.query.psi.structure.QueryPayload
import cn.cloudself.query.psi.structure.QueryStructure
import cn.cloudself.query.psi.structure.QueryStructureAction
import cn.cloudself.query.config.SqlAndParams
import org.intellij.lang.annotations.Language

/**
 * 主要用于解析对象并执行或直接执行`SQL`
 *
 * 包含
 * 1. 解析并执行`QueryStructure`接口，
 * 2. 直接执行单条`SQL`接口
 * 3. 执行多条`SQL`查询语句接口
 * 4. 动态插入接口（根据数据库结构以及提供的`JavaBean`生成`INSERT`语句并执行）
 */
interface IQueryStructureResolver {
    class PreparedSql<T> private constructor(
        private val doInit: () -> Any?,
        private val sqlAndParamsList: List<SqlAndParams>,
        private val run: (SqlAndParams, Any?, Int) -> Unit,
        @Suppress("UNCHECKED_CAST") private var doCollect: (data: Any?) -> T = { it as T },
    ) {
        private class Ref<R>(var value: R)
        companion object {
            @Suppress("UNCHECKED_CAST")
            fun <R> oneLine(sql: String, params: Array<out Any?>, run: (SqlAndParams) -> R): PreparedSql<R> {
                return PreparedSql({Ref<R?>(null)}, listOf(SqlAndParams(sql, params)), { tsp, ref, _ -> (ref as Ref<R>).value = run(tsp)}, { (it as Ref<R>).value })
            }
            fun <D, R> multiLine(init: () -> D, sqlAndParams: List<SqlAndParams>, run: (SqlAndParams, D, Int) -> Unit): PreparedSql<R> {
                @Suppress("UNCHECKED_CAST")
                return PreparedSql(init, sqlAndParams, run as (SqlAndParams, Any?, Int) -> Unit)
            }
            fun <D, R> multiLine(init: () -> D, sqlAndParams: List<SqlAndParams>, run: (SqlAndParams, D, Int) -> Unit, collect: (D) -> R): PreparedSql<R> {
                @Suppress("UNCHECKED_CAST")
                return PreparedSql(init, sqlAndParams, run as (SqlAndParams, Any?, Int) -> Unit, collect as (Any?) -> R)
            }
        }

        private var data: Any? = null
        fun init() {
            data = doInit()
        }
        fun lines(): Iterable<Pair<SqlAndParams, (SqlAndParams) -> Unit>> {
            return sqlAndParamsList.mapIndexed { index, sqlAndParams ->
                sqlAndParams to { sap: SqlAndParams -> run(sap, data, index) }
            }
        }
        fun collect(): T {
            return doCollect(data)
        }
        fun map(transformer: (T) -> T): PreparedSql<T> {
            val oldDoCollect = doCollect
            doCollect = {
                transformer(oldDoCollect(it))
            }
            return this
        }
    }

    /**
     * 将QueryStructure解析至SQL并执行
     *
     * @param queryStructure [QueryStructure]
     * @param clazz QSR模版支持`JavaBean`, `Map`, 以及`Long`, `String`, `Boolean`等基本类型。另外，默认使用的JdbcQSR(为QSR的子类)还额外支持`IResultSetWalker`
     * @return clazz为基本类型时，可能返回List<T?> 当动作为insert时，返回值为List<ID>
     */
    fun <T: Any> resolve(queryStructure: QueryStructure, payload: QueryPayload, clazz: Class<T>): PreparedSql<List<T>>

    /**
     * 动态插入
     * 通过 objs 配合数据库表名，动态生成sql语句，并执行
     *
     * @param objs 需要插入的Map类型的集合
     * @param table 数据库表名
     * @param idColumnClazz 如需返回ID，指定ID的类型
     */
    fun <ID: Any, V: Any?, M: Map<String, V>> insert(objs: Collection<M>, table: String, idColumnClazz: Class<ID>?): PreparedSql<List<ID>>

    /**
     * 执行一个SQL查询
     *
     * @param sql 单条sql语句 e.g. SELECT * FROM user WHERE user.id = ?
     * @param params 参数数组 e.g. [1]
     * @param clazz `JavaBean`, `Map`, and basic type e.g. `Long`, `String`, `Boolean` etc. for select by default; [Int] for update
     * @param action [QueryStructureAction]
     * @return clazz为基本类型时，可能返回List<T?>
     */
    fun <T: Any> resolve(@Language("SQL") sql: String, params: Array<out Any?>, clazz: Class<T>, action: QueryStructureAction): PreparedSql<List<T>>

    /**
     * 使用多条语句和参数执行更新，创建，删除等非select语句
     *
     * @param sqlArr 多条或单条SQL语句
     * @param paramsArr sqlArr 的长度为1时，params的长度任意。代表同一语句包含多参数
     *                  sqlArr 的长度不为1时，params的长度必须和 sqlArr的长度相等。
     * @param clazz 指定返回结果的类型，支持 List<Integer> Integer[] Int Boolean(不准确)
     */
    fun <T: Any> exec(sqlArr: Array<out String>, paramsArr: Array<out Array<out Any?>>, clazz: Class<T>): PreparedSql<T>
}
