package cn.cloudself.query.config

import cn.cloudself.query.exception.IllegalCall
import cn.cloudself.query.exception.IllegalImplements
import cn.cloudself.query.exception.UnSupportException
import cn.cloudself.query.resolver.DbType
import cn.cloudself.query.resolver.IQueryStructureResolver
import cn.cloudself.query.util.LogLevel
import org.springframework.web.context.request.RequestAttributes
import org.springframework.web.context.request.RequestContextHolder
import java.sql.Connection
import java.sql.PreparedStatement
import java.sql.ResultSet
import java.sql.SQLException
import java.util.*
import javax.sql.DataSource
import kotlin.reflect.KClass

fun interface ResultSetGetter<T> {
    @Throws(SQLException::class)
    fun get(rs: ResultSet, i: Int): T?
}

fun interface SqlParamSetter<T> {
    @Throws(SQLException::class)
    fun set(ps: PreparedStatement, i: Int, obj: T)
}

fun interface ResultSetParserEx {
    @Throws(SQLException::class)
    fun parse(rs: ResultSet, clazz: Class<*>, i: Int): Optional<Any>
}

data class DbColumnInfo(
    val type: String,
    val label: String,
)

/**
 * 通用的配置选项(读取接口)
 */
interface IQueryProConfigDb {
    fun dbType(): DbType?
    /** 读取Connection配置（如有），与dataSource二选一即可 */
    fun maxParameterSize(): Int?
    fun connection(): Connection?
    fun dataSource(): DataSource?
    fun beautifySql(): Boolean?
    fun printLog(): Boolean?
    fun printLogLevel(): LogLevel?
    fun printLargeElementWholly(): Boolean?
    fun printCallByInfo(): Boolean?
    fun printCallByInfoLevel(): LogLevel?
    fun printResult(): Boolean?
    fun printResultLevel(): LogLevel?
    fun dryRun(): Boolean?
    fun logicDelete(): Boolean?
    fun logicDeleteField(): String?
    fun logicDeleteTrue(): Any?
    fun logicDeleteFalse(): Any?
}

/**
 * 通用的配置选项(写入接口)
 */
interface IQueryProConfigDbWriteable<T: IQueryProConfigDbWriteable<T>> {
    fun dbType(dbType: DbType): T
    fun maxParameterSize(size: Int): T
    fun connection(connection: Connection): T
    fun dataSource(dataSource: DataSource): T
    fun beautifySql(enable: Boolean): T
    fun printLog(enable: Boolean): T
    fun printLog(enable: Boolean, level: LogLevel): T
    fun printLargeElementWholly(print: Boolean): T
    fun printCallByInfo(enable: Boolean): T
    fun printCallByInfo(enable: Boolean, level: LogLevel): T
    fun printResult(enable: Boolean): T
    fun printResult(enable: Boolean, level: LogLevel): T

    /**
     * 占位，未充分测试，不推荐使用
     */
    fun dryRun(enable: Boolean): T
    fun logicDelete(enable: Boolean): T
    fun logicDelete(enable: Boolean, field: String, trueValue: Any?, falseValue: Any?): T
}

/**
 * 通用的`DB`实现(写入部分)(构造函数的参数中需传入一个`Store`)
 */
abstract class DefaultQueryProConfigDbWriteable<T: IQueryProConfigDbWriteable<T>>(
    private val store: Store
): IQueryProConfigDbWriteable<T> {
    protected abstract fun that(): T
    /* 更改数据库类型，默认为mysql */
    override fun dbType(dbType: DbType) = that().also { store.set("dbType", dbType) }
    override fun maxParameterSize(size: Int) = that().also { store.set("maxParameterSize", size) }
    override fun connection(connection: Connection) = that().also { store.set("connection", connection) }
    override fun dataSource(dataSource: DataSource) = that().also { store.set("dataSource", dataSource) }
    override fun beautifySql(enable: Boolean) = that().also { store.set("beautifySql", enable) }
    override fun printLog(enable: Boolean) = that().also { store.set("printLog", enable); store.set("printLogLevel", LogLevel.INFO) }
    override fun printLog(enable: Boolean, level: LogLevel) = that().also { store.set("printLog", enable); store.set("printLogLevel", level) }
    override fun printLargeElementWholly(print: Boolean) = that().also { store.set("printLargeElementWholly", print) }
    override fun printCallByInfo(enable: Boolean) = that().also { store.set("printCallByInfo", enable) }
    override fun printCallByInfo(enable: Boolean, level: LogLevel) = printCallByInfo(enable).also { store.set("printCallByInfoLevel", level) }
    override fun printResult(enable: Boolean) = that().also { store.set("printResult", enable) }
    override fun printResult(enable: Boolean, level: LogLevel) = printResult(enable).also { store.set("printResultLevel", level) }
    override fun dryRun(enable: Boolean) = that().also { if (enable) throw UnSupportException("占位，未充分测试，不推荐使用"); store.set("dryRun", enable) }
    override fun logicDelete(enable: Boolean) = that().also { store.set("logicDelete", enable) }
    override fun logicDelete(enable: Boolean, field: String, trueValue: Any?, falseValue: Any?) = logicDelete(enable).also {
        store.set("logicDeleteField", field)
        store.set("logicDeleteTrue", trueValue)
        store.set("logicDeleteFalse", falseValue)
    }
}

interface Store {
    fun get(key: String): Any?
    fun set(key: String, value: Any?)
}

/**
 * 通用的`DB`实现(构造函数的参数中需传入一个`Store`)
 */
open class QueryProConfigDb(private val store: Store): IQueryProConfigDb, DefaultQueryProConfigDbWriteable<QueryProConfigDb>(store) {
    override fun that() = this

    override fun dbType()                 = store.get("dbType") as DbType?
    override fun maxParameterSize()       = store.get("maxParameterSize") as Int?
    /**
     * 配置connection，与dataSource二选一即可。
     *
     * 相比起[connection]更推荐使用[dataSource], 因为[dataSource]会自动管理链接的开启与关闭。
     * 而[connection]需要手动管理链接的开启与关闭(这在存在事务的时候需要额外注意)
     */
    override fun connection()             = store.get("connection") as Connection?
    override fun dataSource()             = store.get("dataSource") as DataSource?
    override fun beautifySql()            = store.get("beautifySql") as Boolean?
    override fun printLog()               = store.get("printLog") as Boolean?
    override fun printLogLevel()          = store.get("printLogLevel") as LogLevel?
    override fun printLargeElementWholly()    = store.get("printLargeElementWholly") as Boolean?
    override fun printCallByInfo()        = store.get("printCallByInfo") as Boolean?
    override fun printCallByInfoLevel()   = store.get("printCallByInfoLevel") as LogLevel?
    override fun printResult()            = store.get("printResult") as Boolean?
    override fun printResultLevel()       = store.get("printResultLevel") as LogLevel?
    override fun dryRun()                 = store.get("dryRun") as Boolean?
    override fun logicDelete()            = store.get("logicDelete") as Boolean?
    override fun logicDeleteField()       = store.get("logicDeleteField") as String?
    override fun logicDeleteTrue()        = store.get("logicDeleteTrue")
    override fun logicDeleteFalse()       = store.get("logicDeleteFalse")
}

/**
 * `hash map` 实现的 `Store` 用于存储全局`global`的配置。
 */
open class HashMapStore: Store {
    private val store = mutableMapOf<String, Any?>()
    override fun get(key: String): Any? = store[key]
    override fun set(key: String, value: Any?) { store[key] = value }
    fun copyFrom(hashMapStore: HashMapStore) = this.also {
        store.clear()
        store.putAll(hashMapStore.store);
    }
    fun toMap(): Map<String, Any?> {
        val map = mutableMapOf<String, Any?>()
        for (key in store.keys) {
            map[key] = this.get(key)
        }
        return map
    }
}

private const val KEY_PREFIX = "QUERY_PRO_CONFIG:REQUEST_CONTEXT:"

/**
 * `spring` `RequestContextHolder` 实现的 `Store`, 用于存储`request`级的配置。
 */
class RequestContextStore: Store {
    private val isRequestContextHolderPresent = try {
        Class.forName("org.springframework.web.context.request.RequestContextHolder")
        true
    } catch (e: Throwable) {
        false
    }

    override fun get(key: String): Any? {
        if (!isRequestContextHolderPresent) {
            return null
        }

        return try {
            RequestContextHolder.currentRequestAttributes().getAttribute("$KEY_PREFIX$key", RequestAttributes.SCOPE_REQUEST)
        } catch (e: Exception) {
            null
        }
    }

    override fun set(key: String, value: Any?) {
        @Suppress("NULLABILITY_MISMATCH_BASED_ON_JAVA_ANNOTATIONS")
        RequestContextHolder.currentRequestAttributes().setAttribute("$KEY_PREFIX$key", value, RequestAttributes.SCOPE_REQUEST)
    }
}

/**
 * `ThreadLocal` 实现的 `Store`, 用于存储 `thread`, `context`, `code`级的配置。
 */
class ThreadContextStore: Store {
    private val store: ThreadLocal<MutableMap<String, Any?>?> = ThreadLocal()
    private var initCalled = false

    fun init() {
        store.remove()
        initCalled = true
    }

    fun clean() {
        init()
    }

    override fun get(key: String): Any? {
        return store.get()?.get(key)
    }

    override fun set(key: String, value: Any?) {
        if (!initCalled) {
            throw IllegalCall("使用thread进行配置时，因为线程池的问题，" +
                    "必须在线程初始化后调用initThreadContextStore，线程结束时调用cleanThreadContextStore，" +
                    "所以spring环境更推荐使用request进行配置。")
        }
        var map = store.get()
        if (map == null) {
            map = mutableMapOf()
            store.set(map)
        }
        map[key] = value
    }

    fun getStore(): Map<String, Any?>? {
        return store.get()
    }

    fun setStore(map: MutableMap<String, Any?>) {
        store.set(map)
    }
}

fun interface DataSourceGetter {
    fun get(clazz: Class<*>?): DataSource?
}

/**
 * 仅用于全局的配置选项
 */
interface IOnlyGlobalConfig {
    fun lifecycle(): Lifecycle
    fun defaultDataSource(): DataSourceGetter
    fun shouldIgnoreFields(): Set<String>
    fun supportedColumnType(): Set<Class<*>>
    fun dbColumnInfoToJavaType(): Map<(column: DbColumnInfo) -> Boolean, Class<*>>
    fun <T> sqlParamSetter(clazz: Class<T>): SqlParamSetter<T>?
    fun <T> resultSetParser(clazz: Class<T>): ResultSetGetter<T>?
    fun resultSetParserEx(): List<ResultSetParserEx>
    fun queryStructureResolver(): IQueryStructureResolver
}

/**
 * 继承自`QueryProConfigDb`, 额外引入了一些仅在全局作用域下支持的配置。
 */
class GlobalQueryProConfigDb: QueryProConfigDb(HashMapStore()), IOnlyGlobalConfig {
    private val lifecycle: Lifecycle = Lifecycle.Internal()
    private val supportedColumnType = mutableSetOf<Class<*>>()
    private val sqlParamSetter = mutableMapOf<Class<*>, SqlParamSetter<*>>()
    private val resultSetParser = mutableMapOf<Class<*>, ResultSetGetter<*>>()
    private var defaultDataSource = DataSourceGetter { null }
    private var queryStructureResolver: IQueryStructureResolver? = null
    internal val resultSetParserEx = mutableListOf<ResultSetParserEx>()
    internal val dbColumnInfoToJavaType = mutableMapOf<(column: DbColumnInfo) -> Boolean, Class<*>>()
    internal val shouldIgnoreFields = mutableSetOf<String>()

    override fun lifecycle() = lifecycle
    override fun shouldIgnoreFields(): Set<String> = shouldIgnoreFields
    override fun supportedColumnType(): Set<Class<*>> = supportedColumnType
    override fun resultSetParserEx(): List<ResultSetParserEx> = resultSetParserEx
    override fun dbColumnInfoToJavaType(): Map<(column: DbColumnInfo) -> Boolean, Class<*>> = dbColumnInfoToJavaType
    override fun defaultDataSource() = defaultDataSource
    override fun queryStructureResolver() = this.queryStructureResolver ?: throw IllegalCall("unknown query structure resolver.")

    /**
     * 读取用户自定义的SqlParamSetter解析器(内置的支持无法读取（出于性能优化目的）)
     */
    override fun <T> sqlParamSetter(clazz: Class<T>): SqlParamSetter<T>? {
        @Suppress("UNCHECKED_CAST")
        return this.sqlParamSetter[clazz] as SqlParamSetter<T>?
    }

    /**
     * ResultSet解析器
     * 内置支持的类型有:
     * BigDecimal, Byte, ByteArray, Date, LocalDate, LocalTime, LocalDateTime, java.sql.Date, Double, Float, Int,
     * Long, Time, Timestamp, Short, String,
     */
    @Suppress("UNCHECKED_CAST")
    override fun <T> resultSetParser(clazz: Class<T>): ResultSetGetter<T>? {
        var res = this.resultSetParser[clazz] as ResultSetGetter<T>?
        if (res == null) {
            for ((key, value) in this.resultSetParser) {
                if (key.isAssignableFrom(clazz)) {
                    res = value as ResultSetGetter<T>
                    break
                }
            }
        }
        return res
    }

    /**
     * 添加一个ResultSet解析器(字段解析器)
     * @param clazz 需要解析至的class, 例如: LocalDate.class
     * @param value 例子 rs -> i -> rs.getDate(i).toLocalDate()
     *
     * @see [resultSetParser]
     * @see [resultSetParserEx]
     */
    fun <T> addResultSetParser(clazz: Class<T>, value: ResultSetGetter<T>) = this.also {
        resultSetParser[clazz] = value
        supportedColumnType.add(clazz)
    }

    /**
     * 添加一个SqlParamSetter
     */
    fun <T> addSqlParamSetter(clazz: Class<T>, value: SqlParamSetter<T>) = this.also {
        sqlParamSetter[clazz] = value
    }

    /**
     * 添加一个ResultSet解析器，与addResultSetParser功能相似，但更推荐使用addResultSetParser，因为性能略好
     */
    fun addResultSetParserEx(parser: ResultSetParserEx) = this.also {
        resultSetParserEx.add(parser)
    }

    fun setQueryStructureResolver(qsr: IQueryStructureResolver) = this.also {
        queryStructureResolver = qsr
    }

    fun setDefaultDataSource(getter: DataSourceGetter) = this.also { defaultDataSource = getter }

    internal fun <T: Any> putToResultSetParser(clazz: KClass<T>, value: ResultSetGetter<T>) {
        val primitiveType = (clazz).javaPrimitiveType
        if (primitiveType != null && primitiveType != clazz.java) {
            supportedColumnType.add(primitiveType)
            resultSetParser[primitiveType] = value
        }
        val objectType = clazz.javaObjectType
        if (objectType != clazz.java) {
            supportedColumnType.add(objectType)
            resultSetParser[objectType] = value
        }

        supportedColumnType.add(clazz.java)
        resultSetParser[clazz.java] = value
    }
}

/**
 * 继承自`QueryProConfigDb`, 添加了一些方法，需手动管理`ThreadLocal`中数据的初始化和销毁，也可以用`use`方法管理。
 */
class FinalQueryProConfigDb(private val configs: Array<IQueryProConfigDb>): IQueryProConfigDb, IOnlyGlobalConfig {
    private fun <R> getBy(getter: IQueryProConfigDb.() -> R?): R {
        for (config in configs) {
            val res = getter(config)
            if (res != null) {
                return res
            }
        }
        throw IllegalImplements("遍历了所有配置但仍然无法找到有效配置")
    }

    private fun <R> getByNullable(getter: IQueryProConfigDb.() -> R?): R? {
        for (config in configs) {
            val res = getter(config)
            if (res != null) {
                return res
            }
        }
        return null
    }

    fun dataSource(clazz: Class<*>?): DataSource? {
        val defaultDataSource = DefaultDataSourceWrapper.getDefault(clazz)
        if (defaultDataSource != null) {
            return defaultDataSource
        }

        val size = configs.size
        val last = configs[size - 1]
        if (last !is GlobalQueryProConfigDb) {
            throw IllegalImplements("最后一项配置必须为GlobalQueryProConfigDb")
        }
        for (i in 0 until size - 1) {
            val res = configs[i].dataSource()
            if (res != null) {
                return res
            }
        }
        return last.dataSource()
    }

    override fun dbType() = getByNullable { dbType() }
    override fun maxParameterSize() = getByNullable { maxParameterSize() }
    override fun connection() = getByNullable { connection() }
    override fun dataSource() = throw IllegalCall("不可直接调用dataSource()，使用dataSource(Class<?>)替代")
    override fun beautifySql() = getBy { beautifySql() }
    override fun printLog() = getBy { printLog() }
    override fun printLogLevel() = getBy { printLogLevel() }
    override fun printLargeElementWholly() = getBy { printLargeElementWholly() }
    fun printDebugLog() = printLog() || printLogLevel() != LogLevel.DEBUG
    override fun printCallByInfo() = getBy { printCallByInfo() }
    override fun printCallByInfoLevel() = getBy { printCallByInfoLevel() }
    override fun printResult() = getBy { printResult() }
    override fun printResultLevel() = getBy { printResultLevel() }
    override fun dryRun() = getBy { dryRun() }
    override fun logicDelete() = getBy { logicDelete() }
    override fun logicDeleteField() = getBy { logicDeleteField() }
    override fun logicDeleteTrue(): Any? {
        for (config in configs) {
            if (config.logicDeleteField() != null) {
                return config.logicDeleteTrue()
            }
        }
        throw IllegalImplements("遍历了所有配置但仍然无法找到有效配置")
    }
    override fun logicDeleteFalse(): Any? {
        for (config in configs) {
            if (config.logicDeleteField() != null) {
                return config.logicDeleteFalse()
            }
        }
        throw IllegalImplements("遍历了所有配置但仍然无法找到有效配置")
    }

    override fun lifecycle() = QueryProConfig.global.lifecycle()
    override fun defaultDataSource() = QueryProConfig.global.defaultDataSource()
    override fun shouldIgnoreFields(): Set<String> = QueryProConfig.global.shouldIgnoreFields()
    override fun supportedColumnType() = QueryProConfig.global.supportedColumnType()
    override fun dbColumnInfoToJavaType() = QueryProConfig.global.dbColumnInfoToJavaType()
    override fun <T> sqlParamSetter(clazz: Class<T>) = QueryProConfig.global.sqlParamSetter(clazz)
    override fun <T> resultSetParser(clazz: Class<T>) = QueryProConfig.global.resultSetParser(clazz)
    override fun resultSetParserEx() = QueryProConfig.global.resultSetParserEx()
    override fun queryStructureResolver() = QueryProConfig.global.queryStructureResolver()
}

/**
 * 继承自`QueryProConfigDb`, 添加了一些方法，需手动管理`ThreadLocal`中数据的初始化和销毁，也可以用`use`方法管理。
 */
open class ThreadQueryProConfigDb(
    internal val store: ThreadContextStore = ThreadContextStore(),
    private val configDb: QueryProConfigDb = QueryProConfigDb(store),
): IQueryProConfigDb by configDb, IQueryProConfigDbWriteable<QueryProConfigDb> by configDb {
    fun init() {
        store.init()
    }

    fun clean() {
        store.clean()
    }

    /**
     * 在回调函数中，维持一个query pro配置的上下文
     * 注意该配置对函数中新开的线程无效
     *
     * context不能嵌套
     *
     * QueryProConfig.context.use(context -> {
     *   context.beautifySql();
     *   UserQueryPro.selectBy().id().equalTo(1);
     * });
     */
    fun use(func: Use) {
        use {
            func.call(it)
        }
    }

    /**
     * 在回调函数中，维持一个query pro配置的上下文
     * 注意该配置对函数中新开的线程无效
     *
     * context不能嵌套
     *
     * QueryProConfig.context.use(context -> {
     *   context.beautifySql();
     *   UserQueryPro.selectBy().id().equalTo(1);
     * });
     */
    fun <T> use(func: UseResult<T>): T {
        return use {
            func.call(it)
        }
    }

    /**
     * 在回调函数中，维持一个query pro配置的上下文
     * 注意该配置对函数中新开的线程无效
     *
     * context不能嵌套
     *
     * QueryProConfig.context.use(context -> {
     *   context.beautifySql();
     *   UserQueryPro.selectBy().id().equalTo(1);
     * });
     */
    @JvmName("_useKt")
    fun <T> use(func: (context: QueryProConfigDb) -> T): T {
        store.init()
        val result = try {
            func(configDb)
        } finally {
            store.clean()
        }
        return result!!
    }

    interface Use {
        @Throws(Exception::class)
        fun call(context: QueryProConfigDb)
    }

    interface UseResult<T> {
        @Throws(Exception::class)
        fun call(context: QueryProConfigDb): T
    }
}

/**
 * 继承自`QueryProConfigDb`, 添加了一些方法，需手动管理`ThreadLocal`中数据的初始化和销毁，也可以用`use`方法管理。
 */
class CodeQueryProConfigDb: ThreadQueryProConfigDb() {
    fun <T> use(store: Map<String, Any?>, func: (context: QueryProConfigDb) -> T): T {
        return use {
            this.store.setStore(store.toMutableMap())
            func(it)
        }
    }
}
