package cn.cloudself.query.config

import cn.cloudself.query.config.Lifecycle.*
import cn.cloudself.query.exception.ConfigException
import cn.cloudself.query.exception.IllegalImplements
import cn.cloudself.query.psi.Const
import cn.cloudself.query.psi.structure.QueryPayload
import cn.cloudself.query.psi.structure.QueryStructure
import cn.cloudself.query.psi.structure.QueryStructureAction
import cn.cloudself.query.util.BeanProxy
import cn.cloudself.query.util.Result

open class Lifecycle {
    class Internal: Lifecycle() {
        // query pro
        public override val beforeExecTransformers = mutableListOf<QueryStructureTransformer>()
        public override val afterExecTransformers = mutableListOf<ResultWithQueryStructureTransformer>()
        // query pro sql & query pro
        public override val beforeRunSqlTransformers = mutableListOf<SqlAndParamsTransformer>()
        public override val afterRunSqlTransformers = mutableListOf<ResultTransformer>()
    }
    // query pro
    protected open val beforeExecTransformers = mutableListOf<QueryStructureTransformer>()
    protected open val afterExecTransformers = mutableListOf<ResultWithQueryStructureTransformer>()
    // query pro sql & query pro
    protected open val beforeRunSqlTransformers = mutableListOf<SqlAndParamsTransformer>()
    protected open val afterRunSqlTransformers = mutableListOf<ResultTransformer>()

    fun interface SqlAndParamsTransformer {
        /**
         */
        fun transform(sqlAndParams: SqlAndParams): Result<SqlAndParams, Throwable>
    }

    fun interface ResultTransformer {
        fun transform(result: Any?): Result<Any?, Throwable>
    }

    fun interface QueryStructureTransformer {
        fun transform(clazz: Class<Any>, qs: QueryStructure, payload: QueryPayload): Result<QueryStructure, Throwable>
    }

    fun interface ResultWithQueryStructureTransformer {
        fun transform(clazz: Class<Any>?, result: Any, qs: QueryStructure, payload: QueryPayload): Result<Any, Throwable>
    }

    abstract class BaseQsTransformersBuilder(private val action: QueryStructureAction) {
        private val transformers = mutableListOf<QueryStructureTransformer>()
        protected fun addTransformerWhenActionMatched(transformer: QueryStructureTransformer) = this.also {
            transformers.add { clazz, queryStructure, payload ->
                if (queryStructure.action == action) {
                    transformer.transform(clazz, queryStructure, payload)
                } else {
                    Result.ok(queryStructure)
                }
            }
        }
        abstract fun addTransformer(transformer: QueryStructureTransformer): BaseQsTransformersBuilder
        fun build() = transformers
    }

    /**
     * 添加，修改，删除返回结果时的过滤器
     * 一般作为可选参数出现
     * 当返回结果为false时代表不执行相应的操作
     */
    fun interface BeforeInsertPredicate {
        fun predicate(bean: BeanProxy.BeanInstance<Any>, qs: QueryStructure, payload: QueryPayload): Boolean
    }

    /**
     * 插入前，对QueryStructure进行转换，
     * 例如，添加创建时间，创建人等
     *
     * @see [addField]
     * @see [overrideField]
     */
    class BeforeInsertTransformersBuilder internal constructor(): BaseQsTransformersBuilder(QueryStructureAction.INSERT) {
        /**
         * 添加一个StructureTransformer转换器
         *
         * 根据提供的 Class<Any>, QueryStructure, QueryPayload 信息，对QueryStructure进行转换并返回
         *
         * @param transformer [QueryStructureTransformer.transform]
         * + _kotlin_:
         * ```kotlin
         * fun transform(clazz: Class<Any>, qs: QueryStructure, payload: QueryPayload): Result<QueryStructure, Throwable>
         * ```
         * + _java_:
         * ```java
         * Result<QueryStructure, Throwable> transform(Class<?> clazz, QueryStructure qs, QueryPayload payload)
         * ```
         */
        override fun addTransformer(transformer: QueryStructureTransformer) =
            this.also { addTransformerWhenActionMatched(transformer) }

        /**
         * 插入前添加某字段，例如create_time等
         * 不会处理QueryProSql执行的插入操作
         * @param fieldClass 有时候，相同的 field 会有多种类型的取值，使用该字段可以区分这种情况
         * @param predicate 可选的过滤器，返回结果为false时，该字段不会被添加
         */
        @JvmOverloads
        fun <T> addField(
            field: String,
            fieldClass: Class<T>,
            value: () -> T,
            predicate: BeforeInsertPredicate = BeforeInsertPredicate { _, _, _ -> true }
        ) = this.also {
            overrideField(field, fieldClass, value, false, predicate)
        }

        /**
         * 插入前添加某字段，例如create_time等
         * 不会处理QueryProSql执行的插入操作
         * @param fieldClass 有时候，相同的 field 会有多种类型的取值，使用该字段可以区分这种情况
         * @param predicate 可选的过滤器，返回结果为false时，该字段不会被添加或覆盖
         */
        @JvmOverloads
        fun <T> overrideField(
            field: String,
            fieldClass: Class<*>,
            value: () -> T,
            predicate: BeforeInsertPredicate = BeforeInsertPredicate { _, _, _ -> true }
        ) = this.also {
            overrideField(field, fieldClass, value, true, predicate)
        }

        private fun <T> overrideField(
            field: String,
            fieldClass: Class<*>,
            getValue: () -> T,
            override: Boolean,
            predicate: BeforeInsertPredicate
        ) {
            addTransformerWhenActionMatched { clazz, queryStructure, payload ->
                val objs = queryStructure.insert?.data ?: throw IllegalImplements("insert 传入了空值。")
                for (obj in objs) {
                    val beanInstance = BeanProxy.fromBean(obj, clazz)
                    if (!predicate.predicate(beanInstance, queryStructure, payload)) {
                        return@addTransformerWhenActionMatched Result.ok(queryStructure)
                    }

                    val fieldType = beanInstance.getPropertyType(field) ?: continue
                    if (fieldType != fieldClass) {
                        continue
                    }

                    if (beanInstance.getProperty(field) == null || override) {
                        val value = getValue()
                        if (value != Const.SKIP) {
                            if (value == Const.NULL) {
                                beanInstance.setProperty(field, null)
                            } else {
                                beanInstance.setProperty(field, value)
                            }
                        }
                    }
                }
                Result.ok(queryStructure)
            }
        }
    }

    /**
     * 添加，修改，删除返回结果时的过滤器
     * 一般作为可选参数出现
     * 当返回结果为false时代表不执行相应的操作
     */
    fun interface BeforeUpdatePredicate {
        fun predicate(bean: BeanProxy.BeanInstance<Any>, qs: QueryStructure, payload: QueryPayload): Boolean
    }

    /**
     * 更新前，对QueryStructure进行转换，
     * 例如，添加更新时间，更新人等
     *
     * @see [addField]
     * @see [overrideField]
     */
    class BeforeUpdateTransformersBuilder internal constructor(): BaseQsTransformersBuilder(QueryStructureAction.UPDATE) {
        /**
         * 添加一个StructureTransformer转换器
         *
         * 根据提供的 Class<Any>, QueryStructure, QueryPayload 信息，对QueryStructure进行转换并返回
         *
         * @param transformer [QueryStructureTransformer.transform]
         * + _kotlin_:
         * ```kotlin
         * fun transform(clazz: Class<Any>, qs: QueryStructure, payload: QueryPayload): Result<QueryStructure, Throwable>
         * ```
         * + _java_:
         * ```java
         * Result<QueryStructure, Throwable> transform(Class<?> clazz, QueryStructure qs, QueryPayload payload)
         * ```
         */
        override fun addTransformer(transformer: QueryStructureTransformer) =
            this.also { addTransformerWhenActionMatched(transformer) }

        /**
         * 更新前添加字段，当未指定某字段的值时，为该字段添加默认值，如update_by等
         * 不会处理QueryProSql执行的插入操作
         * @param fieldClass 有时候，相同的 field 会有多种类型的取值，使用该字段可以区分这种情况
         * @param predicate 可选的过滤器，返回结果为false时，该字段不会被添加
         */
        @JvmOverloads
        fun <T> addField(
            field: String,
            fieldClass: Class<T>,
            value: () -> T,
            predicate: BeforeUpdatePredicate = BeforeUpdatePredicate { _, _, _ -> true }
        ) = this.also {
            overrideField(field, fieldClass, value, false, predicate)
        }

        /**
         * 更新前添加字段，覆盖字段，无论是否指定此字段，始终覆盖，例如：更新时间
         * 不会处理QueryProSql执行的插入操作
         * 可使用 QueryProConst(Kt).NULL 实现deleteField
         * @param fieldClass 有时候，相同的 field 会有多种类型的取值，使用该字段可以区分这种情况
         * @param predicate 可选的过滤器，返回结果为false时，该字段不会被添加或覆盖
         */
        @JvmOverloads
        fun <T> overrideField(
            field: String,
            fieldClass: Class<T>,
            value: () -> T,
            predicate: BeforeUpdatePredicate = BeforeUpdatePredicate { _, _, _ -> true }
        ) = this.also {
            overrideField(field, fieldClass, value, true, predicate)
        }

        /**
         * 更新时忽略某字段
         * 不会处理QueryProSql执行的插入操作
         * @param fieldClass 有时候，相同的 field 会有多种类型的取值，使用该字段可以区分这种情况，可为null，为null时代表不区分类型
         * @param predicate 可选的过滤器，返回结果为false时，该字段不会被移除
         */
        @JvmOverloads
        fun deleteField(
            field: String,
            fieldClass: Class<Any>?,
            predicate: BeforeUpdatePredicate = BeforeUpdatePredicate { _, _, _ -> true }
        ) = this.also {
            overrideField(field, fieldClass, { Const.NULL }, true, predicate)
        }

        private fun <T> overrideField(
            field: String,
            fieldClass: Class<*>?,
            getValue: () -> T,
            override: Boolean,
            predicate: BeforeUpdatePredicate
        ) {
            addTransformerWhenActionMatched { clazz, queryStructure, payload ->
                val data = queryStructure.update?.data ?: throw IllegalImplements("update 传入了空值。")
                val beanInstance = BeanProxy.fromBean(data, clazz)

                if (!predicate.predicate(beanInstance, queryStructure, payload)) {
                    return@addTransformerWhenActionMatched Result.ok(queryStructure)
                }

                val fieldType = beanInstance.getPropertyType(field)
                    ?: return@addTransformerWhenActionMatched Result.ok(queryStructure)

                if (fieldClass != null && fieldType != fieldClass) {
                    return@addTransformerWhenActionMatched Result.ok(queryStructure)
                }

                val oldValue = beanInstance.getProperty(field)
                if (oldValue != null && !override) {
                    return@addTransformerWhenActionMatched Result.ok(queryStructure)
                }
                val value = getValue() ?: throw ConfigException("beforeUpdate.add(override)Field, 不能传入null值, 如需将值更新为null，使用QueryProConst(Kt).NULL")
                if (value != Const.SKIP) {
                    if (value == Const.NULL) {
                        beanInstance.setProperty(field, null)
                    } else {
                        beanInstance.setProperty(field, value)
                    }
                }
                Result.ok(queryStructure)
            }
        }
    }

    class BeforeSelectTransformersBuilder internal constructor(): BaseQsTransformersBuilder(QueryStructureAction.SELECT) {
        /**
         * 添加一个StructureTransformer转换器
         *
         * 根据提供的 Class<Any>, QueryStructure, QueryPayload 信息，对QueryStructure进行转换并返回
         *
         * @param transformer [QueryStructureTransformer.transform]
         * + _kotlin_:
         * ```kotlin
         * fun transform(clazz: Class<Any>, qs: QueryStructure, payload: QueryPayload): Result<QueryStructure, Throwable>
         * ```
         * + _java_:
         * ```java
         * Result<QueryStructure, Throwable> transform(Class<?> clazz, QueryStructure qs, QueryPayload payload)
         * ```
         */
        override fun addTransformer(transformer: QueryStructureTransformer) =
            this.also { addTransformerWhenActionMatched(transformer) }
    }

    class BeforeDeleteTransformersBuilder internal constructor(): BaseQsTransformersBuilder(QueryStructureAction.DELETE) {
        /**
         * 添加一个StructureTransformer转换器
         *
         * 根据提供的 Class<Any>, QueryStructure, QueryPayload 信息，对QueryStructure进行转换并返回
         *
         * @param transformer [QueryStructureTransformer.transform]
         * + _kotlin_:
         * ```kotlin
         * fun transform(clazz: Class<Any>, qs: QueryStructure, payload: QueryPayload): Result<QueryStructure, Throwable>
         * ```
         * + _java_:
         * ```java
         * Result<QueryStructure, Throwable> transform(Class<?> clazz, QueryStructure qs, QueryPayload payload)
         * ```
         */
        override fun addTransformer(transformer: QueryStructureTransformer) =
            this.also { addTransformerWhenActionMatched(transformer) }
    }

    /**
     * ## 生命周期回调之：
     *
     * ### Before Select
     *
     * 会在`QueryPro.selectBy`的`QueryStructure`生成之后，`SQL`生成之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     *
     * @see [BeforeSelectTransformersBuilder]
     */
    fun beforeSelect(builder: (builder: BeforeSelectTransformersBuilder) -> BeforeSelectTransformersBuilder) = this.also {
        beforeExecTransformers.addAll(builder(BeforeSelectTransformersBuilder()).build())
    }

    /**
     * ## 生命周期回调之：
     *
     * ### Before Update
     *
     * 会在`QueryPro.updateSet`的`QueryStructure`生成之后，`SQL`生成之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     *
     * @see BeforeUpdateTransformersBuilder
     */
    fun beforeUpdate(builder: (builder: BeforeUpdateTransformersBuilder) -> BeforeUpdateTransformersBuilder) = this.also {
        beforeExecTransformers.addAll(builder(BeforeUpdateTransformersBuilder()).build())
    }

    /**
     * ## 生命周期回调之：
     *
     * ### Before Delete
     *
     * 会在`QueryPro.deleteBy`的`QueryStructure`生成之后，`SQL`生成之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     *
     * @see [BeforeDeleteTransformersBuilder]
     */
    fun beforeDelete(builder: (builder: BeforeDeleteTransformersBuilder) -> BeforeDeleteTransformersBuilder) = this.also {
        beforeExecTransformers.addAll(builder(BeforeDeleteTransformersBuilder()).build())
    }

    /**
     * ## 生命周期回调之：
     *
     * ### Before Insert
     *
     * 会在`QueryPro.insert`的`QueryStructure`生成之后，`SQL`生成之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     *
     * @see [BeforeInsertTransformersBuilder]
     */
    fun beforeInsert(builder: (builder: BeforeInsertTransformersBuilder) -> BeforeInsertTransformersBuilder) = this.also {
        beforeExecTransformers.addAll(builder(BeforeInsertTransformersBuilder()).build())
    }

    /**
     * ## 生命周期回调之：
     *
     * ### Before Exec
     *
     * 会在`QueryPro.selectBy`, `QueryPro.updateSet`, `QueryPro.deleteBy`, `QueryPro.insert`的`QueryStructure`生成之后，`SQL`生成之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     * @param transformer [QueryStructureTransformer.transform]
     * + _kotlin_:
     * ```kotlin
     * fun transform(clazz: Class<Any>, qs: QueryStructure, payload: QueryPayload): Result<QueryStructure, Throwable>
     * ```
     * + _java_:
     * ```java
     * Result<QueryStructure, Throwable> transform(Class<?> clazz, QueryStructure qs, QueryPayload payload)
     * ```
     */
    fun beforeExec(transformer: QueryStructureTransformer) = this.also {
        beforeExecTransformers.add(transformer)
    }

    abstract class BaseResultTransformersBuilder(private val action: QueryStructureAction) {
        private val transformers = mutableListOf<ResultWithQueryStructureTransformer>()
        protected fun addTransformerWhenActionMatched(transformer: ResultWithQueryStructureTransformer) = this.also {
            transformers.add { clazz, result, queryStructure, payload ->
                if (queryStructure.action == action) {
                    transformer.transform(clazz, result, queryStructure, payload)
                } else {
                    Result.ok(result)
                }
            }
        }
        abstract fun addTransformer(transformer: ResultWithQueryStructureTransformer): BaseResultTransformersBuilder
        fun build() = transformers
    }

    /**
     * 添加，修改，删除返回结果时的过滤器
     * 一般作为可选参数出现
     * 当返回结果为false时代表不执行相应的操作
     */
    fun interface AfterSelectPredicate {
        fun predicate(beanInstance: BeanProxy.BeanInstance<Any>?, qs: QueryStructure, payload: QueryPayload): Boolean
    }

    class AfterSelectTransformerBuilder internal constructor(): BaseResultTransformersBuilder(QueryStructureAction.SELECT) {
        override fun addTransformer(transformer: ResultWithQueryStructureTransformer) =
            this.also { addTransformerWhenActionMatched(transformer) }

        /**
         * 为select的返回结果添加某属性
         * 不会处理QueryProSql的返回结果
         * @param fieldClass 有时候，相同的 field 会有多种类型的取值，使用该字段可以区分这种情况
         * @param predicate 可选的过滤器，返回结果为false时，该字段不会被添加
         */
        @JvmOverloads
        fun <T> addField(
            field: String,
            fieldClass: Class<*>,
            value: () -> T,
            predicate: AfterSelectPredicate = AfterSelectPredicate { _, _, _ -> true }
        ) = this.also {
            overrideField(field, fieldClass, value, false, predicate)
        }

        /**
         * 为select的返回结果重写某属性
         * 不会处理QueryProSql的返回结果
         * @param fieldClass 有时候，相同的 field 会有多种类型的取值，使用该字段可以区分这种情况
         * @param predicate 可选的过滤器，返回结果为false时，该字段不会被重写
         */
        @JvmOverloads
        fun <T> overrideField(
            field: String,
            fieldClass: Class<*>,
            value: () -> T,
            predicate: AfterSelectPredicate = AfterSelectPredicate { _, _, _ -> true }
        ) = this.also {
            overrideField(field, fieldClass, value, true, predicate)
        }

        /**
         * 删除某属性
         * 不会处理QueryProSql的返回结果
         * @param fieldClass 可为null，为null时代表不区分类型
         * @param predicate 可选的过滤器，返回结果为false时，该字段不会被移除
         */
        @JvmOverloads
        fun <T> deleteField(
            field: String,
            fieldClass: Class<*>?,
            predicate: AfterSelectPredicate = AfterSelectPredicate { _, _, _ -> true }
        ) = this.also {
            overrideField(field, fieldClass, { Const.NULL }, true, predicate)
        }

        private fun <T> overrideField(
            field: String,
            fieldClass: Class<*>?,
            getValue: () -> T,
            override: Boolean,
            predicate: AfterSelectPredicate
        ) {
            super.addTransformerWhenActionMatched { clazz, result, queryStructure, payload ->
                fun doTransform(row: Any?) {
                    val beanInstance =  BeanProxy.fromBean(row ?: return, clazz ?: return)
                    if (!predicate.predicate(beanInstance, queryStructure, payload)) {
                        return
                    }

                    val fieldType = beanInstance.getPropertyType(field) ?: return
                    if (fieldClass != null && fieldType != fieldClass) {
                        return
                    }

                    val oldValue = beanInstance.getProperty(field)
                    if (oldValue != null && !override) {
                        return
                    }
                    val value = getValue() ?: throw ConfigException("afterSelect.add(override,delete)Field, 不能传入null值, 如需将值更新为null，使用QueryProConst(Kt).NULL")
                    if (value != Const.SKIP) {
                        if (value == Const.NULL) {
                            beanInstance.setProperty(field, null)
                        } else {
                            beanInstance.setProperty(field, value)
                        }
                    }
                }

                if (result is Iterable<*>) {
                    for (row in result) {
                        doTransform(row)
                    }
                } else {
                    doTransform(result)
                }

                return@addTransformerWhenActionMatched Result.ok(result)
            }
        }
    }

    class AfterUpdateTransformerBuilder internal constructor(): BaseResultTransformersBuilder(QueryStructureAction.UPDATE) {
        override fun addTransformer(transformer: ResultWithQueryStructureTransformer) =
            this.also { addTransformerWhenActionMatched(transformer) }
    }

    class AfterDeleteTransformerBuilder internal constructor(): BaseResultTransformersBuilder(QueryStructureAction.DELETE) {
        override fun addTransformer(transformer: ResultWithQueryStructureTransformer) =
            this.also { addTransformerWhenActionMatched(transformer) }
    }

    class AfterInsertTransformerBuilder internal constructor(): BaseResultTransformersBuilder(QueryStructureAction.INSERT) {
        override fun addTransformer(transformer: ResultWithQueryStructureTransformer) =
            this.also { addTransformerWhenActionMatched(transformer) }
    }

    /**
     * ## 生命周期回调之：
     *
     * ### After Select
     *
     * 会在`QueryPro.selectBy`执行之后，返回之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     */
    fun afterSelect(builder: (builder: AfterSelectTransformerBuilder) -> AfterSelectTransformerBuilder) = this.also {
        afterExecTransformers.addAll(builder(AfterSelectTransformerBuilder()).build())
    }

    /**
     * ## 生命周期回调之：
     *
     * ### After Update
     *
     * 会在`QueryPro.updateSet`执行之后，返回之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     */
    fun afterUpdate(builder: (builder: AfterUpdateTransformerBuilder) -> AfterUpdateTransformerBuilder) = this.also {
        afterExecTransformers.addAll(builder(AfterUpdateTransformerBuilder()).build())
    }

    /**
     * ## 生命周期回调之：
     *
     * ### After Delete
     *
     * 会在`QueryPro.deleteBy`执行之后，返回之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     */
    fun afterDelete(builder: (builder: AfterDeleteTransformerBuilder) -> AfterDeleteTransformerBuilder) = this.also {
        afterExecTransformers.addAll(builder(AfterDeleteTransformerBuilder()).build())
    }

    /**
     * ## 生命周期回调之：
     *
     * ### After Insert
     *
     * 会在`QueryPro.insert`执行之后，返回之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     */
    fun afterInsert(builder: (builder: AfterInsertTransformerBuilder) -> AfterInsertTransformerBuilder) = this.also {
        afterExecTransformers.addAll(builder(AfterInsertTransformerBuilder()).build())
    }

    /**
     * ## 生命周期回调之：
     *
     * ### After Exec
     *
     * 会在`QueryPro.selectBy`, `QueryPro.updateSet`, `QueryPro.deleteBy`, `QueryPro.insert`执行之后，返回之前调用
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     * @param transformer [ResultWithQueryStructureTransformer.transform]
     * + _kotlin_:
     * ```kotlin
     * fun transform(clazz: Class<Any>?, result: Any, qs: QueryStructure, payload: QueryPayload): Result<Any, Throwable>
     * ```
     * + _java_:
     * ```java
     * Result<Object, Throwable> transform(Class<?> clazz, Object result, QueryStructure qs, QueryPayload payload)
     * ```
     */
    fun afterExec(transformer: ResultWithQueryStructureTransformer) = this.also {
        afterExecTransformers.add(transformer)
    }

    /**
     * ## 生命周期回调之：
     *
     * ### 运行`SQL`前：
     *
     * 会在执行`SQL`之前调用，以便记录`SQL`信息或对`SQL and it's params`做一些转换或阻止`SQL`的执行
     *
     * ## &nbsp;
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     * @param transformer [SqlAndParamsTransformer.transform]
     * + _kotlin_:
     * ```kotlin
     * fun transform(sqlAndParams: SqlAndParams): Result<SqlAndParams, Throwable>
     * ```
     * + _java_:
     * ```java
     * Result<SqlAndParams , Throwable> transform(SqlAndParams sqlAndParams)
     * ```
     */
    fun beforeRunSql(transformer: SqlAndParamsTransformer) = this.also {
        beforeRunSqlTransformers.add(transformer)
    }

    /**
     * ## 生命周期回调之：
     *
     * ### 运行`SQL`后：
     *
     * 会在执行`SQL`之后调用，以便
     * 1. 对返回结果进行记录
     * 2. 对返回结果进行转换
     * 3. 阻止结果的返回
     *
     * ## 其它参考信息：
     *
     * **`.selectBy()...`, `.updateSet()...`, `.deleteBy()...`, `(QueryPro).insert...`的生命周期：**
     * ```
     * [beforeExec, beforeSelect](取决于先调用哪个) ->
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * [afterExec, afterSelect] ->
     * return result
     * ```
     *
     * **`QueryProSql...`的生命周期**
     * ```
     * [beforeRunSql, runSqlAround] ->
     * [afterRunSql, runSqlAround] ->
     * return result
     * ```
     * @param transformer [SqlAndParamsTransformer.transform]
     * + _kotlin_:
     * ```kotlin
     * fun transform(result: Any): Result<Any, Throwable>
     * ```
     * + _java_:
     * ```java
     * Result<Object, Throwable> transform(Object result)
     * ```
     */
    fun afterRunSql(transformer: ResultTransformer) = this.also {
        afterRunSqlTransformers.add(transformer)
    }
}