package cn.cloudself.query;

import cn.cloudself.query.config.HashMapStore;
import cn.cloudself.query.config.QueryProConfigDb;
import cn.cloudself.query.exception.IllegalCall;
import cn.cloudself.query.exception.IllegalParameters;
import cn.cloudself.query.psi.structure.QueryStructureAction;
import cn.cloudself.query.resolver.DbType;
import cn.cloudself.query.resolver.Resolver;
import cn.cloudself.query.util.IO;
import cn.cloudself.query.util.Pair;
import cn.cloudself.query.util.SqlUtils;
import org.intellij.lang.annotations.Language;
import org.jetbrains.annotations.NotNull;

import javax.sql.DataSource;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 用于：
 * 1. 直接执行`SQL`语句
 * 2. 动态生成插入语句并执行
 */
public class QueryProSql {
    /**
     * 直接执行{@code sql}语句, 例子:
     * <pre>{@code
     * QueryProSql.create("SELECT * FROM user WHERE username = ?", "hb").query();
     * }</pre>
     * @param sql sql语句，参数使用 ? 的格式
     */
    public static Action create(@NotNull String sql, Object... params) {
        return new Action(sql, formatVararg(params));
    }

    /**
     * 直接执行{@code sql}语句
     * @param inputStream sql语句
     */
    public static Action create(@NotNull InputStream inputStream, Object... params) {
        return new Action(IO.readBytesAsString(inputStream), formatVararg(params));
    }

    /**
     * 直接执行classpath下的{@code sql}语句(例如resources目录下), 例子:
     * <pre>{@code
     * // 例如，resources下有个sql文件夹，里面有个user.sql文件
     * QueryProSql.createFromClassPath("sql/user.sql").query();
     * }</pre>
     * @param path sql文件在 resources, src/main/java, src/main/kotlin等路径下的路径
     */
    public static Action createFromClassPath(@NotNull String path, Object... params) {
        try (final InputStream is = QueryProSql.class.getClassLoader().getResourceAsStream(path)) {
            if (is == null) {
                throw new IllegalParameters("路径{0}可能不是标准的ClassPath", path);
            }
            return create(is, params);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static InsertAction create() {
        return new InsertAction();
    }

    public static BatchAction createBatch() {
        return new BatchAction();
    }

    public static abstract class AbstractAction<T> {
        protected final HashMapStore store = new HashMapStore();
        protected final QueryProConfigDb config = new QueryProConfigDb(store);

        protected abstract T that();

        /**
         * 更改数据库类型，默认为mysql
         */
        public T dbType(DbType dbType) {
            this.config.dbType(dbType);
            return that();
        }

        /**
         * take 方法，用于写if 条件等
         */
        public T take(Function<T, T> factor) {
            return factor.apply(that());
        }

        /**
         * 临时切换数据源(使用Connection切换)
         * 相比起[ofConnection]更推荐使用[ofDataSource], 因为[ofDataSource]会自动管理链接的开启与关闭。
         * 而[ofConnection]需要手动管理链接的开启与关闭(这在存在事务的时候需要额外注意)
         */
        public T ofConnection(Connection connection) {
            this.config.connection(connection);
            return that();
        }

        /**
         * 临时切换数据源
         */
        public T ofDataSource(DataSource dataSource) {
            this.config.dataSource(dataSource);
            return that();
        }
    }

    public static class Action extends AbstractAction<Action> {
        @Language("SQL")
        private final String sql;
        private final Object[] params;
        public Action(@Language("SQL") String sql, Object[] params) {
            this.sql = sql;
            this.params = params;
        }
        @Override
        protected Action that() {
            return this;
        }

        /**
         * 查询单个对象
         *
         * @param clazz 支持JavaBean, 支持Map, 支持基本类型(Long, String, Date, Enum等, 具体参考[QueryProConfig.global.addResultSetParser])
         */
        public <T> T queryOne(@NotNull Class<T> clazz) {
            return query(clazz).get(0);
        }

        /**
         * 查询单个对象
         */
        public Map<String, Object> queryOne() {
            return query().get(0);
        }

        /**
         * 查询多个对象
         *
         * @param clazz 支持JavaBean, 支持Map, 支持基本类型(Long, String, Date, Enum等, 具体参考[QueryProConfig.global.addResultSetParser])
         */
        public <T> List<T> query(@NotNull Class<T> clazz) {
            return Resolver
                    .create(() -> store)
                    .use(resolver -> resolver.resolve(sql, params, clazz, QueryStructureAction.SELECT));
        }

        /**
         * 查询多个对象
         */
        public List<Map<String, Object>> query() {
            return query(HashMap.class).stream().map(m -> {
                //noinspection unchecked
                return (Map<String, Object>) m;
            }).collect(Collectors.toList());
        }

        /**
         * 使用单条语句执行更新，创建，删除等非select语句
         */
        public int exec() {
            return Resolver
                    .create(() -> store)
                    .use(resolver -> resolver.resolve(sql, params, Integer.class, QueryStructureAction.UPDATE))
                    .get(0);
        }

        public BatchAction autoSplit() {
            final BatchAction batchAction = new BatchAction().autoSplit(sql, params);
            batchAction.store.copyFrom(this.store);
            return batchAction;
        }
    }

    public static class BatchAction extends AbstractAction<BatchAction> {
        private String[] sqlArray;
        private Object[][] paramsArray;

        @Override
        protected BatchAction that() {
            return this;
        }

//    /**
//     * 使用多条语句和参数执行更新，创建，删除等非select语句
//     */
//    public static BatchAction createBatch(@NotNull List<Pair<String, Object[]>> sqlAndParamsList) {
//        final int len = sqlAndParamsList.size();
//        final String[] sqlArray = new String[len];
//        final Object[][] paramsArray = new Object[len][];
//        for (int i = 0; i < len; i++) {
//            final Pair<String, Object[]> sqlAndParams = sqlAndParamsList.get(i);
//            sqlArray[i] = sqlAndParams.first();
//            paramsArray[i] = sqlAndParams.second();
//        }
//        return new BatchAction(sqlArray, paramsArray);
//    }
//
//    public static BatchAction createBatch(@NotNull List<String> sqlArr) {
//        return createBatch(sqlArr.toArray(new String[]{}), null);
//    }
//
//    public static BatchAction createBatch(@NotNull String[] sqlArr) {
//        return createBatch(sqlArr, null);
//    }
//
//    /**
//     * 使用多条语句和参数执行更新，创建，删除等非select语句
//     *
//     * @param sqlArr 多条sql语句
//     * @param params 参数数组数组，长度必须和sqlArr一支
//     */
//        @JvmStatic
//        @JvmOverloads
//    public static BatchAction createBatch(@NotNull String[] sqlArr, Object[][] params: Array<Array<Any?>>? = null): BatchAction {
//        var nonNullParams = params
//        if (nonNullParams == null) {
//            nonNullParams = sqlArr.map { arrayOf<Any?>() }.toTypedArray()
//        } else if (sqlArr.size != nonNullParams.size && sqlArr.size != 1) {
//            throw IllegalParameters("sqlArr的长度必须和params的长度一致")
//        }
//        return BatchAction(sqlArr, nonNullParams)
//    }

//        /**
//         * 使用单条语句和多组参数执行更新，创建，删除等非select语句
//         *
//         * @param sql 单条sql语句 e.g. INSERT INTO user (id, name) VALUES (?, ?)
//         * @param params 参数数组数组 e.g. [[1, 'hb'], [2, 'l']]
//         */
//        @JvmStatic
//        @JvmOverloads
//        fun createBatch(@Language("SQL") sql: String, params: Array<Array<Any?>> = arrayOf()): BatchAction {
//            return BatchAction(arrayOf(sql), params)
//        }

        /**
         * 使用含;的复合语句组合执行更新操作,
         * 该语句会对传入的sql按照;进行拆分
         *
         * @param sqlGroup 复合sql语句 e.g. INSERT INTO user (id, name) VALUES (?, ?); UPDATE user SET name = UPPER(name) WHERE id = ?
         * @param params e.g. [1, 'hb, 1]
         */
        public BatchAction autoSplit(@Language("SQL") String sqlGroup, Object... params) {
            final List<Pair<String, Integer>> sqlAndCountArr = SqlUtils.splitBySemicolonAndCountQuestionMark(sqlGroup);
            final int size = sqlAndCountArr.size();
            sqlArray = new String[size];
            paramsArray = new Object[size][];

            int i = 0;
            int j = 0;
            for (Pair<String, Integer> sqlAndCount : sqlAndCountArr) {
                final String sql = sqlAndCount.first();
                final Integer count = sqlAndCount.second();
                sqlArray[i] = sql;
                paramsArray[i] = Arrays.copyOfRange(params, j, count);
                i++;
                j += count;
            }
            return this;
        }


        /**
         * 批量更新
         *
         * @param clazz 支持的值有：
         * * Int::class.java 总共影响的条数
         * * IntArray::class.java, List::class.java, listOf<Int>().javaClass等 代表每条语句更新的条数
         * * Boolean::class.java 不建议使用，仅简单的判断了总更新条数是否大于1
         */
        public <T> T exec(Class<T> clazz) {
            return Resolver.create(() -> store).use(resolver -> resolver.exec(sqlArray, paramsArray, clazz));
        }

        /**
         * 批量更新
         */
        public int exec() {
            return exec(Integer.class);
        }
    }

    public static class InsertAction extends AbstractAction<InsertAction> {
        @Override
        protected InsertAction that() {
            return this;
        }

        public InsertAction maxParameterSize(Integer size) {
            this.config.maxParameterSize(size);
            return that();
        }

        /**
         * 通过 objs 配合数据库表名，动态生成sql语句，并执行
         */
        @SafeVarargs
        public final <V> void insert(String table, Map<String, V>... objs) {
            insert(table, Arrays.asList(objs), null);
        }

        /**
         * 通过 objs 配合数据库表名，动态生成sql语句，并执行
         */
        public <V, M extends Map<String, V>> void insert(String table, Collection<M> objs) {
            insert(table, objs, null);
        }

        /**
         * 通过 objs 配合数据库表名，动态生成sql语句，并执行
         *
         * @param idColumnClazz 指定需要返回的ID的类型
         */
        public <ID, V, M extends Map<String, V>> List<ID> insert(String table, Collection<M> objs, Class<ID> idColumnClazz) {
            return Resolver.create(() -> store).use(resolver -> resolver.insert(objs, table, idColumnClazz));
        }
    }

    /**
     * 将 [Collection] 转为 Object[]
     */
    private static Object[] formatVararg(Object[] vararg) {
        if (vararg == null) {
            return null;
        }
        if (vararg.length > 0) {
            final Object first = vararg[0];
            if (first instanceof Collection) {
                if (vararg.length > 1) {
                    throw new IllegalCall("第一个参数为collection时，参数个数不能大于1。");
                }
                return ((Collection<?>) first).toArray();
            }
        }
        return vararg;
    }
}
