package cn.cloudself.query

import cn.cloudself.query.config.*
import cn.cloudself.query.exception.IllegalCall
import cn.cloudself.query.exception.IllegalImplements
import cn.cloudself.query.plus.Alias
import cn.cloudself.query.plus.KQueryProPlus
import cn.cloudself.query.plus.QueryProPlus
import cn.cloudself.query.resolver.Resolver
import cn.cloudself.query.psi.CreateQueryField
import cn.cloudself.query.psi.AbstractExpression
import cn.cloudself.query.psi.Const
import cn.cloudself.query.psi.UpdateSetDefinedExpression
import cn.cloudself.query.psi.annotations.DefaultValue
import cn.cloudself.query.psi.annotations.JvHidden
import cn.cloudself.query.psi.annotations.KtHidden
import cn.cloudself.query.psi.annotations.PureContract
import cn.cloudself.query.psi.structure.*
import cn.cloudself.query.util.*
import org.jetbrains.annotations.Contract

class QueryPro<
        T: Any,
        ID: Any,
        SELECT_BY_FIELD: AbstractExpression<*, *, *, *, *, *>,
        ORDER_BY_FIELD: AbstractExpression<*, *, *, *, *, *>,
        UPDATE_SET_FIELD: UpdateSetDefinedExpression<UPDATE_BY_FIELD>,
        UPDATE_BY_FIELD: AbstractExpression<*, *, *, *, *, *>,
        DELETE_BY_FIELD: AbstractExpression<*, *, *, *, *, *>
> constructor(
    private val queryProClass: Class<*>,
    private val clazz: Class<T>,
    private val queryStructure: QueryStructure,
    private val createSelectByField: CreateQueryField<SELECT_BY_FIELD>,
    private val createOrderByField: CreateQueryField<ORDER_BY_FIELD>,
    private val createUpdateSetField: CreateQueryField<UPDATE_SET_FIELD>,
    private val createUpdateByField: CreateQueryField<UPDATE_BY_FIELD>,
    private val createDeleteByField: CreateQueryField<DELETE_BY_FIELD>,
    private val payload: QueryPayload,
    codeStoreDbW: CodeStoreDbW<QueryPro<T, ID, SELECT_BY_FIELD, ORDER_BY_FIELD, UPDATE_SET_FIELD, UPDATE_BY_FIELD, DELETE_BY_FIELD>> = CodeStoreDbW(payload.configs),
): IQueryProConfigDbWriteable<QueryPro<T, ID, SELECT_BY_FIELD, ORDER_BY_FIELD, UPDATE_SET_FIELD, UPDATE_BY_FIELD, DELETE_BY_FIELD>> by codeStoreDbW {
    /**
     * 该构造函数为了兼容Java
     */
    constructor(
        queryProClass: Class<*>,
        clazz: Class<T>,
        queryStructure: QueryStructure,
        createSelectByField: CreateQueryField<SELECT_BY_FIELD>,
        createOrderByField: CreateQueryField<ORDER_BY_FIELD>,
        createUpdateSetField: CreateQueryField<UPDATE_SET_FIELD>,
        createUpdateByField: CreateQueryField<UPDATE_BY_FIELD>,
        createDeleteByField: CreateQueryField<DELETE_BY_FIELD>,
    ): this(queryProClass, clazz, queryStructure, createSelectByField, createOrderByField, createUpdateSetField, createUpdateByField, createDeleteByField, QueryPayload(clazz, queryProClass, HashMapStore()))

    init {
        codeStoreDbW.that = this
    }

    class CodeStoreDbW<T: IQueryProConfigDbWriteable<T>>(store: Store): DefaultQueryProConfigDbWriteable<T>(store) {
        var that: T? = null
        override fun that() = that ?: throw IllegalImplements("Qp必须手动初始化that")
    }

    private val logger = LogFactory.getLog(QueryPro::class.java)
    private val parsedClass = ClassParser.parse(clazz)

    /**
     * 查询操作
     */
    fun selectBy(): SELECT_BY_FIELD {
        queryStructure.action = QueryStructureAction.SELECT
        return createSelectByField(queryStructure, payload)
    }

    /**
     * 查询全部数据
     */
    fun selectAll(): List<T> {
        @Suppress("UNCHECKED_CAST")
        return selectBy().run() as List<T>
    }

    fun selectByObj(obj: T): SELECT_BY_FIELD {
        val javaNameMapDbName = mutableMapOf<String, String>()
        for ((_, column) in parsedClass.columns) {
            javaNameMapDbName[column.javaName] = column.dbName
        }

        val whereClauses = ObjectUtil.toSequence(obj).toList()
            .map { column -> column.dbName to column.value }
            .filter { (_, v) -> v != null }
            .map { (k, v) ->
                WhereClause(Field(null, k), Const.EQUAL, v)
            }

        queryStructure.action = QueryStructureAction.SELECT
        queryStructure.where = whereClauses
        queryStructure.limit = 0 to 1
        return createSelectByField(queryStructure, payload)
    }

    /**
     * 使用主键查询
     */
    fun selectByPrimaryKey(value: Any): T? {
        val idColumn = parsedClass.idColumn ?: throw IllegalCall("Class {0} 没有找到主键", clazz.name)
        val whereClause = WhereClause(Field(null, idColumn), Const.EQUAL, value)
        queryStructure.action = QueryStructureAction.SELECT
        queryStructure.where = listOf(whereClause)
        queryStructure.limit = 0 to 1
        @Suppress("UNCHECKED_CAST")
        return createSelectByField(queryStructure, payload).runLimit1() as T?
    }

    /**
     * 排序操作
     */
    fun orderBy(): ORDER_BY_FIELD {
        queryStructure.action = QueryStructureAction.SELECT
        return createOrderByField(queryStructure, payload)
    }

    /**
     * 更新操作 例子：updateSet().kee("new-key").where.id.equalTo(1).run()
     * 注意如果要更新的值传入null(例子中是kee), 则会报错,
     * 如确实需要更新为null, 使用
     * kotlin: updateSet().kee(NULL).where.id.equalTo(1).run()
     * java: updateSet().kee(QueryProConstKt.NULL).where.id().equalTo(1).run()
     */
    @PureContract
    @Contract(pure = true)
    fun updateSet(): UPDATE_SET_FIELD {
        queryStructure.action = QueryStructureAction.UPDATE
        queryStructure.update = Update(
            data = mutableMapOf<String, Any>(), override = false, id = parsedClass.idColumn
        )
        return createUpdateSetField(queryStructure, payload)
    }

    /**
     * 更新操作
     *
     * updateSet(Apple(id = 2021, name = "iPhone13", type = null)).run()
     * 如果 需要更新的值为null, 则跳过该字段不更新
     * 如确实需要更新, 使用
     * updateSet(Apple(id = 2021, name = "iPhone13", type = null), true).run()
     * 如果需要更新的值更新的值为null, 会将其更新为null
     */
    @PureContract
    @Contract(pure = true)
    @JvmOverloads
    fun updateSet(obj: T, @DefaultValue("false") override: Boolean = false): UpdateSetDefinedExpression<UPDATE_BY_FIELD> {
        val columns = parsedClass.columns.map { it.key }
        val update = Update(data = obj, columns = columns, override = override, id = parsedClass.idColumn)
        queryStructure.action = QueryStructureAction.UPDATE
        queryStructure.update = update
        return UpdateSetDefinedExpression(queryStructure, payload, createUpdateByField)
    }

    @PureContract
    @Contract(pure = true)
    fun updateSet(obj: Map<String, Any?>): UpdateSetDefinedExpression<UPDATE_BY_FIELD> {
        val data = mutableMapOf<String, Any?>()
        for ((key, value) in obj) {
            if (parsedClass.getColumnDbFieldName(key) != null) {
                data[key] = value
            } else {
                val column = parsedClass.getColumnByJavaPropertyName(key)
                if (column == null) {
                    logger.warn("字段{0}已被跳过", key)
                    continue
                }
                data[column.dbName] = value
            }
        }
        val update = Update(data, override = true)
        queryStructure.action = QueryStructureAction.UPDATE
        queryStructure.update = update
        return UpdateSetDefinedExpression(queryStructure, payload, createUpdateByField)
    }

    /**
     * 删除操作
     */
    fun deleteBy(): DELETE_BY_FIELD {
        queryStructure.action = QueryStructureAction.DELETE
        return createDeleteByField(queryStructure, payload)
    }

    /**
     * 使用主键删除
     */
    fun deleteByPrimaryKey(keyValue: Any): Boolean {
        queryStructure.action = QueryStructureAction.DELETE
        queryStructure.where = listOf(WhereClause(Field(null, parsedClass.idColumn ?: "id"), Const.EQUAL, keyValue))
        return createDeleteByField(queryStructure, payload).run() == true
    }

    /**
     * 插入操作
     */
    fun insert(obj: T): ID? {
        return insert(listOf(obj)).getOrNull(0)
    }

    /**
     * 插入操作
     */
    fun insert(vararg objs: T): List<ID?> {
        return insert(listOf(*objs))
    }

    @SafeVarargs
    @Suppress("UNCHECKED_CAST")
    fun insert(vararg objs: Map<String, Any?>): List<ID?> {
        return insert(listOf(*objs) as Collection<T>)
    }

    /**
     * 批量插入
     */
    @Suppress("UNCHECKED_CAST")
    fun insert(collection: Collection<T>): List<ID?> {
        queryStructure.action = QueryStructureAction.INSERT
        queryStructure.insert = Insert(collection)
        return Resolver.create(clazz as Class<Any>, queryStructure, payload)
            .use { resolve(it, payload, clazz) } as List<ID?>
    }

    @KtHidden
    @JvmName("plus")
    fun plusJava(): QueryProPlus<T, T> = QueryProPlus(queryStructure, payload, clazz)

    @KtHidden
    @JvmName("plus")
    fun plusJava(alias: String): QueryProPlus<T, Alias> = QueryProPlus(queryStructure, payload, clazz, alias)

    @JvHidden
    @JvmName("_kt_plus")
    fun plus(): KQueryProPlus<T> = KQueryProPlus(queryStructure, payload)

    @JvHidden
    @JvmName("_kt_plus")
    fun plus(alias: String): KQueryProPlus<Alias> = KQueryProPlus(queryStructure, payload, alias)
}
