package cn.cloudself.query.util

import cn.cloudself.query.config.QueryProConfig
import cn.cloudself.query.exception.UnSupportException
import kotlin.collections.HashMap
import kotlin.collections.LinkedHashMap

private class Ref<R>(var value: R)

private val beanProxyCaches = mutableMapOf<Class<*>, BeanProxy<*>>()

private val logger = LogFactory.getLog(BeanProxy::class.java)

/**
 * Bean代理，
 * 支持生成三种类型的数据: `Map`, `基本对象`, `JavaBean`
 *
 * `BeanProxy`的主要目的是虚拟出一个对象
 *
 * 使用[BeanProxy.fromClass]或[BeanProxy.fromBean]构造该对象
 * 使用[BeanProxy.newInstance]创建临时对象[BeanInstance]
 * 使用[BeanInstance.setProperty]设置目标对象的属性
 * 使用[BeanInstance.getPropertyType]获取目标某字段的类型
 * 使用[BeanInstance.toBean]将临时对象转为目标对象
 */
class BeanProxy<B: Any> private constructor(
    private val clazz: Class<B>,
    private val createInstance: () -> Any,
    private val getParsedClass: () -> ClassParser.ParsedClass,
    private val setProperty: (o: Any, p: String, v: Any?) -> Unit,
    private val getProperty: (o: Any, p: String) -> Any?,
    private val getPropertyType: (p: String) -> Class<*>?,
    private val toResult: (o: Any) -> B?,
) {

    class BeanInstance<B: Any>(
        private val instance: B,
        private val proxy: BeanProxy<B>,
        private val clazz: Class<Any>? = null
    ) {
        /**
         * 设置一个属性，该方法不会自动转换value的类型。
         *
         * 如代理对象的`id`是`Long`或者`Int`类型，现在有一个`BigDecimal`类型的值，如何将这个`BigDecimal`设置到`id`上呢？
         *
         * 参考如下代码：
         *
         * ```java
         * BeanInstance<Object> instance = BeanProxy.fromBean(bean);
         * String field_name = "id";
         *
         * Object typedValue = new ObjectMapper().convertValue(bigDecimalId, instance.getPropertyType(field_name));
         * instance.setProperty(field_name, typedValue);
         * ```
         */
        fun setProperty(db_field_name: String, value: Any?) = this.also { proxy.setProperty(instance, db_field_name, value) }

        /**
         * get
         */
        fun getProperty(db_field_name: String) = proxy.getProperty(instance, db_field_name)

        /**
         * 获取某属性的类型
         */
        fun getPropertyType(db_field_name: String): Class<*>? {
            if (clazz != null) {
                if (clazz != proxy.clazz) {
                    return ClassParser.parse(clazz).getColumnDbFieldName(db_field_name)?.javaType
                }
            }
            return if (instance is Map<*, *>)
                instance[db_field_name]?.javaClass
            else
                proxy.getPropertyType(db_field_name)
        }

        /**
         * 获取bean
         */
        fun toBean(): B? = proxy.toResult(instance)

        /**
         * 获取bean的类型
         */
        fun getBeanType(): Class<B> = proxy.clazz

        fun getParsedClass() = proxy.getParsedClass()

        /**
         * 转Sequence
         *
         * key: column_name,
         * value: value
         */
        fun toSequence(): Sequence<Map.Entry<String, Any?>> {
            if (instance is Map<*, *>) {
                return sequence {
                for ((key, value) in instance) {
                    if (key is String) {
                        yield(java.util.AbstractMap.SimpleImmutableEntry(key, value))
                    } else {
                        throw UnSupportException("不支持非Map<String, *>类型的Map")
                    }
                }
            }
            }
            val columns = getParsedClass().columns
            return sequence {
                for ((key, column) in columns) {
                    yield(java.util.AbstractMap.SimpleImmutableEntry(key, column.getter(instance)))
                }
            }
        }

        /**
         * 以Map.Entry的形式迭代。
         */
        fun iterator() = toSequence().iterator()
    }

    companion object {
        @Suppress("UNCHECKED_CAST")
        private fun <T: Any, R: Any> create(
            clazz: Class<R>,
            /**
             * 创建目标对象或临时对象
             */
            createInstance: () -> T,
            /**
             * 获取属性列表
             */
            getParsedClass: () -> ClassParser.ParsedClass,
            /**
             * 设置属性
             */
            setProperty: (o: T, p: String, v: Any?) -> Unit,
            /**
             * 获取属性
             */
            getProperty: (o: T, p: String) -> Any?,
            /**
             * 获取某属性的类型
             * 当clazz为Map类型时，返回null
             */
            getPropertyType: (p: String) -> Class<*>?,
            /**
             * 转为目标对象
             */
            toResult: (o: T) -> R? = { it as R? },
        ) = BeanProxy(
            clazz,
            createInstance,
            getParsedClass,
            setProperty as (Any, String, Any?) -> Unit,
            getProperty as (Any, String) -> Any?,
            getPropertyType,
            toResult as (Any) -> R
        )

        @JvmStatic
        fun <R: Any> fromClass(clazz: Class<R>): BeanProxy<R> {
            val cachedBeanProxy = beanProxyCaches[clazz]
            if (cachedBeanProxy != null) {
                @Suppress("UNCHECKED_CAST")
                return cachedBeanProxy as BeanProxy<R>
            }

            val beanProxy = when {
                // 需要的类型是一个map
                Map::class.java.isAssignableFrom(clazz) -> {
                    val createInstance = when {
                        LinkedHashMap::class.java.isAssignableFrom(clazz) -> { { LinkedHashMap<String, Any?>() } }
                        HashMap::class.java.isAssignableFrom(clazz) -> { { HashMap<String, Any?>() } }
                        else -> { throw UnSupportException("不支持LinkedHashMap与HashMap<String, Object>以外的Map") }
                    }
                    create(
                        clazz,
                        createInstance,
                        getParsedClass = { throw UnSupportException("不支持Map转properties") },
                        setProperty = { result, property, value -> result[property] = value },
                        getProperty = { result, property -> result[property] },
                        getPropertyType = { null }
                    )
                }
                // 需要的类型是一个数据库基本类型(Long, Int, Date等)
                QueryProConfig.final.supportedColumnType().any { it.isAssignableFrom(clazz) } -> {
                    create<Ref<R?>, R>(
                        clazz,
                        createInstance = { Ref(null)  },
                        getParsedClass = { throw UnSupportException("不支持基本类型转properties") },
                        setProperty = { o, _, v ->
                            @Suppress("UNCHECKED_CAST")
                            o.value = v as R
                        },
                        getProperty = { o, _ -> o.value },
                        getPropertyType = { clazz },
                        toResult = { it.value }
                    )
                }
                // 需要返回的是一个JavaBean
                else -> {
                    try {
                        val noArgConstructor = clazz.getDeclaredConstructor()
                        val parsedClass = ClassParser.parse(clazz)
                        val columns = parsedClass.columns

                        create<R, R>(
                            clazz,
                            createInstance = { noArgConstructor.newInstance() },
                            getParsedClass = { parsedClass },
                            setProperty = { r, p, v ->
                                val column = columns[p]
                                if (column == null) {
                                    logger.warn("给定的类型${clazz.simpleName}中不存在字段$p, 该字段的赋值已跳过。")
                                    return@create
                                }
                                column.setter(r, v)
                            },
                            getProperty = { r, p ->
                                val column = columns[p]
                                if (column == null) {
                                    logger.warn("没有找到该字段 $p")
                                    return@create null
                                }
                                column.getter(r)
                            },
                            getPropertyType = { p -> columns[p]?.javaType }
                        )
                    } catch (e: Exception) {
                        throw UnSupportException(e, "{0} 没有找到无参构造函数，该类是一个JavaBean吗, " +
                                "对于Kotlin，需使用kotlin-maven-noarg生成默认的无参构造函数" +
                                "或在生成工具中配置QueryProFileMaker.disableKtNoArgMode()来生成默认的构造函数", clazz)
                    }
                }
            }

            beanProxyCaches[clazz] = beanProxy
            return beanProxy
        }

        /**
         * 构造一个BeanInstance
         *
         * @param bean 只能为`Java Bean`, 不能为`Map`, 如需为`Map`, 参考[fromBean(R, Class<Any>)]
         */
        @JvmStatic
        fun <R: Any> fromBean(bean: R): BeanInstance<R> {
            return BeanInstance(bean, fromClass(bean.javaClass))
        }

        /**
         * 构造一个BeanInstance
         *
         * @param bean 可以为Map或者JavaBean,
         * 为Map时，BeanInstance.getParsedClass无法调用，如需要可以手动调用`BeanProxy.fromClass(refer).getParsedClass()`获取
         *
         * @param refer 该Map参考的`Bean Class`. (目前仅`getPropertyType`方法使用到, V1.1.4)
         */
        @JvmStatic
        fun <R: Any> fromBean(bean: R, refer: Class<Any>): BeanInstance<R> {
            return BeanInstance(bean, fromClass(bean.javaClass), refer)
        }
    }

    @Suppress("UNCHECKED_CAST")
    fun newInstance() = BeanInstance(createInstance() as B, this)
}
