package cn.cloudself.query.resolver

import cn.cloudself.query.NULL
import cn.cloudself.query.QueryPayload
import cn.cloudself.query.QueryStructure
import cn.cloudself.query.QueryStructureAction
import cn.cloudself.query.config.QueryProConfig
import cn.cloudself.query.config.SqlAndParams
import cn.cloudself.query.exception.IllegalImplements
import cn.cloudself.query.exception.IllegalParameters
import cn.cloudself.query.exception.UnSupportException
import cn.cloudself.query.util.*

import cn.cloudself.query.resolver.IQueryStructureResolver.Executor

/**
 * IQueryStructureResolver的模板类，
 * 一般继承该类重写 相应的抽象方法即可
 * queryStructureToSql, insertObjectToSql, getColumnsDynamic也是可以重写的
 *
 * 如需自行实现IQueryStructureResolver, 需要处理生命周期方法beforeRunSql, afterRunSql以及添加对dryRun模式的支持
 */
abstract class QSR: IQueryStructureResolver {
    /**
     * @param clazz 需返回的对象类型
     */
    abstract fun <T: Any> doSelect(queryPro: Class<*>?, sql: String, params: Array<out Any?>, clazz: Class<T>): List<T>

    /**
     * @param clazz 需返回的对象的类型，一般添加对 Integer, Boolean的支持即可
     */
    abstract fun <T: Any> doUpdate(queryPro: Class<*>?, sql: String, params: Array<out Any?>, clazz: Class<T>): T

    /**
     * @param clazz insert方法需返回id列表，clazz是指ID的类型 如Long, Integer, String等
     */
    abstract fun <ID: Any> doInsert(queryPro: Class<*>?, sql: String, params: Array<out Any?>, clazz: Class<ID>?): List<ID>

    /**
     * 提供`QueryStructure`，将其解析为`SQL`
     *
     * `SQL`中的参数 默认的实现为`JDBC`支持的`?`的风格
     *
     * 该方法可重写
     *
     * @sample ToSqlByQueryStructure
     */
    open fun queryStructureToSql(queryStructure: QueryStructure) =
        ToSqlByQueryStructure(queryStructure).toSqlWithIndexedParams().let { it.first to it.second.toTypedArray() }

    class Column(
        val column: String,
        val getter: (o: Any) -> Any?,
        val isId: Boolean,
    )

    /**
     * 提供：表名，列信息，
     * 将需要插入的对象转为sql语句
     *
     * 约定：当返回结果的size为1且groupedParams有值。表明数据量较大，使用批量插入模式
     * (可优化)
     *
     * @sample ToSqlByInsertObjects.toSql
     */
    open fun insertObjectToSql(objs: Collection<Any>, table: String, columns: Collection<Column>): List<SqlAndParams> {
        return ToSqlByInsertObjects.toSql(objs, table, columns)
    }

    /**
     * 提供：表名
     *
     * 生成列信息（如需使用动态插入语句，例如 QueryProSql.create().insert(Map<String, Object>，则必须重写以实现该功能，可参考JdbcQSR.getColumnsDynamic的实现)
     *
     * 该方法可重写
     *
     * @sample JdbcQSR.getColumnsDynamic
     */
    open fun getColumnsDynamic(table: String): Collection<Column> {
        throw IllegalImplements("必须重写以实现动态获取列信息，才可调用动态插入语句。可参考JdbcQSR.getColumnsDynamic的实现。")
    }

    /**
     * 将QueryStructure解析至SQL并执行
     *
     * @param queryStructure [QueryStructure]
     * @param clazz QSR模版支持`JavaBean`, `Map`, 以及`Long`, `String`, `Boolean`等基本类型。另外，默认使用的JdbcQSR(为QSR的子类)还额外支持`IResultSetWalker`
     * @return clazz为基本类型时，可能返回List<T?> 当动作为insert时，返回值为List<ID>
     */
    override fun <T : Any> resolve(queryStructure: QueryStructure, payload: QueryPayload, clazz: Class<T>): Executor<List<T>> {
        val action = queryStructure.action
        if (action == QueryStructureAction.INSERT) {
            val objs = queryStructure.insert?.data ?: throw IllegalImplements("INSERT操作必须传qs.insert.data")

            val parsedClass = ClassParser.parse(clazz)
            val idColumn = parsedClass.idColumn
            val sqlForInsertList = insertObjectToSql(objs, parsedClass.dbName, parsedClass.columns.values.map { Column(it.dbName, it.getter, idColumn == it.dbName) })

            @Suppress("UNCHECKED_CAST") val idColumnType = ClassParser.parse(clazz).idColumnType as Class<T>?
            if (idColumnType == null) {
                logger.warn("没有找到主键或其对应的Class, 不会有返回结果")
            }

            return insert(payload.queryProClass, sqlForInsertList, idColumnType)
        }
        val (sql, params) = queryStructureToSql(queryStructure)
        return resolve(payload.queryProClass, sql, params, clazz, action)
    }

    /**
     * 执行一个SQL查询
     *
     * @param sql 单条sql语句 e.g. SELECT * FROM user WHERE user.id = ?
     * @param params 参数数组 e.g. [1]
     * @param clazz `JavaBean`, `Map`, and basic type e.g. `Long`, `String`, `Boolean` etc. for select by default; [Int] for update
     * @param action [QueryStructureAction]
     * @return clazz为基本类型时，可能返回List<T?>
     */
    override fun <T : Any> resolve(sql: String, params: Array<out Any?>, clazz: Class<T>, action: QueryStructureAction): Executor<List<T>> {
        return resolve(null, sql, params, clazz, action)
    }

    private fun <T : Any> resolve(queryPro: Class<*>?, sql: String, params: Array<out Any?>, clazz: Class<T>, action: QueryStructureAction): Executor<List<T>> {
        return when (action) {
            QueryStructureAction.SELECT -> Executor.create(sql, params) { withLog(it.sql, it.params) { doSelect(queryPro, it.sql, it.params, clazz) } }
            QueryStructureAction.INSERT -> Executor.create(sql, params) { withLog(it.sql, it.params) { doInsert(queryPro, it.sql, it.params, clazz) } }
            QueryStructureAction.DELETE, QueryStructureAction.UPDATE -> Executor.create(sql, params) { withLog(it.sql, it.params) { listOf(doUpdate(queryPro, it.sql, it.params, clazz)) } }
        }
    }

    /**
     * 使用多条语句和参数执行更新，创建，删除等非select语句
     *
     * @param sqlArr 多条或单条SQL语句
     * @param paramsArr sqlArr 的长度为1时，params的长度任意。代表同一语句包含多参数
     *                  sqlArr 的长度不为1时，params的长度必须和 sqlArr的长度相等。
     * @param clazz 指定返回结果的类型，支持 List<Integer> Integer[] Int Boolean(不准确)
     */
    override fun <T: Any> exec(sqlArr: Array<out String>, paramsArr: Array<out Array<out Any?>>, clazz: Class<T>): Executor<T> {
        val sqlArraySize = sqlArr.size
        val paramsSize = paramsArr.size
        if (sqlArraySize == 0) {
            throw IllegalParameters("sqlArr的长度不能为空")
        }
        if (sqlArraySize != 1 && sqlArraySize != paramsSize) {
            throw IllegalParameters("sqlArr的长度必须为1或者与paramsArr长度一致")
        }
        logger.debug("exec: sql-size: {0} params-size: {1}", sqlArraySize, paramsSize)
        val sqlAndParamsList = mutableListOf<SqlAndParams>()
        if (sqlArraySize == 1) {
            for (params in paramsArr) {
                val sql = sqlArr[0]
                val formattedParams = params.map { it ?: NULL }.toTypedArray()
                sqlAndParamsList.add(SqlAndParams(sql, formattedParams))
            }
        } else {
            for ((i, sql) in sqlArr.withIndex()) {
                val params = paramsArr.getOrNull(i)?.map { it ?: NULL }?.toTypedArray() ?: arrayOf()
                sqlAndParamsList.add(SqlAndParams(sql, params))
            }
        }

        return Executor.create(
            init = { IntArray(sqlAndParamsList.size) {0} },
            sqlAndParamsList,
            run = { sqlAndParams, affectRows, i ->
                val affectRowNum = withLog(sqlAndParams.sql, sqlAndParams.params) {
                    doUpdate(null, sqlAndParams.sql, sqlAndParams.params, Int::class.java)
                }
                affectRows[i] = affectRowNum
            },
            collect = { affectRows ->
                @Suppress("UNCHECKED_CAST")
                when {
                    List::class.java.isAssignableFrom(clazz) -> {
                        val affectRowList: List<Int> = affectRows.toList()
                        affectRowList as T
                    }
                    IntArray::class.java.isAssignableFrom(clazz) -> {
                        affectRows as T
                    }
                    else -> {
                        if (clazz.compatibleWithBool()) {
                            if (affectRows[0] == 0) {
                                false as T
                            } else {
                                (affectRows.sum() > 0) as T
                            }
                        } else if (clazz.compatibleWithInt()) {
                            affectRows.sum() as T
                        } else {
                            throw UnSupportException("不支持的class: $clazz, 目前只支持List::class.java, listOf<Int>().javaClass, Int, Boolean")
                        }
                    }
                }
            }
        )
    }


    /**
     * 动态插入
     * 通过 objs 配合数据库表名，动态生成sql语句，并执行
     *
     * @param objs 需要插入的Map类型的集合
     * @param table 数据库表名
     * @param idColumnClazz 如需返回ID，指定ID的类型
     */
    override fun <ID: Any, V: Any?, M: Map<String, V>> insert(objs: Collection<M>, table: String, idColumnClazz: Class<ID>?): Executor<List<ID>> {
        return insert(null, insertObjectToSql(objs, table, getColumnsDynamic(table)), idColumnClazz)
    }

    /**
     * 本实现额外引入了对大数据量数据批量插入的支持
     */
    private fun <ID: Any> insert(queryPro: Class<*>?, sqlAndParamsList: List<SqlAndParams>, idColumnClazz: Class<ID>?): Executor<List<ID>> {
        return Executor.create({ mutableListOf<ID>() }, sqlAndParamsList) { sqlAndParams, results, _ ->
            val sql = sqlAndParams.sql
            val params = sqlAndParams.params
            val idList = withLog(sql, params) {
                doInsert(queryPro, sql, params, idColumnClazz)
            }
            results.addAll(idList)
        }
    }

    /**
     * 添加日志输出的支持(这里可以换成lifecycle)
     */
    private fun <P: Any?, R: Any?> withLog(sql: String, params: Array<P>, query: () -> R): R {
        if (QueryProConfig.final.dryRun()) {
            logger.info("dry run mode, skip querying.")
            @Suppress("UNCHECKED_CAST")
            return listOf<Any?>() as R
        }

        return SqlLog.withQuery(logger, sql, params, query)
    }

    private val logger = LogFactory.getLog(QSR::class.java)
}