package cn.cloudself.query

import cn.cloudself.query.util.LogFactory
import java.sql.Connection
import javax.sql.DataSource
import kotlin.jvm.Throws

/**
 * QueryPro默认支持Spring管理的事务（例如@Transactional注解）
 * 也可用该类手动管理一个事务
 */
object QueryProTransaction {
    private val logger = LogFactory.getLog(QueryProTransaction::class.java)
    internal var isActualTransactionActive = ThreadLocal.withInitial { false }
    private val connectionThreadLocal = ThreadLocal<Connection?>()

    fun interface Block {
        @Throws(Exception::class)
        fun exec()
    }

    fun interface BlockR<R> {
        @Throws(Exception::class)
        fun exec(): R
    }

    @JvmStatic
    @JvmName("use")
    fun useJava(block: Block) {
        use { block.exec() }
    }

    @JvmStatic
    @JvmName("use")
    fun <R> useJava(block: BlockR<R>): R {
        return use { block.exec() }
    }

    @JvmName("_getConnection")
    internal fun getConnection(datasource: DataSource): Connection {
        val oldConnection = connectionThreadLocal.get()
        if (oldConnection != null) {
            return oldConnection
        }
        val connection = datasource.connection
        logger.debug("connection got.")
        connection.autoCommit = false
        connectionThreadLocal.set(connection)
        return connection
    }

    @JvmName("_useKt")
    fun <R> use(block: () -> R): R {
        isActualTransactionActive.set(true)
        try {
            logger.debug("connection managed by QueryProTransaction.")
            return block()
        } catch (e: Exception) {
            val connection = connectionThreadLocal.get()
            if (connection != null) {
                logger.warn("遇到错误，准备回滚中")
                connection.rollback()
                logger.info("回滚完毕")
            } else {
                logger.info("遇到错误，错误发生在执行sql之前，无需回滚")
            }
            throw e
        } finally {
            connectionThreadLocal.get().also {
                val connection = it ?: return@also
                connection.commit()
                connection.autoCommit = true
                logger.debug("transaction committed.")
                if (!connection.isClosed) {
                    connection.close()
                    logger.debug("connection closed.")
                }
            }
            connectionThreadLocal.set(null)
            isActualTransactionActive.set(false)
        }
    }
}