package cn.cloudself.query

import cn.cloudself.query.config.HashMapStore
import cn.cloudself.query.config.QueryProConfig
import cn.cloudself.query.config.QueryProConfigDb
import cn.cloudself.query.exception.IllegalCall
import cn.cloudself.query.exception.IllegalParameters
import cn.cloudself.query.resolver.Resolver
import cn.cloudself.query.util.SqlUtils
import org.intellij.lang.annotations.Language
import java.io.InputStream
import javax.sql.DataSource

/**
 * 用于：
 * 1. 直接执行`SQL`语句
 * 2. 动态生成插入语句并执行
 */
class QueryProSql {
    companion object {
        /**
         * @param sql language=SQL
         */
        @JvmStatic
        @JvmOverloads
        fun create(@Language("SQL") sql: String, vararg params: Any? = arrayOf()) = Action(sql, params)

        @JvmStatic
        fun create() = InsertAction()

        @JvmStatic
        @JvmOverloads
        fun create(inputStream: InputStream, vararg params: Any? = arrayOf()) = Action(String(inputStream.readBytes()), params)

        @JvmStatic
        @JvmOverloads
        fun createFromClassPath(path: String, vararg params: Any? = arrayOf()) =
            QueryProSql::class.java.classLoader.getResourceAsStream(path)?.let {
                Action(String(it.readBytes()), params)
            } ?: throw IllegalParameters("路径{0}可能不是标准的ClassPath", path)

        /**
         * 使用多条语句和参数执行更新，创建，删除等非select语句
         *
         * @param sqlAndParams Pair<SQL, Params>[]
         */
        @JvmStatic
        fun createBatch(sqlAndParams: Array<Pair<String, Array<Any?>>>): BatchAction {
            val size = sqlAndParams.size
            val emptyArr = arrayOf<Any?>()
            val sqlArr = Array(size) { "" }
            val paramsArr = Array(size) { emptyArr }
            var i = 0
            for ((sql, params) in sqlAndParams) {
                sqlArr[i] = sql
                paramsArr[i] = params
                i++
            }
            return BatchAction(sqlArr, paramsArr)
        }

        /**
         * 使用多条语句和参数执行更新，创建，删除等非select语句
         *
         * @param sqlArr 多条sql语句
         * @param params 参数数组数组，长度必须和sqlArr一支
         */
        @JvmStatic
        @JvmOverloads
        fun createBatch(sqlArr: Array<String>, params: Array<Array<Any?>>? = null): BatchAction {
            var nonNullParams = params
            if (nonNullParams == null) {
                nonNullParams = sqlArr.map { arrayOf<Any?>() }.toTypedArray()
            } else if (sqlArr.size != nonNullParams.size && sqlArr.size != 1) {
                throw IllegalParameters("sqlArr的长度必须和params的长度一致")
            }
            return BatchAction(sqlArr, nonNullParams)
        }

        /**
         * 使用单条语句和多组参数执行更新，创建，删除等非select语句
         *
         * @param sql 单条sql语句 e.g. INSERT INTO user (id, name) VALUES (?, ?)
         * @param params 参数数组数组 e.g. [[1, 'hb'], [2, 'l']]
         */
        @JvmStatic
        @JvmOverloads
        fun createBatch(@Language("SQL") sql: String, params: Array<Array<Any?>> = arrayOf()): BatchAction {
            return BatchAction(arrayOf(sql), params)
        }

        /**
         * 使用含;的复合语句组合执行更新操作,
         * 该语句会对传入的sql按照;进行拆分
         *
         * @param sqlGroup 复合sql语句 e.g. INSERT INTO user (id, name) VALUES (?, ?); UPDATE user SET name = UPPER(name) WHERE id = ?
         * @param params e.g. [1, 'hb, 1]
         */
        @JvmStatic
        @JvmOverloads
        fun createBatchBySqlGroup(@Language("SQL") sqlGroup: String, params: Array<Any?> = arrayOf()): BatchAction {
            val sqlAndCountArr = SqlUtils.splitBySemicolonAndCountQuestionMark(sqlGroup)
            val size = sqlAndCountArr.size
            val sqlArr = Array(size) { "" }
            val emptyArr = arrayOf<Any?>()
            val paramsArr = Array(size) { emptyArr }
            var i = 0
            var j = 0
            for ((sql, count) in sqlAndCountArr) {
                sqlArr[i] = sql
                paramsArr[i] = params.copyOfRange(j, count)
                i++
                j += count
            }
            return BatchAction(sqlArr, paramsArr)
        }
    }

    class BatchAction(
        private val sqlArr: Array<String>,
        private val params: Array<Array<Any?>>
    ) {
        private val store = HashMapStore()
        private val config = QueryProConfigDb(store)

        /**
         * 临时切换数据源
         */
        fun ofDataSource(dataSource: DataSource) = this.also {
            this.config.dataSource(dataSource)
        }

        /**
         * 批量更新
         *
         * @param clazz 支持的值有：
         * * Int::class.java 总共影响的条数
         * * IntArray::class.java, List::class.java, listOf<Int>().javaClass等 代表每条语句更新的条数
         * * Boolean::class.java 不建议使用，仅简单的判断了总更新条数是否大于1
         */
        fun <T: Any> exec(clazz: Class<T>): T {
            return Resolver.create { store }.use {
                exec(sqlArr, params, clazz)
            }
        }

        /**
         * 批量更新
         */
        fun exec(): Int {
            return exec(Int::class.java)
        }
    }

    class InsertAction {
        private val store = HashMapStore()
        private val config = QueryProConfigDb(store)

        /**
         * 临时切换数据源
         */
        fun ofDataSource(dataSource: DataSource) = this.also {
            this.config.dataSource(dataSource)
        }

        /**
         * 通过 objs 配合数据库表名，动态生成sql语句，并执行
         */
        fun <V: Any?> insert(table: String, vararg objs: Map<String, V>) {
            insert<Unit, V, Map<String, V>>(table, listOf(*objs), null)
        }

        /**
         * 通过 objs 配合数据库表名，动态生成sql语句，并执行
         */
        fun <V: Any?, M: Map<String, V>> insert(table: String, objs: Collection<M>) {
            insert<Unit, V, M>(table, objs, null)
        }

        /**
         * 通过 objs 配合数据库表名，动态生成sql语句，并执行
         *
         * @param idColumnClazz 指定需要返回的ID的类型
         */
        fun <ID: Any, V: Any?, M: Map<String, V>> insert(table: String, objs: Collection<M>, idColumnClazz: Class<ID>?): List<ID> {
            return Resolver.create { store }.use {
                insert(objs, table, idColumnClazz)
            }
        }
    }

    class Action(
        private val sql: String,
        private val params: Array<out Any?>,
    ) {
        private val store = HashMapStore()
        private val config = QueryProConfigDb(store)

        /**
         * 临时切换数据源
         */
        fun ofDataSource(dataSource: DataSource) = this.also {
            this.config.dataSource(dataSource)
        }

        /**
         * 查询单个对象
         *
         * @param clazz 支持JavaBean, 支持Map, 支持基本类型(Long, String, Date, Enum等, 具体参考[QueryProConfig.global.addResultSetParser])
         */
        fun <T: Any> queryOne(clazz: Class<T>) = query(clazz).getOrNull(0)

        /**
         * 查询单个对象
         */
        fun queryOne() = query().getOrNull(0)

        /**
         * 查询多个对象
         *
         * @param clazz 支持JavaBean, 支持Map, 支持基本类型(Long, String, Date, Enum等, 具体参考[QueryProConfig.global.addResultSetParser])
         */
        fun <T: Any> query(clazz: Class<T>): List<T> {
            return Resolver.create { store }.use {
                resolve(sql, params, clazz, QueryStructureAction.SELECT)
            }
        }

        /**
         * 查询多个对象
         */
        fun query(): List<Map<String, Any?>> {
            return query(HashMap::class.java).map { map ->
                @Suppress("UNCHECKED_CAST")
                map as Map<String, Any?>
            }
        }

        /**
         * 使用单条语句执行更新，创建，删除等非select语句
         */
        fun exec(): Int {
            return Resolver.create { store }.use {
                resolve(sql, params, Int::class.java, QueryStructureAction.UPDATE)
            }[0]
        }

        /**
         * 执行sql，
         * 根据 数据库连接配置，决定是否能同时执行多条sql
         */
        fun autoSplit(): BatchAction {
            val sqlAndCountList = SqlUtils.splitBySemicolonAndCountQuestionMark(sql)
            val sqlAndCountListSize = sqlAndCountList.size
            val sqlArray = arrayOfNulls<String>(sqlAndCountListSize)
            val paramsArray = arrayOfNulls<Array<out Any?>>(sqlAndCountListSize)

            val paramsSize = params.size
            var i = 0
            var paramsIndex = 0
            for ((sql, count) in sqlAndCountList) {
                val endIndex = paramsIndex + count
                if (endIndex > paramsSize) {
                    throw IllegalCall("参数个数可能不对, sql信息: {0}, 实际参数个数: {1}", sqlAndCountList, paramsSize)
                }

                sqlArray[i] = sql
                paramsArray[i] = params.copyOfRange(paramsIndex, endIndex)
                i++
                paramsIndex = endIndex
            }

            @Suppress("UNCHECKED_CAST")
            return BatchAction(sqlArray as Array<String>, paramsArray as Array<Array<Any?>>)
        }
    }
}
