package cn.cloudself.query

import cn.cloudself.query.exception.IllegalCall
import cn.cloudself.query.exception.IllegalParameters
import cn.cloudself.query.util.onlyAZaz_
import org.jetbrains.annotations.Contract

class QueryKeywords<F : QueryField<*, *, *, *, *, *>>(
    private val field: Field,
    private val queryStructure: QueryStructure,
    private val payload: QueryPayload,
    private val createQueryField: CreateQueryField<F>,
) {
    @get:Contract(pure = true)
    @get:JvmName("is")
    val `is` = this

    @get:Contract(pure = true)
    @get:JvmName("not")
    val not = QueryWithNotKeywords(field, queryStructure, payload, createQueryField)

    @get:Contract(pure = true)
    @get:JvmName("ignoreCase")
    val ignoreCase = QueryIgnoreCaseKeywords(field, queryStructure, payload, createQueryField)

    @Contract(pure = true)
    fun eq(value: Any) = with(WhereClause(field, "=", value))
    @Contract(pure = true)
    fun equalsTo(value: Any) = with(WhereClause(field, "=", value))
    @Contract(pure = true)
    fun between(start: Any, end: Any) = with(WhereClause(field, "between", arrayOf(start, end)))
    @Contract(pure = true)
    fun lessThan(value: Any) = with(WhereClause(field, "<", value))
    @Contract(pure = true)
    fun lessThanOrEqual(value: Any) = with(WhereClause(field, "<=", value))
    @Contract(pure = true)
    fun graterThan(value: Any) = with(WhereClause(field, ">", value))
    @Contract(pure = true)
    fun graterThanOrEqual(value: Any) = with(WhereClause(field, ">=", value))
    @Contract(pure = true)
    fun like(str: String) = with(WhereClause(field, "like", str))

    /**
     * in 查询 支持vararg, eg: in(1, 2, 3), in(intArray), etc.  <br/>
     * 支持有且仅有一个Collection作为参数, eg: in(list), in(set), etc.
     */
    @Contract(pure = true)
    fun `in`(vararg values: Any): F {
        val inArr = if (values.isNotEmpty() && values[0] is Collection<*>) {
            if (values.size > 1) {
                throw IllegalCall("in查询的第一个参数为collection时，不支持多个collection。")
            }
            (values[0] as Collection<*>).toTypedArray()
        } else {
            values
        }

        return if (inArr.isEmpty()) {
            val table = field.table
            with(WhereClause(null, operator = "", sql = "false/* WARN: ${if (table == null) "" else onlyAZaz_(table)}${if (table == null) "" else "."}${onlyAZaz_(field.column)} in empty*/"))
        } else {
            with(WhereClause(field, "in", inArr))
        }
    }
    @Contract(pure = true)
    fun nul() = with(WhereClause(field = field, operator = "is null"))
    @Contract(pure = true)
    fun isNull() = with(WhereClause(field = field, operator = "is null"))
    @Contract(pure = true)
    fun isNotNull() = with(WhereClause(field = field, operator = "is not null"))
    @Contract(pure = true)
    fun sql(sql: String) = with(WhereClause(field = field, operator = "", sql = sql))

    private fun with(whereClause: WhereClause): F {
        queryStructure.where = queryStructure.where + whereClause
        return createQueryField(queryStructure, payload)
    }
}

class QueryWithNotKeywords<F : QueryField<*, *, *, *, *, *>>(
    private val field: Field,
    private val queryStructure: QueryStructure,
    private val payload: QueryPayload,
    private val createQueryField: CreateQueryField<F>,
) {
    @Contract(pure = true)
    fun eq(value: Any) = with(WhereClause(field, "<>", value))
    @Contract(pure = true)
    fun equalsTo(value: Any) = with(WhereClause(field, "<>", value))
    @Contract(pure = true)
    fun between(start: Any, end: Any) = with(WhereClause(field, "not between", arrayOf(start, end)))
    @Contract(pure = true)
    fun like(str: String) = with(WhereClause(field, "not like", str))
    @Contract(pure = true)
    fun `in`(vararg values: Any): F {
        val inArr = if (values.isNotEmpty() && values[0] is Collection<*>) {
            if (values.size > 1) {
                throw IllegalCall("not in查询的第一个参数为collection时，不支持多个collection。")
            }
            (values[0] as Collection<*>).toTypedArray()
        } else {
            values
        }

        return if (inArr.isEmpty())
            throw IllegalParameters("not in查询不能设置空参数")
        else
            with(WhereClause(field, "not in", inArr))
    }
    @Contract(pure = true)
    fun nul() = with(WhereClause(field = field, operator = "is not null"))

    private fun with(whereClause: WhereClause): F {
        queryStructure.where = queryStructure.where + whereClause
        return createQueryField(queryStructure, payload)
    }
}

class QueryIgnoreCaseKeywords<F : QueryField<*, *, *, *, *, *>>(
    private val field: Field,
    private val queryStructure: QueryStructure,
    private val payload: QueryPayload,
    private val createQueryField: CreateQueryField<F>,
) {
    @Contract(pure = true)
    fun eq(value: Any) = with(WhereClause(upperField(field), "=", value, WhereClauseCommands.UPPER_CASE))
    @Contract(pure = true)
    fun equalsTo(value: Any) = with(WhereClause(upperField(field), "=", value, WhereClauseCommands.UPPER_CASE))
    @Contract(pure = true)
    fun like(str: String) = with(WhereClause(upperField(field), "like", str, WhereClauseCommands.UPPER_CASE))
    @Contract(pure = true)
    fun `in`(vararg values: Any) =
        if (values.isEmpty()) throw IllegalParameters("in查询+ignore case不能设置空参数")
        else with(WhereClause(upperField(field), "in", values, WhereClauseCommands.UPPER_CASE))

    private fun upperField(field: Field) = Field(table = field.table, column = field.column)
    private fun with(whereClause: WhereClause): F {
        queryStructure.where = queryStructure.where + whereClause
        return createQueryField(queryStructure, payload)
    }
}

class QueryOrderByKeywords<F: QueryField<*, *, *, *, *, *>>(
    private val field: Field,
    private val queryStructure: QueryStructure,
    private val payload: QueryPayload,
    private val createQueryField: CreateQueryField<F>,
) {
    @Contract(pure = true)
    fun asc() = with(OrderByClause(field, "asc"))
    @Contract(pure = true)
    fun desc() = with(OrderByClause(field, "desc"))

    private fun with(orderBy: OrderByClause): F {
        queryStructure.orderBy = queryStructure.orderBy + orderBy
        return createQueryField(queryStructure, payload)
    }
}
