package cn.cloudself.query

import cn.cloudself.query.exception.IllegalCall
import cn.cloudself.query.exception.IllegalImplements
import cn.cloudself.query.resolver.Resolver
import java.util.*

enum class QueryFieldType {
    WHERE_FIELD,
    ORDER_BY_FIELD,
    OTHER_FIELD,
}

typealias CreateQueryField<F> = (QueryStructure, QueryPayload) -> F

abstract class FinalSelectField<
        T : Any,
        RUN_RES,
        COLUMN_LIMITER_FILED: FinalSelectField<T, RUN_RES, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED>,
        COLUMNS_LIMITER_FILED: FinalSelectField<T, RUN_RES, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED>,
> {
    protected abstract val queryStructure: QueryStructure
    protected abstract val payload: QueryPayload
    protected abstract val clazz: Class<T>
    protected abstract fun createColumnLimitField(qs: QueryStructure, payload: QueryPayload): COLUMN_LIMITER_FILED
    protected abstract fun createColumnsLimitField(qs: QueryStructure, payload: QueryPayload): COLUMNS_LIMITER_FILED
    protected abstract fun createField(qs: QueryStructure, payload: QueryPayload): FinalSelectField<T, RUN_RES, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED>

    @Suppress("UNCHECKED_CAST")
    private fun <R: Any> resolve(clazz: Class<R>): List<R> {
        return Resolver.create(this.clazz as Class<Any>, queryStructure, payload).use {
            resolve(it, payload, clazz)
        }
    }

    fun distinct(): FinalSelectField<T, RUN_RES, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED> {
        if (queryStructure.orderBy.isNotEmpty()) {
            throw IllegalCall("不支持 distinct + orderBy. 可使用Java代码手动去重。")
        }
        queryStructure.distinct = true
        return createField(queryStructure, payload)
    }

    fun limit(limit: Int): FinalSelectField<T, RUN_RES, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED> {
        return limit(0, limit)
    }

    @Suppress("MemberVisibilityCanBePrivate")
    fun limit(start: Int, limit: Int): FinalSelectField<T, RUN_RES, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED> {
        queryStructure.limit = start to limit
        return createField(queryStructure, payload)
    }

    protected open fun <T: Any>getColumn(field: Field, clazz: Class<T>): List<T?> {
        queryStructure.fields = queryStructure.fields + field
        val rows = createField(queryStructure, payload).runAsMap()
        return rows.map {
            val f = it[field.column] ?: return@map null
            val fieldJavaClass = f.javaClass
            if (clazz.isAssignableFrom(f.javaClass)) {
                @Suppress("UNCHECKED_CAST")
                return@map f as T
            }
            val targetPrimitiveType = clazz.kotlin.javaPrimitiveType ?: return@map null
            val objPrimitiveType = fieldJavaClass.kotlin.javaPrimitiveType
            if (targetPrimitiveType == objPrimitiveType) {
                @Suppress("UNCHECKED_CAST")
                f as T
            } else {
                null
            }
        }
    }

    fun columnsLimiter(): COLUMNS_LIMITER_FILED {
        return createColumnsLimitField(queryStructure, payload)
    }

    fun columnLimiter(): COLUMN_LIMITER_FILED {
        return createColumnLimitField(queryStructure, payload)
    }

    @Suppress("MemberVisibilityCanBePrivate")
    fun count(): Int {
        if (queryStructure.action != QueryStructureAction.SELECT) {
            throw IllegalCall("非SELECT语句不能使用count方法")
        }
        queryStructure.fields = listOf(Field(column = "count(*)"))
        return resolve(Int::class.java)[0]
    }

    fun runLimit1Opt(): Optional<T> {
        return Optional.ofNullable(runLimit1())
    }

    fun runLimit1(): T? {
        queryStructure.limit = 0 to 1
        val results = createField(queryStructure, payload).queryAll()
        return if (results.isEmpty()) null else results[0]
    }

    fun run(): RUN_RES {
        @Suppress("UNCHECKED_CAST")
        return when (queryStructure.action) {
            QueryStructureAction.SELECT -> {
                queryAll() as RUN_RES
            }
            QueryStructureAction.DELETE, QueryStructureAction.UPDATE -> {
                val results = queryAll()
                if (results.isEmpty())
                    throw IllegalImplements("DELETE, UPDATE需返回长度为1的List<Boolean>")
                else
                    results[0] as RUN_RES
            }
            QueryStructureAction.INSERT -> {
                throw IllegalCall("run方法不支持INSERT")
            }
        }
    }

    fun runAsMap(): List<Map<String, Any?>> = resolve(mutableMapOf<String, Any>().javaClass)

    private fun queryAll(): List<T> = resolve(clazz)

    fun pageable(): Pageable<T> {
        return Pageable.create(
            { count() },
            { start, limit ->
                queryStructure.limit = start to limit
                resolve(clazz)
            }
        )
    }
}

abstract class QueryField<
        T: Any,
        RUN_RES,
        WHERE_FIELD: QueryField<T, RUN_RES, WHERE_FIELD, ORDER_BY_FIELD, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED>,
        ORDER_BY_FIELD: QueryField<T, RUN_RES, WHERE_FIELD, ORDER_BY_FIELD, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED>,
        COLUMN_LIMITER_FILED: QueryField<T, RUN_RES, WHERE_FIELD, ORDER_BY_FIELD, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED>,
        COLUMNS_LIMITER_FILED: QueryField<T, RUN_RES, WHERE_FIELD, ORDER_BY_FIELD, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED>,
>: FinalSelectField<T, RUN_RES, COLUMN_LIMITER_FILED, COLUMNS_LIMITER_FILED>() {
    protected abstract val fieldType: QueryFieldType
    protected abstract fun createWhereField(qs: QueryStructure, payload: QueryPayload): WHERE_FIELD
    protected abstract fun createOrderByField(qs: QueryStructure, payload: QueryPayload): ORDER_BY_FIELD
    override fun createField(qs: QueryStructure, payload: QueryPayload): WHERE_FIELD {
        return createWhereField(qs, payload)
    }

    fun customColumn(column: String) = QueryKeywords(Field(column = column), queryStructure, payload, this::createWhereField)

    fun and(): WHERE_FIELD {
        if (fieldType != QueryFieldType.WHERE_FIELD) {
            throw RuntimeException("$fieldType can not call and, usage: .orderBy().id.desc().name.asc()")
        }
        @Suppress("UNCHECKED_CAST")
        return this as WHERE_FIELD
    }

    @JvmOverloads
    fun or(factor: ((f: WHERE_FIELD) -> WHERE_FIELD)? = null): WHERE_FIELD {
        if (fieldType != QueryFieldType.WHERE_FIELD) {
            throw RuntimeException("$fieldType can not call and, usage: .orderBy().id.desc().name.asc()")
        }

        if (factor == null) {
            queryStructure.where = queryStructure.where + WhereClause(operator = OP_OR)
            return createWhereField(queryStructure, payload)
        }

        val vTempQueryStructure = QueryStructure(from = QueryStructureFrom("v_temp")) // v_temp会消失, 只取where
        val orWhereClauses = factor(createWhereField(vTempQueryStructure, payload)).queryStructure.where
        queryStructure.where = queryStructure.where + WhereClause(operator = OP_OR, value = orWhereClauses)
        return createWhereField(queryStructure, payload)
    }

    fun par(factor: ((f: WHERE_FIELD) -> WHERE_FIELD)): WHERE_FIELD {
        if (fieldType != QueryFieldType.WHERE_FIELD) {
            throw RuntimeException("$fieldType can not call and, usage: .orderBy().id.desc().name.asc()")
        }

        val vTempQueryStructure = QueryStructure(from = QueryStructureFrom("v_temp")) // v_temp会消失, 只取where
        val parWhereClauses = factor(createWhereField(vTempQueryStructure, payload)).queryStructure.where
        val newWhereClause = queryStructure.where + WhereClause(operator = "(") + parWhereClauses + WhereClause(operator = ")")
        queryStructure.where = newWhereClause
        return createWhereField(queryStructure, payload)
    }

    fun parLeft(): WHERE_FIELD {
        queryStructure.where = queryStructure.where + WhereClause(operator = "(")
        return createWhereField(queryStructure, payload)
    }

    fun parRight(): WHERE_FIELD {
        queryStructure.where = queryStructure.where + WhereClause(operator = ")")
        return createWhereField(queryStructure, payload)
    }

    fun sql(sql: String): WHERE_FIELD {
        queryStructure.where = queryStructure.where + WhereClause(operator = "", sql = sql)
        return createWhereField(queryStructure, payload)
    }

    fun orderBy(): ORDER_BY_FIELD {
        if (queryStructure.distinct) {
            throw IllegalCall("不支持 distinct + orderBy. 可使用Java代码手动去重。")
        }
        return createOrderByField(queryStructure, payload)
    }
}

open class UpdateField<UPDATE_BY_FIELD: QueryField<*, *, *, *, *, *>>(
    private val queryStructure: QueryStructure,
    private val payload: QueryPayload,
    private val createUpdateByField: CreateQueryField<UPDATE_BY_FIELD>,
) {
    @get:JvmName("where")
    val where = createUpdateByField(queryStructure.apply { action = QueryStructureAction.UPDATE }, payload)

    fun run(): Boolean = createUpdateByField(queryStructure, payload).run() as Boolean
}
