package cn.chiship.sdk.third.wechat.network;

import cn.chiship.sdk.core.base.constants.BaseConstants;
import cn.chiship.sdk.core.exception.ExceptionUtil;
import cn.chiship.sdk.core.exception.custom.BusinessException;
import cn.chiship.sdk.core.exception.custom.SystemErrorException;
import cn.chiship.sdk.core.util.*;
import cn.chiship.sdk.core.util.http.HttpUtils;
import cn.chiship.sdk.third.core.common.ThirdConstants;
import cn.chiship.sdk.third.core.common.WeiXinLanguageEnum;
import cn.chiship.sdk.third.wechat.core.common.WxPubMaterialTypeEnum;
import cn.chiship.sdk.third.wechat.core.common.WxPubCommonConstants;
import cn.chiship.sdk.third.wechat.core.common.WeChatCommonResult;
import cn.chiship.sdk.third.wechat.core.config.WeiXinConfig;

import cn.chiship.sdk.third.wechat.core.entity.pub.*;
import cn.chiship.sdk.third.wechat.core.entity.pub.message.MessageTemplateSend;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import javax.net.ssl.HttpsURLConnection;
import java.io.*;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * 微信服务工具类
 *
 * @author lijian
 */
public class WeiXinPubServicesUtil {
    private static final String TAG_ID = "tagid";
    private static final String OPEN_ID = "openid";
    private static final String APP_ID = "appid";
    private static final String ACCESS_TOKEN = "access_token";

    private WeiXinConfig weiXinConfig;
    private String accessToken = null;
    private static WeiXinPubServicesUtil instance;

    private WeiXinPubServicesUtil() {
    }

    public static WeiXinPubServicesUtil getInstance() {
        if (instance == null) {
            synchronized (WeiXinPubServicesUtil.class) {
                if (instance == null) {
                    instance = new WeiXinPubServicesUtil();
                }
            }
        }
        return instance;
    }

    public WeiXinPubServicesUtil config() {
        try {
            this.weiXinConfig = new WeiXinConfig(
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("WEI_XIN_PUB_APP_KEY"),
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("WEI_XIN_PUB_APP_SECRET")
            );
            if (StringUtil.isNullOrEmpty(weiXinConfig.getAppKey()) ||
                    StringUtil.isNullOrEmpty(weiXinConfig.getAppSecret())) {
                throw new SystemErrorException("兄弟,请确保微信公众号的各个属性[WEI_XIN_PUB_APP_KEY、WEI_XIN_PUB_APP_SECRET]配置存在或值不为空!");
            }
        } catch (MissingResourceException e) {
            throw new SystemErrorException(ThirdConstants.ERROR_EXIST_TIP_1);
        }
        return this;
    }

    public WeiXinPubServicesUtil config(WeiXinConfig weiXinConfig) {
        this.weiXinConfig = weiXinConfig;
        return this;
    }

    public WeiXinPubServicesUtil token() {
        WeChatCommonResult weChatCommonResult = getToken();
        if (!weChatCommonResult.isSuccess()) {
            throw new BusinessException(JSON.toJSONString(weChatCommonResult.getData()));
        }
        this.accessToken = weChatCommonResult.getData().toString();
        return this;
    }

    /**
     * 获得accessToken
     * 1.access_token是公众号的全局唯一接口调用凭据，公众号调用各接口时都需使用access_token。
     * 2.开发者需要进行妥善保存。
     * 3.access_token的存储至少要保留512个字符空间。
     * 4.access_token的有效期目前为2个小时，需定时刷新，重复获取将导致上次获取的access_token失效。
     *
     * @return 结果 返回token
     */
    public WeChatCommonResult getToken() {
        String key = ThirdConstants.REDIS_WEIXIN_PUB_ACCESS_TOKEN + ":" + this.weiXinConfig.getAppKey();
        String token = RedisUtil.get(key);
        if (!StringUtil.isNullOrEmpty(token)) {
            return WeChatCommonResult.ok(token);
        }
        Map<String, String> query = new HashMap<>(7);
        query.put("grant_type", "client_credential");
        query.put(APP_ID, this.weiXinConfig.getAppKey());
        query.put("secret", this.weiXinConfig.getAppSecret());
        try {
            String responseResult = HttpUtils.doGet(WxPubCommonConstants.API_WEI_XIN_SERVER_HOST, WxPubCommonConstants.GET_TOKEN, WxPubCommonConstants.commonHeaders(), query);
            WeChatCommonResult result = WxPubCommonConstants.analysisPubHttpResponse(responseResult);
            if (!result.isSuccess()) {
                return result;
            }
            JSONObject dataJson = (JSONObject) result.getData();
            String accessToken = dataJson.getString(ACCESS_TOKEN);
            String expiresIn = dataJson.getString("expires_in");
            result.setData(accessToken);
            RedisUtil.set(key, accessToken, Integer.valueOf(expiresIn));
            return result;
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }


    /**
     * 获取用户基本信息
     * 1.在关注者与公众号产生消息交互后，公众号可获得关注者的OpenID（加密后的微信号，每个用户对每个公众号的OpenID是唯一的。对于不同公众号，同一用户的openid不同）。
     * 2.公众号可通过本接口来根据OpenID获取用户基本信息，包括语言和关注时间。
     *
     * @param openId 用户标识
     * @return 结果
     */
    public WeChatCommonResult getUserInfo(String openId) {
        return getUserInfo(openId, WeiXinLanguageEnum.WX_ZH_CN);
    }

    /**
     * 获得用户基本信息
     *
     * @param openId             用户标识
     * @param weiXinLanguageEnum 语言
     * @return 结果
     */
    public WeChatCommonResult getUserInfo(String openId, WeiXinLanguageEnum weiXinLanguageEnum) {
        Map<String, String> query = new HashMap<>(7);
        query.put(ACCESS_TOKEN, getAccessToken());
        query.put(OPEN_ID, openId);
        query.put("lang", weiXinLanguageEnum.getLang());
        try {
            String responseResult = HttpUtils.doGet(WxPubCommonConstants.API_WEI_XIN_SERVER_HOST, WxPubCommonConstants.GET_USER_INFO, WxPubCommonConstants.commonHeaders(), query);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));

        }
    }

    /**
     * 设置用户备注名
     *
     * @param openId
     * @param remark 新的备注名，长度必须小于30字节，大于30个字节，将会自动截取,自动截取有可能会出现乱码
     * @return 结果
     */
    public WeChatCommonResult updateUserRemark(String openId, String remark) {
        byte[] bytes = remark.getBytes(StandardCharsets.UTF_8);
        if (bytes.length > 30) {
            bytes = Arrays.copyOfRange(bytes, 0, 30);
            remark = new String(bytes);
        }
        Map<String, String> query = new HashMap<>(7);
        query.put(ACCESS_TOKEN, getAccessToken());

        Map<String, String> body = new HashMap<>(7);
        body.put(OPEN_ID, openId);
        body.put("remark", remark);
        try {
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.UPDATE_USER_REMARK,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(body)
            );
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));

        }
    }


    /**
     * 第一个拉取的OPENID，不填默认从头开始拉取
     * 1.公众号可通过本接口来获取帐号的关注者列表，关注者列表由一串OpenID（加密后的微信号，每个用户对每个公众号的OpenID是唯一的）组成。
     * 2.一次拉取调用最多拉取10000个关注者的OpenID，可以通过多次拉取的方式来满足需求。
     *
     * @return 结果
     */
    public WeChatCommonResult syncUser() {
        return syncUser(null);
    }

    public WeChatCommonResult syncUser(String nextOpenId) {
        try {

            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            query.put("next_openid", nextOpenId);
            String responseResult = HttpUtils.doGet(WxPubCommonConstants.API_WEI_XIN_SERVER_HOST, WxPubCommonConstants.GET_USER_URL, WxPubCommonConstants.commonHeaders(), query);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 创建自定义菜单
     * 1.自定义菜单最多包括3个一级菜单，每个一级菜单最多包含5个二级菜单。
     * 2.一级菜单最多4个汉字，二级菜单最多8个汉字，多出来的部分将会以“...”代替。
     * 3.创建自定义菜单后，菜单的刷新策略是，在用户进入公众号会话页或公众号profile页时，如果发现上一次拉取菜单的请求在5分钟以前，就会拉取一下菜单，如果菜单有更新，就会刷新客户端的菜单。测试时可以尝试取消关注公众账号后再次关注，则可以看到创建后的效果。​
     *
     * @param menu 菜单实体
     * @return 结果
     */
    public WeChatCommonResult createMenu(Menu menu) {
        if (menu.getButton().size() > 3) {
            return WeChatCommonResult.error("一级按钮个数最多3个");
        }

        Map<String, String> query = new HashMap<>(7);
        query.put(ACCESS_TOKEN, getAccessToken());
        Map<String, Object> buttonMap = new HashMap<>(7);
        List<Map<String, Object>> firstButton = new ArrayList<>();
        for (Button button : menu.getButton()) {
            /**
             * 赋值
             */
            Map<String, Object> menuMap = new HashMap<>(2);
            menuMap.put("name", button.getName());
            menuMap.put("type", button.getType());
            if (WxPubCommonConstants.MENU_CLICK.equals(button.getType())) {
                menuMap.put("key", ((ClickButton) button).getKey());
            } else if (WxPubCommonConstants.MENU_VIEW.equals(button.getType())) {
                menuMap.put("url", ((ViewButton) button).getUrl());
            } else if (WxPubCommonConstants.MENU_MINIPROGRAM.equals(button.getType())) {
                menuMap.put("url", ((MiniprogramButton) button).getUrl());
                menuMap.put(APP_ID, ((MiniprogramButton) button).getAppid());
                menuMap.put("pagepath", ((MiniprogramButton) button).getPagepath());
            }

            if (button.getSub_button() == null) {
                firstButton.add(menuMap);
            } else {
                Map<String, Object> secondMap = new HashMap<>(7);
                secondMap.put("name", button.getName());
                List<Map<String, Object>> secondButton = new ArrayList<>();
                for (Button subButton : button.getSub_button()) {
                    Map<String, Object> second = new HashMap<>(7);
                    second.put("type", subButton.getType());
                    second.put("name", subButton.getName());
                    if (WxPubCommonConstants.MENU_CLICK.equals(subButton.getType())) {
                        second.put("key", ((ClickButton) subButton).getKey());
                        secondButton.add(second);
                    } else if (WxPubCommonConstants.MENU_VIEW.equals(subButton.getType())) {
                        second.put("url", ((ViewButton) subButton).getUrl());
                        secondButton.add(second);
                    } else if (WxPubCommonConstants.MENU_MINIPROGRAM.equals(subButton.getType())) {
                        second.put("url", ((MiniprogramButton) subButton).getUrl());
                        second.put(APP_ID, ((MiniprogramButton) subButton).getAppid());
                        second.put("pagepath", ((MiniprogramButton) subButton).getPagepath());
                        secondButton.add(second);
                    }
                }
                secondMap.put("sub_button", secondButton);
                firstButton.add(secondMap);
            }
            buttonMap.put("button", firstButton);
        }
        try {
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.CREATE_MENU_URL,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(buttonMap)
            );
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 获得微信标签
     *
     * @return 结果  Example {"code":200,"data":{"tags":[{"name":"星标组","count":0,"id":2}]},"success":true,"message":"操作成功"}
     */
    public WeChatCommonResult getTags() {
        try {

            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            String responseResult = HttpUtils.doGet(WxPubCommonConstants.API_WEI_XIN_SERVER_HOST, WxPubCommonConstants.GET_TAGS, WxPubCommonConstants.commonHeaders(), query);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 创建微信用户标签
     *
     * @param tagName 标签名称
     * @return 结果   Example {"code":200,"data":{"tag":{"name":"特别关注","id":100}},"success":true,"message":"操作成功"}
     */
    public WeChatCommonResult createTag(String tagName) {
        try {
            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, Map<String, String>> body = new HashMap<>(7);
            Map<String, String> tag = new HashMap<>(7);
            tag.put("name", tagName);
            body.put("tag", tag);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.CREATE_TAGS,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(body));
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 编辑用户标签
     *
     * @param tagId   标签ID
     * @param tagName 标签名称
     * @return 结果   Example {"code":200,"data":{"errcode":0,"errmsg":"ok"},"success":true,"message":"操作成功"}
     */
    public WeChatCommonResult updateTag(String tagId, String tagName) {
        try {

            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, Map<String, String>> body = new HashMap<>(7);
            Map<String, String> tag = new HashMap<>(7);
            tag.put("name", tagName);
            tag.put("id", tagId);
            body.put("tag", tag);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.UPDATE_TAGS,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(body));
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 删除用户标签
     *
     * @param tagId 标签ID
     * @return 结果   Example {"code":200,"data":{"errcode":0,"errmsg":"ok"},"success":true,"message":"操作成功"}
     */
    public WeChatCommonResult deleteTag(String tagId) {
        try {

            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, Map<String, String>> body = new HashMap<>(7);
            Map<String, String> tag = new HashMap<>(7);
            tag.put("id", tagId);
            body.put("tag", tag);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.DELETE_TAGS,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(body));
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 获得标签下用户
     *
     * @param tagId 标签ID
     * @return 结果   Example {"code":200,"data":{"data":{"openid":["ofZpz1tFF7Ek0C5ci4VJ398upvEo"]},"count":1,"next_openid":"ofZpz1tFF7Ek0C5ci4VJ398upvEo"},"success":true,"message":"操作成功"}
     */
    public WeChatCommonResult tagGetUser(String tagId) {
        return tagGetUser(tagId, null);
    }

    /**
     * 获得标签下用户
     *
     * @param tagId      标签ID
     * @param nextOpenId 下一个用户标识
     * @return 结果
     */
    public WeChatCommonResult tagGetUser(String tagId, String nextOpenId) {
        try {
            if (StringUtil.isNull(nextOpenId)) {
                nextOpenId = "";
            }

            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, String> body = new HashMap<>(7);
            body.put(TAG_ID, tagId);
            body.put("next_openid", nextOpenId);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.TAG_GET_USER,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(body));
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 用户绑定标签
     *
     * @param openId 用户标识
     * @param tagId  标签ID
     * @return 结果
     */
    public WeChatCommonResult memberBatchTag(String openId, String tagId) {
        List<String> openIds = new ArrayList<>();
        openIds.add(openId);
        return memberBatchTag(openIds, tagId);
    }

    /**
     * 用户绑定标签
     *
     * @param openIds 用户集合
     * @param tagId   标签ID
     * @return 结果
     */
    public WeChatCommonResult memberBatchTag(List<String> openIds, String tagId) {
        try {
            if (openIds.isEmpty()) {
                return WeChatCommonResult.error("用户集合不能为空!");
            }

            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, Object> body = new HashMap<>(7);
            body.put(TAG_ID, tagId);
            body.put("openid_list", openIds);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.MEMBER_BATCH_TAGS,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(body));
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 用户取消绑定标签
     *
     * @param openId 用户标识
     * @param tagId  标签ID
     * @return 结果
     */
    public WeChatCommonResult memberBatchUnTag(String openId, String tagId) {
        List<String> openIds = new ArrayList<>();
        openIds.add(openId);
        return memberBatchUnTag(openIds, tagId);
    }

    /**
     * 用户取消绑定标签
     *
     * @param openIds 用户集合
     * @param tagId   标签ID
     * @return 结果
     */
    public WeChatCommonResult memberBatchUnTag(List<String> openIds, String tagId) {
        try {
            if (openIds.isEmpty()) {
                return WeChatCommonResult.error("用户集合不能为空!");
            }

            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, Object> body = new HashMap<>(7);
            body.put(TAG_ID, tagId);
            body.put("openid_list", openIds);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.MEMBER_BATCH_UN_TAGS,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(body));
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 永久二维码
     *
     * @param sceneStr 场景 长度1-64
     * @return 结果
     */
    public WeChatCommonResult getQrCodeByForever(String sceneStr) {
        String params = "{\"action_name\": \"QR_LIMIT_STR_SCENE\", \"action_info\": {\"scene\": {\"scene_str\": \"" + sceneStr + "\"}}}";
        return getQrCode(params);
    }

    /**
     * 临时二维码
     *
     * @param expireSeconds 单位s 二维码有效时间 二维码生成后的30天（即2592000秒）后过期;若超过30天自动设置30天
     * @param sceneStr      场景 长度1-64
     * @return 结果
     */
    public WeChatCommonResult getQrCodeByTemporary(Integer expireSeconds, String sceneStr) {
        if (expireSeconds > 2592000) {
            expireSeconds = 2592000;
        }
        String params = "{\"expire_seconds\": " + expireSeconds + ", \"action_name\": \"QR_STR_SCENE\", \"action_info\": {\"scene\": {\"scene_str\": \"" + sceneStr + "\"}}}";
        return getQrCode(params);
    }

    /**
     * 获取二维码
     *
     * @param params
     * @return
     */
    private WeChatCommonResult getQrCode(String params) {
        Map<String, String> query = new HashMap<>(7);
        try {

            query.put(ACCESS_TOKEN, getAccessToken());
            String responseTicketResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.GET_QRCODE_TICKET,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    params);
            WeChatCommonResult ticketResult = WxPubCommonConstants.analysisPubHttpResponse(responseTicketResult);
            if (!ticketResult.isSuccess()) {
                ticketResult.setData("获取Ticket报错，" + ticketResult.getData());
                return ticketResult;
            }
            String ticket = JSON.parseObject(JSON.toJSONString(ticketResult.getData())).getString("ticket");
            String qrCodeUrl = WxPubCommonConstants.MP_WEI_XIN_SERVER_HOST + WxPubCommonConstants.CREATE_QRCODE_URL + "?ticket=" + ticket;
            return WeChatCommonResult.ok(ImageUtil.getImgByteByUrl(qrCodeUrl));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取模板列表
     *
     * @return 结果
     */
    public WeChatCommonResult getAllPrivateMessageTemplate() {
        Map<String, String> query = new HashMap<>(2);

        query.put(ACCESS_TOKEN, getAccessToken());
        try {
            String responseResult = HttpUtils.doGet(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.GET_ALL_PRIVATE_TEMPLATE,
                    WxPubCommonConstants.commonHeaders(),
                    query);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 删除消息模板
     *
     * @param templateId 模板ID
     * @return 结果
     */
    public WeChatCommonResult deleteTemplate(String templateId) {
        try {

            Map<String, String> query = new HashMap<>(7);
            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, String> body = new HashMap<>(7);
            body.put("template_id", templateId);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.DEL_PRIVATE_TEMPLATE,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    JSON.toJSONString(body));
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 模板消息发送
     *
     * @param messageTemplateSend 模板
     * @return 结果
     */
    public WeChatCommonResult messageTemplateSend(MessageTemplateSend messageTemplateSend) {
        Map<String, String> query = new HashMap<>(2);

        query.put(ACCESS_TOKEN, getAccessToken());
        try {
            String body = JSON.toJSONString(messageTemplateSend);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.MESSAGE_TEMPLATE_SEND,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    body);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }


    /**
     * 获得授权认证连接
     * 网页授权 第一步
     *
     * @param redirectUri 重定向的域名一定要在公众号内进行配置
     * @param params
     * @param isSilence   是否静默授权  true：snsapi_base，false：snsapi_userinfo
     * @return 结果
     */
    public String getConnectOauth2Url(String redirectUri, String params, Boolean isSilence) {
        try {
            String url = WxPubCommonConstants.AGREE_URL
                    .replace("APPID", weiXinConfig.getAppKey())
                    .replace("REDIRECT_URI", URLEncoder.encode(redirectUri, BaseConstants.UTF8))
                    .replace("SCOPE", Boolean.TRUE.equals(isSilence) ? "snsapi_base" : "snsapi_userinfo");
            if (params != null) {
                url = url.replace("STATE", params);
            }
            return url;
        } catch (Exception e) {
            throw new BusinessException("获得授权认证连接发生错误:" + e.getLocalizedMessage());
        }
    }

    /**
     * 通过code换取网页授权access_token
     * 网页授权    第二步
     *
     * @param code
     * @return 结果
     */
    public WeChatCommonResult oauth2AccessToken(String code) {

        try {
            String url = WxPubCommonConstants.EXCHANGE_URL
                    .replace("APPID", weiXinConfig.getAppKey())
                    .replace("SECRET", weiXinConfig.getAppSecret())
                    .replace("CODE", code);
            String responseResult = HttpUtils.doGet(url, "", WxPubCommonConstants.commonHeaders(), null);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 拉取用户信息(需scope为 snsapi_userinfo)
     * 获得授权的用户信息
     *
     * @param openId
     * @param token
     * @return 结果
     */
    public WeChatCommonResult getAccessUserInfo(String token, String openId) {

        Map<String, String> query = new HashMap<>(7);
        query.put(ACCESS_TOKEN, token);
        query.put(OPEN_ID, openId);
        query.put("lang", "zh_CN");
        try {
            String responseResult = HttpUtils.doGet(WxPubCommonConstants.API_WEI_XIN_SERVER_HOST, WxPubCommonConstants.GET_PERSON_URL, WxPubCommonConstants.commonHeaders(), query);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));

        }
    }

    /**
     * 新增其他类型永久素材
     *
     * @param file             文件
     * @param wxPubMaterialTypeEnum 类型
     * @return 结果
     */
    public WeChatCommonResult addMaterial(File file, WxPubMaterialTypeEnum wxPubMaterialTypeEnum) {
        return addMaterial(file, null, null, wxPubMaterialTypeEnum);
    }

    /**
     * 新增其他类型永久素材
     *
     * @param file             文件
     * @param title            标题
     * @param desc             描述
     * @param wxPubMaterialTypeEnum 类型
     * @return 结果   {"success":true, "code":200, "message":"操作成功", "data":{"item":[],"media_id":"","url":""}}
     */
    public WeChatCommonResult addMaterial(File file, String title, String desc, WxPubMaterialTypeEnum wxPubMaterialTypeEnum) {
        if (!file.exists() || !file.isFile()) {
            throw new BusinessException("文件不存在");
        }

        try (FileInputStream fileInputStream = new FileInputStream(file)) {
            return uploadMedia(fileInputStream, file.getName(), title, desc, wxPubMaterialTypeEnum.getType());
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 上传文件流到素材
     *
     * @param inputStream  文件流
     * @param fileName     文件名称
     * @param title        标题
     * @param introduction 描述
     * @param fileType     素材类型  图片（image）、语音（voice）、视频（video）
     * @return 结果
     * @throws Exception 异常
     */
    public WeChatCommonResult uploadMedia(InputStream inputStream, String fileName, String title, String introduction, String fileType) throws Exception {
        String accessToken = getAccessToken();
        String urlString = WxPubCommonConstants.API_WEI_XIN_SERVER_HOST + WxPubCommonConstants.ADD_MATERIAL + "?access_token=" + accessToken + "&type=" + fileType;

        String result = null;
        URL url = new URL(urlString);
        HttpsURLConnection conn = (HttpsURLConnection) url.openConnection();
        conn.setRequestMethod("POST");
        conn.setDoInput(true);
        conn.setDoOutput(true);
        conn.setUseCaches(false);
        //设置请求头信息
        conn.setRequestProperty("Connection", "Keep-Alive");
        conn.setRequestProperty("Charset", BaseConstants.UTF8);
        //设置边界
        String BOUNDARY = "----------" + System.currentTimeMillis();
        conn.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + BOUNDARY);
        //请求正文信息
        //第一部分
        StringBuilder sb = new StringBuilder();
        sb.append("--");
        sb.append(BOUNDARY);
        sb.append("\r\n");
        sb.append("Content-Disposition: form-data;name=\"media\"; filename=\"" + fileName + "\"\r\n");
        sb.append("Content-Disposition: form-data;name=\"media\"; filename=\"" + fileName + "\"\r\n");
        sb.append("Content-Type:application/octet-stream\r\n\r\n");


        //获得输出流
        OutputStream out = new DataOutputStream(conn.getOutputStream());
        if (WxPubMaterialTypeEnum.MATERIAL_TYPE_VIDEO.getType().equals(fileType)) {
            out.write(("--" + BOUNDARY + "\r\n").getBytes());
            out.write("Content-Disposition: form-data; name=\"description\";\r\n\r\n".getBytes());
            out.write(String.format("{\"title\":\"%s\", \"introduction\":\"%s\"}", title, introduction).getBytes());
            out.write(("\r\n--" + BOUNDARY + "--\r\n\r\n").getBytes());
        }
        //输出表头
        out.write(sb.toString().getBytes(StandardCharsets.UTF_8));
        //文件正文部分
        //把文件以流的方式 推送道URL中
        DataInputStream din = new DataInputStream(inputStream);
        int bytes = 0;
        byte[] buffer = new byte[1024];
        while ((bytes = din.read(buffer)) != -1) {
            out.write(buffer, 0, bytes);
        }
        din.close();
        //结尾部分
        byte[] foot = ("\r\n--" + BOUNDARY + "--\r\n").getBytes(StandardCharsets.UTF_8);
        out.write(foot);
        out.flush();
        out.close();
        if (HttpsURLConnection.HTTP_OK == conn.getResponseCode()) {
            StringBuilder builder;
            BufferedReader reader = null;
            try {
                builder = new StringBuilder();
                reader = new BufferedReader(new InputStreamReader(conn.getInputStream()));
                String lineString;
                while ((lineString = reader.readLine()) != null) {
                    builder.append(lineString);
                }
                if (StringUtil.isNull(result)) {
                    return WxPubCommonConstants.analysisPubHttpResponse(builder.toString());
                } else {
                    return WeChatCommonResult.error("上传素材出现错误");

                }
            } catch (IOException e) {
                e.printStackTrace();
                return WeChatCommonResult.error("发送POST请求出现异常！" + e);
            } finally {
                if (reader != null) {
                    reader.close();
                }
            }
        }
        return WeChatCommonResult.error("上传素材出现错误");
    }

    /**
     * 获取永久素材
     *
     * @param mediaId 媒体ID
     * @return 结果
     */
    public WeChatCommonResult getMaterial(String mediaId) {
        Map<String, String> query = new HashMap<>(7);
        try {

            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, String> body = new HashMap<>(7);
            body.put("media_id", mediaId);
            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.GET_MATERIAL,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    body);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 素材列表
     *
     * @param pageIndex        第几页
     * @param wxPubMaterialTypeEnum 类型
     * @return 结果
     */
    public WeChatCommonResult getBatchMaterial(Integer pageIndex, WxPubMaterialTypeEnum wxPubMaterialTypeEnum) {
        Map<String, String> query = new HashMap<>(7);
        try {

            query.put(ACCESS_TOKEN, getAccessToken());
            Map<String, String> body = new HashMap<>(7);
            body.put("type", wxPubMaterialTypeEnum.getType());
            body.put("offset", String.valueOf((pageIndex - 1) * 20));
            body.put("count", String.valueOf(20));

            String responseResult = HttpUtils.doPost(
                    WxPubCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WxPubCommonConstants.GET_BATCH_MATERIAL,
                    WxPubCommonConstants.commonHeaders(),
                    query,
                    body);
            return WxPubCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    public String getAccessToken() {
        if (StringUtil.isNullOrEmpty(accessToken)) {
            throw new BusinessException("token为空！请链式调用如下方法：WeiXinPubServicesUtil.getInstance().config().token()获得Token");
        }
        return accessToken;
    }

    public static void main(String[] args) throws Exception {
        WeiXinPubServicesUtil weiXinPubServicesUtil = WeiXinPubServicesUtil.getInstance().config().token();
        //System.out.println(weiXinPubServicesUtil.getUserInfo("ofZpz1uzddtuWyu9qYGDLuoMQTHk"));
        /*WeChatCommonResult chatCommonResult = weiXinPubServicesUtil.addMaterial(
                new File("C:\\Users\\lj\\Desktop\\f619bf4365e8d74da57c34210ba44840.mp4"),
                MaterialTypeEnum.MATERIAL_TYPE_VIDEO);*/
        // WeChatCommonResult chatCommonResult = weiXinPubServicesUtil.getMaterial("moewLZD7R3cQUhexRJL33pVbIIbYBekr77C-BGaItVtHXViRxjJNIFuQNjJBfLvy");
        //System.out.println(chatCommonResult);
        WeChatCommonResult chatCommonResult = weiXinPubServicesUtil.uploadMedia(new FileInputStream(new File("C:\\Users\\lijian\\Desktop\\f619bf4365e8d74da57c34210ba44840.mp4")), "f619bf4365e8d74da57c34210ba44840.mp4", "313213", "243423", "video");
        System.out.println(chatCommonResult);

    }


}
