package cn.chiship.sdk.third.baidu;

import cn.chiship.sdk.core.exception.ExceptionUtil;
import cn.chiship.sdk.core.exception.custom.BusinessException;
import cn.chiship.sdk.core.exception.custom.SystemErrorException;
import cn.chiship.sdk.core.util.ImageUtil;
import cn.chiship.sdk.core.util.PropertiesFileUtil;
import cn.chiship.sdk.core.util.RedisUtil;
import cn.chiship.sdk.core.util.StringUtil;
import cn.chiship.sdk.core.util.http.HttpUtils;
import cn.chiship.sdk.third.baidu.singleton.AipOcrClientSingleton;
import cn.chiship.sdk.third.core.common.ThirdConstants;
import cn.chiship.sdk.third.core.common.ThirdResult;
import cn.chiship.sdk.third.core.common.ThirdResultEnum;
import cn.chiship.sdk.third.core.model.BaiDuOcrConfigModel;


import com.alibaba.fastjson.JSON;
import com.baidu.aip.ocr.AipOcr;

import org.json.JSONObject;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.MissingResourceException;

/**
 * @author lijian
 */
public class BaiDuOcrUtils {
    public static final String IMAGE_JPEG_BASE64 = "data:image/jpeg;base64,";
    public static final String IMAGE_PNG_BASE64 = "data:image/png;base64,";
    public static final String STRING_URL = "url";
    public static final String STRING_IMAGE = "image";
    private static final String ACCESS_TOKEN = "access_token";

    private BaiDuOcrConfigModel baiDuOcrConfigModel;
    private static BaiDuOcrUtils instance;
    private String accessToken = null;

    private BaiDuOcrUtils() {
    }

    public static BaiDuOcrUtils getInstance() {
        if (instance == null) {
            synchronized (BaiDuOcrUtils.class) {
                if (instance == null) {
                    instance = new BaiDuOcrUtils();
                }
            }
        }
        return instance;
    }

    public BaiDuOcrUtils config() {
        try {
            this.baiDuOcrConfigModel = new BaiDuOcrConfigModel(
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("BAI_DU_AI_API_KEY"),
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("BAI_DU_AI_SECRET_KEY"),
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("BAI_DU_AI_APP_ID")
            );
            if (StringUtil.isNullOrEmpty(baiDuOcrConfigModel.getAppKey()) ||
                    StringUtil.isNullOrEmpty(baiDuOcrConfigModel.getAppSecret()) ||
                    StringUtil.isNullOrEmpty(baiDuOcrConfigModel.getAppId())) {
                throw new SystemErrorException("兄弟,请确保百度AI的各个属性配置存在或值不为空!");
            }
        } catch (MissingResourceException e) {
            throw new SystemErrorException(ThirdConstants.ERROR_EXIST_TIP_1);
        }
        return this;
    }

    public BaiDuOcrUtils config(BaiDuOcrConfigModel baiDuOcrConfigModel) {
        this.baiDuOcrConfigModel = baiDuOcrConfigModel;
        return this;
    }

    public BaiDuOcrUtils token() {
        ThirdResult thirdResult = getToken();
        if (!thirdResult.isSuccess()) {
            throw new BusinessException(JSON.toJSONString(thirdResult.getData()));
        }
        this.accessToken = thirdResult.getData().toString();
        return this;
    }


    /**
     * 获得accessToken
     *
     * @return 结果 返回token
     */
    public ThirdResult getToken() {
        String key = ThirdConstants.REDIS_BAI_DU_AIP_ACCESS_TOKEN + ":" + this.baiDuOcrConfigModel.getAppKey();
        String token = RedisUtil.get(key);
        if (!StringUtil.isNullOrEmpty(token)) {
            return ThirdResult.ok(token);
        }
        Map<String, String> query = new HashMap<>(7);
        query.put("grant_type", "client_credentials");
        query.put("client_id", this.baiDuOcrConfigModel.getAppKey());
        query.put("client_secret", this.baiDuOcrConfigModel.getAppSecret());
        try {
            String responseResult = HttpUtils.doPost(BaiDuAiConstant.AIP_SERVER_HOST, BaiDuAiConstant.GET_TOKEN, BaiDuAiConstant.commonHeaders(), query);
            ThirdResult result = BaiDuAiConstant.analysisAipHttpResponse(responseResult);
            if (!result.isSuccess()) {
                return result;
            }
            JSONObject dataJson = (JSONObject) result.getData();
            String accessToken = dataJson.getString("access_token");
            int expiresIn = dataJson.getInt("expires_in");
            result.setData(accessToken);
            RedisUtil.set(key, accessToken, expiresIn);
            return result;
        } catch (Exception e) {
            return ThirdResult.error(ExceptionUtil.formatException(e));
        }
    }


    /**
     * 身份证识别
     *
     * @param image   图片路径  base64图片或网络图片
     * @param isFront 身份证类型,true:正面,false:反面
     * @return 结果
     */
    public ThirdResult idCardOcr(String image, Boolean isFront, Boolean isNetwork) {
        String idCardSide;
        if (Boolean.TRUE.equals(isFront)) {
            idCardSide = "front";
        } else {
            idCardSide = "back";
        }

        HashMap<String, String> query = new HashMap<>(2);
        query.put(ACCESS_TOKEN, getAccessToken());
        HashMap<String, String> body = new HashMap<>(2);
        if (isNetwork) {
            body.put(STRING_URL, image);
        } else {
            body.put(STRING_IMAGE, replaceImage(image));
        }
        body.put("id_card_side", idCardSide);
        body.put("detect_direction", "true");
        body.put("detect_risk", "false");
        String responseResult = HttpUtils.doPost(BaiDuAiConstant.AIP_SERVER_HOST, BaiDuAiConstant.ID_CARD, BaiDuAiConstant.commonHeaders(), query, body);
        ThirdResult result = BaiDuAiConstant.analysisAipHttpResponse(responseResult);
        return BaiDuAiResultAnalyze.analyzeIdCard(result);
    }

    /**
     * 银行卡识别
     *
     * @param image     图片路径  base64图片或网络图片
     * @param isNetwork 是否网络图片
     * @return 结果
     */
    public ThirdResult bankCard(String image, Boolean isNetwork) {

        HashMap<String, String> query = new HashMap<>(2);
        query.put(ACCESS_TOKEN, getAccessToken());
        HashMap<String, String> body = new HashMap<>(2);
        if (isNetwork) {
            body.put(STRING_URL, image);
        } else {
            body.put(STRING_IMAGE, replaceImage(image));
        }
        String responseResult = HttpUtils.doPost(BaiDuAiConstant.AIP_SERVER_HOST, BaiDuAiConstant.BANK_CARD, BaiDuAiConstant.commonHeaders(), query, body);
        ThirdResult result = BaiDuAiConstant.analysisAipHttpResponse(responseResult);
        return BaiDuAiResultAnalyze.analyzeBankCard(result);


    }


    /**
     * 营业执照
     *
     * @param image     图片路径  base64图片或网络图片
     * @param isNetwork 是否网络图片
     * @return 结果
     */
    public ThirdResult businessLicense(String image, Boolean isNetwork) {
        HashMap<String, String> query = new HashMap<>(2);
        query.put(ACCESS_TOKEN, getAccessToken());
        HashMap<String, String> body = new HashMap<>(2);
        if (isNetwork) {
            body.put(STRING_URL, image);
        } else {
            body.put(STRING_IMAGE, replaceImage(image));
        }
        String responseResult = HttpUtils.doPost(BaiDuAiConstant.AIP_SERVER_HOST, BaiDuAiConstant.BUSINESS_LICENSE, BaiDuAiConstant.commonHeaders(), query, body);
        ThirdResult result = BaiDuAiConstant.analysisAipHttpResponse(responseResult);
        return BaiDuAiResultAnalyze.analyzeBusinessLicense(result);

    }

    /**
     * 护照
     *
     * @param image     图片路径  base64图片或网络图片
     * @param isNetwork 是否网络图片
     * @return 结果
     */
    public ThirdResult passport(String image, Boolean isNetwork) {
        HashMap<String, String> query = new HashMap<>(2);
        query.put(ACCESS_TOKEN, getAccessToken());
        HashMap<String, String> body = new HashMap<>(2);
        if (isNetwork) {
            body.put(STRING_URL, image);
        } else {
            body.put(STRING_IMAGE, replaceImage(image));
        }
        String responseResult = HttpUtils.doPost(BaiDuAiConstant.AIP_SERVER_HOST, BaiDuAiConstant.PASS_PORT, BaiDuAiConstant.commonHeaders(), query, body);
        ThirdResult result = BaiDuAiConstant.analysisAipHttpResponse(responseResult);
        return BaiDuAiResultAnalyze.analyzePassport(result);

    }

    /**
     * 社保卡
     *
     * @param image     图片路径  base64图片或网络图片
     * @param isNetwork 是否网络图片
     * @return 结果
     */
    public ThirdResult socialSecurityCard(String image, Boolean isNetwork) {
        HashMap<String, String> query = new HashMap<>(2);
        query.put(ACCESS_TOKEN, getAccessToken());
        HashMap<String, String> body = new HashMap<>(2);
        if (isNetwork) {
            body.put(STRING_URL, image);
        } else {
            body.put(STRING_IMAGE, replaceImage(image));
        }
        String responseResult = HttpUtils.doPost(BaiDuAiConstant.AIP_SERVER_HOST, BaiDuAiConstant.SOCIAL_SECURITY_CARD, BaiDuAiConstant.commonHeaders(), query, body);
        ThirdResult result = BaiDuAiConstant.analysisAipHttpResponse(responseResult);
        return BaiDuAiResultAnalyze.analyzeSocialSecurityCard((result));

    }

    /**
     * 车牌识别
     *
     * @param image     图片路径  base64图片或网络图片
     * @param isNetwork 是否网络图片
     * @return 结果
     */
    public ThirdResult licensePlate(String image, Boolean isNetwork) {
        HashMap<String, String> query = new HashMap<>(2);
        query.put(ACCESS_TOKEN, getAccessToken());
        HashMap<String, String> body = new HashMap<>(2);
        if (isNetwork) {
            body.put(STRING_URL, image);
        } else {
            body.put(STRING_IMAGE, replaceImage(image));
        }
        String responseResult = HttpUtils.doPost(BaiDuAiConstant.AIP_SERVER_HOST, BaiDuAiConstant.LICENSE_PLATE, BaiDuAiConstant.commonHeaders(), query, body);
        ThirdResult result = BaiDuAiConstant.analysisAipHttpResponse(responseResult);
        return BaiDuAiResultAnalyze.analyzeLicensePlate((result));

    }



    private String replaceImage(String image) {
        return image.replace(IMAGE_PNG_BASE64, "").replace(IMAGE_JPEG_BASE64, "");
    }

    public String getAccessToken() {
        return accessToken;
    }

    public static void main(String[] args) throws IOException {
        BaiDuOcrUtils baiDuOcrUtils = BaiDuOcrUtils.getInstance().config().token();
        //ThirdResult result = baiDuOcrUtils.passport(ImageUtil.getImgStrByFile("C:\\Users\\lj\\Desktop\\5_170330160228_1.jpg"),  false);
        //System.out.println(JSON.toJSONString(result));
        ThirdResult result = baiDuOcrUtils.licensePlate("http://n.sinaimg.cn/sinacn11/480/w640h640/20180720/8dd7-hfqtahh4815353.jpg", true);
        //ThirdResult result = baiDuOcrUtils.licensePlate(ImageUtil.getImgStrByFile("C:\\Users\\lj\\Desktop\\test\\bankCard.jpg"), false);
        //ThirdResult result = baiDuOcrUtils.idCardOcr("http://chiship.oss-cn-shanghai.aliyuncs.com/test/ocr/idCardBack.jpg", false, true);
        //ThirdResult result = baiDuOcrUtils.socialSecurityCard(ImageUtil.getImgStrByFile("C:\\Users\\lj\\Desktop\\test\\passport.jpg"), false);
        System.out.println(JSON.toJSONString(result));
    }


}





