package cn.chiship.sdk.third.dingtalk;


import cn.chiship.sdk.core.exception.ExceptionUtil;
import cn.chiship.sdk.core.exception.custom.SystemErrorException;
import cn.chiship.sdk.core.util.PropertiesFileUtil;
import cn.chiship.sdk.core.util.StringUtil;
import cn.chiship.sdk.core.util.http.HttpUtils;

import cn.chiship.sdk.third.core.common.ThirdConstants;
import cn.chiship.sdk.third.core.common.ThirdResult;
import cn.chiship.sdk.third.core.common.ThirdResultEnum;
import cn.chiship.sdk.third.core.model.BaseConfigModel;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.MissingResourceException;

/**
 * 钉钉服务工具类
 *
 * @author lijian
 */
public class DingTalkServicesUtils {
    private BaseConfigModel baseConfigModel;
    private static DingTalkServicesUtils instance;

    private DingTalkServicesUtils() {
    }

    public static DingTalkServicesUtils getInstance() {
        if (instance == null) {
            synchronized (DingTalkServicesUtils.class) {
                if (instance == null) {
                    instance = new DingTalkServicesUtils();
                }
            }
        }
        return instance;
    }

    public DingTalkServicesUtils config() {
        try {
            this.baseConfigModel = new BaseConfigModel(
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("DING_TALK_APP_KEY"),
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("DING_TALK_APP_SECRET")
            );
            if (StringUtil.isNullOrEmpty(baseConfigModel.getAppKey()) ||
                    StringUtil.isNullOrEmpty(baseConfigModel.getAppSecret())) {
                throw new SystemErrorException("兄弟,请确保钉钉的各个属性配置存在或值不为空!");
            }
        } catch (MissingResourceException e) {
            throw new SystemErrorException(ThirdConstants.ERROR_EXIST_TIP_1);
        }
        return this;
    }

    public DingTalkServicesUtils config(BaseConfigModel baseConfigModel) {
        this.baseConfigModel = baseConfigModel;
        return this;
    }

    /**
     * 获得accessToken
     *
     * @return 结果 结果
     */
    public ThirdResult getToken() {
        Map<String, String> query = new HashMap<>(2);
        query.put("appkey", this.baseConfigModel.getAppKey());
        query.put("appsecret", this.baseConfigModel.getAppSecret());
        try {
            String responseResult = HttpUtils.doGet(DingTalkConstant.DING_TALK_SERVER_HOST, DingTalkConstant.GET_TOKEN, DingTalkConstant.commonHeaders(), query);
            ThirdResult thirdResult = DingTalkConstant.analysisHttpResponse(responseResult);
            if (thirdResult.isSuccess()) {
                JSONObject jsonObject = (JSONObject) thirdResult.getData();
                return ThirdResult.ok(jsonObject.getString("access_token"));
            } else {
                return thirdResult;
            }
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_DING_TALK, ExceptionUtil.formatExceptionVo(e));
        }
    }


    /**
     * 获取部门列表
     *
     * @return 结果 结果
     */
    public ThirdResult getDepartmentList() {
        ThirdResult tokenResult = getToken();
        if (!tokenResult.isSuccess()) {
            return tokenResult;
        }
        String accessToken = tokenResult.getData().toString();
        Map<String, String> query = new HashMap<>(2);
        query.put("access_token", accessToken);
        query.put("fetch_child", String.valueOf(Boolean.TRUE));
        query.put("id", String.valueOf(1));
        try {
            String responseResult = HttpUtils.doGet(DingTalkConstant.DING_TALK_SERVER_HOST, DingTalkConstant.GET_DEPARTMENT_LIST, DingTalkConstant.commonHeaders(), query);
            return DingTalkConstant.analysisHttpResponse(responseResult);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_DING_TALK, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 获取部门用户
     *
     * @param departmentId 部门ID
     * @return 结果 结果
     */
    public ThirdResult getUserList(Long departmentId) {
        ThirdResult tokenResult = getToken();
        if (!tokenResult.isSuccess()) {
            return tokenResult;
        }
        String accessToken = tokenResult.getData().toString();
        Map<String, String> query = new HashMap<>(2);
        query.put("access_token", accessToken);
        query.put("department_id", String.valueOf(departmentId));
        try {
            String responseResult = HttpUtils.doGet(DingTalkConstant.DING_TALK_SERVER_HOST, DingTalkConstant.GET_USER_LIST, DingTalkConstant.commonHeaders(), query);
            return DingTalkConstant.analysisHttpResponse(responseResult);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_DING_TALK, ExceptionUtil.formatExceptionVo(e));

        }
    }

    /**
     * 获取部门用户详情
     *
     * @param departmentId 部门ID
     * @param page         当前页码
     * @return 结果 结果
     */
    public ThirdResult getUserListByPage(Long departmentId, Integer page) {
        ThirdResult tokenResult = getToken();
        if (!tokenResult.isSuccess()) {
            return tokenResult;
        }
        String accessToken = tokenResult.getData().toString();
        Map<String, String> query = new HashMap<>(2);
        query.put("access_token", accessToken);
        query.put("department_id", String.valueOf(departmentId));
        query.put("offset", String.valueOf((page - 1) * 50));
        query.put("size", String.valueOf(50));
        try {
            String responseResult = HttpUtils.doGet(DingTalkConstant.DING_TALK_SERVER_HOST, DingTalkConstant.GET_USER_LIST_BY_PAGE, DingTalkConstant.commonHeaders(), query);
            return DingTalkConstant.analysisHttpResponse(responseResult);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_DING_TALK, ExceptionUtil.formatExceptionVo(e));

        }
    }

    /**
     * 获取角色列表
     *
     * @return 结果 结果
     */
    public ThirdResult getRoleList() {
        ThirdResult tokenResult = getToken();
        if (!tokenResult.isSuccess()) {
            return tokenResult;
        }
        String accessToken = tokenResult.getData().toString();
        Map<String, String> query = new HashMap<>(2);
        query.put("access_token", accessToken);
        query.put("offset", String.valueOf(0));
        query.put("size", String.valueOf(50));
        try {
            String responseResult = HttpUtils.doGet(DingTalkConstant.DING_TALK_SERVER_HOST, DingTalkConstant.GET_ROLE_LIST, DingTalkConstant.commonHeaders(), query);
            return DingTalkConstant.analysisHttpResponse(responseResult);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_DING_TALK, ExceptionUtil.formatExceptionVo(e));

        }
    }

    /**
     * 获取角色下的员工列表
     *
     * @param roleId 角色
     * @return 结果 结果
     */
    public ThirdResult getRoleUserList(Long roleId) {
        ThirdResult tokenResult = getToken();
        if (!tokenResult.isSuccess()) {
            return tokenResult;
        }
        String accessToken = tokenResult.getData().toString();
        Map<String, String> query = new HashMap<>(2);
        query.put("access_token", accessToken);
        query.put("role_id", String.valueOf(roleId));
        query.put("offset", String.valueOf(0));
        query.put("size", String.valueOf(50));
        try {
            String responseResult = HttpUtils.doGet(DingTalkConstant.DING_TALK_SERVER_HOST, DingTalkConstant.GET_ROLE_USER_LIST, DingTalkConstant.commonHeaders(), query);
            return DingTalkConstant.analysisHttpResponse(responseResult);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_DING_TALK, ExceptionUtil.formatExceptionVo(e));

        }
    }

    /**
     * 获得日志
     *
     * @param templateName
     * @param day
     * @param userId
     * @return 结果 结果
     */
    public ThirdResult getReportList(String templateName, Integer day, String userId) {
        ThirdResult tokenResult = getToken();
        if (!tokenResult.isSuccess()) {
            return tokenResult;
        }
        String accessToken = tokenResult.getData().toString();
        Map<String, String> query = new HashMap<>(2);
        if (null == templateName) {
            templateName = "日报";
        }
        if (null == day) {
            day = 1;
        }
        query.put("access_token", accessToken);
        query.put("start_time", String.valueOf(System.currentTimeMillis() - day * 24 * 60 * 60 * 1000));
        query.put("end_time", String.valueOf(System.currentTimeMillis()));
        query.put("template_name", templateName);
        if (null != userId) {
            query.put("userid", userId);
        }
        query.put("cursor", String.valueOf(0));
        query.put("size", String.valueOf(50));
        try {
            String responseResult = HttpUtils.doGet(DingTalkConstant.DING_TALK_SERVER_HOST, DingTalkConstant.GET_REPORT_LIST, DingTalkConstant.commonHeaders(), query);
            return DingTalkConstant.analysisHttpResponse(responseResult);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_DING_TALK, ExceptionUtil.formatExceptionVo(e));
        }
    }


}
