package cn.chiship.sdk.third.baidu;

import cn.chiship.sdk.core.exception.ExceptionUtil;
import cn.chiship.sdk.core.exception.custom.SystemErrorException;
import cn.chiship.sdk.core.util.ImageUtil;
import cn.chiship.sdk.core.util.PropertiesFileUtil;
import cn.chiship.sdk.core.util.StringUtil;
import cn.chiship.sdk.third.baidu.singleton.AipOcrClientSingleton;
import cn.chiship.sdk.third.core.common.ThirdConstants;
import cn.chiship.sdk.third.core.common.ThirdResult;
import cn.chiship.sdk.third.core.common.ThirdResultEnum;
import cn.chiship.sdk.third.core.model.BaiDuOcrConfigModel;
import com.baidu.aip.ocr.AipOcr;

import org.json.JSONObject;

import java.util.Base64;
import java.util.HashMap;
import java.util.MissingResourceException;

/**
 * @author lijian
 */
public class BaiDuOcrUtils {
    private BaiDuOcrConfigModel baiDuOcrConfigModel;
    private static BaiDuOcrUtils instance;

    private BaiDuOcrUtils() {
    }

    public static BaiDuOcrUtils getInstance() {
        if (instance == null) {
            synchronized (BaiDuOcrUtils.class) {
                if (instance == null) {
                    instance = new BaiDuOcrUtils();
                }
            }
        }
        return instance;
    }

    public BaiDuOcrUtils config() {
        try {
            this.baiDuOcrConfigModel = new BaiDuOcrConfigModel(
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("BAI_DU_AI_API_KEY"),
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("BAI_DU_AI_SECRET_KEY"),
                    PropertiesFileUtil.getInstance(ThirdConstants.PROPERTIES_FILE_NAME).get("BAI_DU_AI_APP_ID")
            );
            if (StringUtil.isNullOrEmpty(baiDuOcrConfigModel.getAppKey()) ||
                    StringUtil.isNullOrEmpty(baiDuOcrConfigModel.getAppSecret()) ||
                    StringUtil.isNullOrEmpty(baiDuOcrConfigModel.getAppId())) {
                throw new SystemErrorException("兄弟,请确保百度AI的各个属性配置存在或值不为空!");
            }
        } catch (MissingResourceException e) {
            throw new SystemErrorException(ThirdConstants.ERROR_EXIST_TIP_1);
        }
        return this;
    }

    public BaiDuOcrUtils config(BaiDuOcrConfigModel baiDuOcrConfigModel) {
        this.baiDuOcrConfigModel = baiDuOcrConfigModel;
        return this;
    }


    /**
     * 身份证识别
     *
     * @param sources 图片路径
     * @param isFront 身份证类型,true:正面,false:反面
     * @return 结果 结果
     */
    public ThirdResult idCardOcr(String sources, Boolean isFront, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("detect_direction", "true");
        options.put("detect_risk", "false");
        String idCardSide;

        if (Boolean.TRUE.equals(isFront)) {
            idCardSide = "front";
        } else {
            idCardSide = "back";
        }
        try {
            JSONObject res = null;
            if (Boolean.TRUE.equals(isNetwork)) {
                sources = ImageUtil.getImgBase64ByUrl(sources);
            } else {
                sources = ImageUtil.getImgStrByFile(sources);
            }
            Base64.Decoder decoder = Base64.getDecoder();
            res = client.idcard(decoder.decode(sources), idCardSide, options);
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 身份证识别
     *
     * @param image   图片路径
     * @param isFront 身份证类型,true:正面,false:反面
     * @return 结果 结果
     */
    public ThirdResult idCardOcr(byte[] image, Boolean isFront) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("detect_direction", "true");
        options.put("detect_risk", "false");
        String idCardSide;

        if (Boolean.TRUE.equals(isFront)) {
            idCardSide = "front";
        } else {
            idCardSide = "back";
        }
        try {
            JSONObject res = client.idcard(image, idCardSide, options);
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 文字识别
     *
     * @param image     图片路径
     * @param isNetwork 是否为网络图片
     * @param type      图片类型
     * @return 结果 结果
     */
    public ThirdResult characterRecognition(String image, Boolean isNetwork, String type) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                image = ImageUtil.getImgBase64ByUrl(image);
            } else {
                image = ImageUtil.getImgStrByFile(image);
            }
            Base64.Decoder decoder = Base64.getDecoder();
            JSONObject res = null;

            if ("word".equals(type)) {
                options.put("language_type", "CHN_ENG");
                options.put("detect_direction", "true");
                options.put("detect_language", "true");
                options.put("probability", "true");
                res = client.basicGeneral(decoder.decode(image), options);
            }
            if ("highPrecisionWord".equals(type)) {
                options.put("detect_direction", "true");
                options.put("probability", "true");
                res = client.basicAccurateGeneral(decoder.decode(image), options);
            }
            if ("locationWord".equals(type)) {
                options.put("recognize_granularity", "big");
                options.put("language_type", "CHN_ENG");
                options.put("detect_direction", "true");
                options.put("detect_language", "true");
                options.put("vertexes_location", "true");
                options.put("probability", "true");
                res = client.general(decoder.decode(image), options);
            }
            if ("highLocationWord".equals(type)) {
                options.put("recognize_granularity", "big");
                options.put("detect_direction", "true");
                options.put("vertexes_location", "true");
                options.put("probability", "true");
                res = client.accurateGeneral(decoder.decode(image), options);
            }
            if ("rareWord".equals(type)) {
                options.put("language_type", "CHN_ENG");
                options.put("detect_direction", "true");
                options.put("detect_language", "true");
                options.put("probability", "true");
                res = client.enhancedGeneral(decoder.decode(image), options);
            }
            if ("webImage".equals(type)) {
                options.put("detect_direction", "true");
                options.put("detect_language", "true");
                res = client.webImage(decoder.decode(image), options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 银行卡识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult bankCard(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("detect_direction", "true");
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.bankcard(file, options);
            } else {
                res = client.bankcard(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 驾驶证识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult drivingLicense(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.drivingLicense(file, options);
            } else {
                res = client.drivingLicense(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 行驶证识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult vehicleLicense(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("detect_direction", "true");
        options.put("accuracy", "normal");
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.vehicleLicense(file, options);
            } else {
                res = client.vehicleLicense(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 车牌号识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult plateLicense(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("multi_detect", "true");
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.plateLicense(file, options);
            } else {
                res = client.plateLicense(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 车牌号识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult businessLicense(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.businessLicense(file, options);
            } else {
                res = client.businessLicense(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 通行票据
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult receipt(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("recognize_granularity", "big");
        options.put("probability", "true");
        options.put("accuracy", "normal");
        options.put("detect_direction", "true");
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.receipt(file, options);
            } else {
                res = client.receipt(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 火车票识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult trainTicket(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.trainTicket(file, options);
            } else {
                res = client.trainTicket(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 出租车识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult taxiReceipt(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.taxiReceipt(file, options);
            } else {
                res = client.taxiReceipt(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 不能用  表格同步获取数据
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult form(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.form(file, options);
            } else {
                res = client.form(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }


    /**
     * 第一个方法获取requestId 第二个方法通过id获取结果
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult tableRecognitionAsync(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.tableRecognitionAsync(file, options);
            } else {
                res = client.tableRecognitionAsync(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    public ThirdResult tableResultGet(String requestId) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("result_type", "json");
        JSONObject res;
        try {
            res = client.tableResultGet(requestId, options);
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 增值税发票识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult vatInvoice(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.vatInvoice(file, options);
            } else {
                res = client.vatInvoice(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 二维码
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult qrcode(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.qrcode(file, options);
            } else {
                res = client.qrcode(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 数字识别
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult numbers(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("recognize_granularity", "big");
        options.put("detect_direction", "true");
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.numbers(file, options);
            } else {
                res = client.numbers(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 彩票 无权限
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult lottery(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("recognize_granularity", "big");
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.lottery(file, options);
            } else {
                res = client.lottery(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 护照
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult passport(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.passport(file, options);
            } else {
                res = client.passport(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 名片 不可用
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult businessCard(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.businessCard(file, options);
            } else {
                res = client.businessCard(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    /**
     * 手写 不准确
     *
     * @param image
     * @param isNetwork
     * @return 结果 结果
     */
    public ThirdResult handwriting(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        HashMap<String, String> options = new HashMap<>(2);
        options.put("recognize_granularity", "big");
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.handwriting(file, options);
            } else {
                res = client.handwriting(image, options);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }

    public ThirdResult tableRecognizeToJson(String image, Boolean isNetwork) {
        AipOcr client = AipOcrClientSingleton.getInstance(this.baiDuOcrConfigModel).getAipOcr();
        JSONObject res;
        try {
            if (Boolean.TRUE.equals(isNetwork)) {
                byte[] file = ImageUtil.getImgByteByUrl(image);
                res = client.tableRecognizeToJson(file, 20000);
            } else {
                res = client.tableRecognizeToJson(image, 20000);
            }
            return BaiDuAiConstant.analysisJson(res);
        } catch (Exception e) {
            return ThirdResult.error(ThirdResultEnum.ERROR_BAIDU_AI, ExceptionUtil.formatExceptionVo(e));
        }
    }


}





