package cn.chiship.sdk.third.wechat.network;

import cn.chiship.sdk.core.exception.ExceptionUtil;
import cn.chiship.sdk.core.util.PropertiesFileUtil;
import cn.chiship.sdk.core.util.RedisUtil;
import cn.chiship.sdk.core.util.StringUtil;
import cn.chiship.sdk.core.util.http.HttpUtils;
import cn.chiship.sdk.third.core.common.ThirdConstants;
import cn.chiship.sdk.third.core.common.WeiXinLanguageEnum;
import cn.chiship.sdk.third.wechat.core.common.WeChatCommonConstants;
import cn.chiship.sdk.third.wechat.core.common.WeChatCommonResult;
import cn.chiship.sdk.third.wechat.core.config.WeiXinConfig;

import cn.chiship.sdk.third.wechat.core.entity.pub.*;
import cn.chiship.sdk.third.wechat.core.entity.pub.message.MessageTemplateSend;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * 微信服务工具类
 *
 * @author lijian
 */
public class WeiXinPubServicesUtil {
    private WeiXinConfig weiXinConfig;
    private static WeiXinPubServicesUtil INSTANCE;

    private WeiXinPubServicesUtil() {
    }

    public static WeiXinPubServicesUtil getInstance() {
        if (INSTANCE == null) {
            synchronized (WeiXinPubServicesUtil.class) {
                if (INSTANCE == null) {
                    INSTANCE = new WeiXinPubServicesUtil();
                }
            }
        }
        return INSTANCE;
    }

    public WeiXinPubServicesUtil config() {
        try {
            WeiXinConfig baseConfigModel = new WeiXinConfig(
                    PropertiesFileUtil.getInstance("chiship-third").get("WEI_XIN_PUB_APP_KEY"),
                    PropertiesFileUtil.getInstance("chiship-third").get("WEI_XIN_PUB_APP_SECRET")
            );
            if (StringUtil.isNullOrEmpty(baseConfigModel.getAppKey()) ||
                    StringUtil.isNullOrEmpty(baseConfigModel.getAppSecret())) {
                throw new RuntimeException("兄弟,请确保微信公众号的各个属性[WEI_XIN_PUB_APP_KEY、WEI_XIN_PUB_APP_SECRET]配置存在或值不为空!");
            }
            config(baseConfigModel);
        } catch (MissingResourceException e) {
            throw new RuntimeException(ThirdConstants.ERROR_EXIST_TIP_1);
        }
        return this;
    }

    public WeiXinPubServicesUtil config(WeiXinConfig weiXinConfig) {
        this.weiXinConfig = weiXinConfig;
        return this;
    }

    /**
     * 获得accessToken
     * 1.access_token是公众号的全局唯一接口调用凭据，公众号调用各接口时都需使用access_token。
     * 2.开发者需要进行妥善保存。
     * 3.access_token的存储至少要保留512个字符空间。
     * 4.access_token的有效期目前为2个小时，需定时刷新，重复获取将导致上次获取的access_token失效。
     *
     * @return 返回token
     */
    public WeChatCommonResult getToken() {
        String key = ThirdConstants.REDIS_WEIXIN_PUB_ACCESS_TOKEN + ":" + this.weiXinConfig.getAppKey();
        String token = RedisUtil.get(key);
        if (!StringUtil.isNullOrEmpty(token)) {
            System.out.println("Redis获取......");
            return WeChatCommonResult.ok(token);
        }
        Map<String, String> query = new HashMap<>(7);
        query.put("grant_type", "client_credential");
        query.put("appid", this.weiXinConfig.getAppKey());
        query.put("secret", this.weiXinConfig.getAppSecret());
        try {
            String responseResult = HttpUtils.doGet(WeChatCommonConstants.API_WEI_XIN_SERVER_HOST, WeChatCommonConstants.GET_TOKEN, WeChatCommonConstants.commonHeaders(), query);
            WeChatCommonResult result = WeChatCommonConstants.analysisPubHttpResponse(responseResult);
            if (!result.isSuccess()) {
                return result;
            }
            JSONObject dataJson = (JSONObject) result.getData();
            String accessToken = dataJson.getString("access_token");
            String expiresIn = dataJson.getString("expires_in");
            result.setData(accessToken);
            RedisUtil.set(key, accessToken, Integer.valueOf(expiresIn));
            System.out.println("接口获取......");
            return result;
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }


    /**
     * 获取用户基本信息
     * 1.在关注者与公众号产生消息交互后，公众号可获得关注者的OpenID（加密后的微信号，每个用户对每个公众号的OpenID是唯一的。对于不同公众号，同一用户的openid不同）。
     * 2.公众号可通过本接口来根据OpenID获取用户基本信息，包括语言和关注时间。
     *
     * @param openId
     * @return
     */
    public WeChatCommonResult getUserInfo(String openId) {
        return getUserInfo(openId, WeiXinLanguageEnum.WX_ZH_CN);
    }

    public WeChatCommonResult getUserInfo(String openId, WeiXinLanguageEnum weiXinLanguageEnum) {
        WeChatCommonResult weChatCommonResult = getToken();
        if (!weChatCommonResult.isSuccess()) {
            return weChatCommonResult;
        }
        String token = weChatCommonResult.getData().toString();
        Map<String, String> query = new HashMap<>(7);
        query.put("access_token", token);
        query.put("openid", openId);
        query.put("lang", weiXinLanguageEnum.getLang());
        try {
            String responseResult = HttpUtils.doGet(WeChatCommonConstants.API_WEI_XIN_SERVER_HOST, WeChatCommonConstants.GET_USER_INFO, WeChatCommonConstants.commonHeaders(), query);
            return WeChatCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));

        }
    }

    /**
     * 设置用户备注名
     *
     * @param openId
     * @param remark 新的备注名，长度必须小于30字节，大于30个字节，将会自动截取,自动截取有可能会出现乱码
     * @return
     */
    public WeChatCommonResult updateUserRemark(String openId, String remark) {
        byte[] bytes = remark.getBytes(StandardCharsets.UTF_8);
        if (bytes.length > 30) {
            bytes = Arrays.copyOfRange(bytes, 0, 30);
            remark = new String(bytes);
        }
        WeChatCommonResult weChatCommonResult = getToken();
        if (!weChatCommonResult.isSuccess()) {
            return weChatCommonResult;
        }
        String token = weChatCommonResult.getData().toString();
        Map<String, String> query = new HashMap<>(7);
        query.put("access_token", token);

        Map<String, String> body = new HashMap<>(7);
        body.put("openid", openId);
        body.put("remark", remark);
        try {
            String responseResult = HttpUtils.doPost(
                    WeChatCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WeChatCommonConstants.UPDATE_USER_REMARK,
                    WeChatCommonConstants.commonHeaders(),
                    query,
                    JSONObject.toJSONString(body)
            );
            return WeChatCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));

        }
    }


    /**
     * 第一个拉取的OPENID，不填默认从头开始拉取
     * 1.公众号可通过本接口来获取帐号的关注者列表，关注者列表由一串OpenID（加密后的微信号，每个用户对每个公众号的OpenID是唯一的）组成。
     * 2.一次拉取调用最多拉取10000个关注者的OpenID，可以通过多次拉取的方式来满足需求。
     *
     * @return
     */
    public WeChatCommonResult syncUser() {
        return syncUser(null);
    }

    public WeChatCommonResult syncUser(String nextOpenId) {
        try {
            WeChatCommonResult weChatCommonResult = getToken();
            if (!weChatCommonResult.isSuccess()) {
                return weChatCommonResult;
            }
            String token = weChatCommonResult.getData().toString();
            Map<String, String> query = new HashMap<>(7);
            query.put("access_token", token);
            query.put("next_openid", nextOpenId);
            String responseResult = HttpUtils.doGet(WeChatCommonConstants.API_WEI_XIN_SERVER_HOST, WeChatCommonConstants.GET_USER_URL, WeChatCommonConstants.commonHeaders(), query);
            WeChatCommonResult result = WeChatCommonConstants.analysisPubHttpResponse(responseResult);
            return result;
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 创建自定义菜单
     * 1.自定义菜单最多包括3个一级菜单，每个一级菜单最多包含5个二级菜单。
     * 2.一级菜单最多4个汉字，二级菜单最多8个汉字，多出来的部分将会以“...”代替。
     * 3.创建自定义菜单后，菜单的刷新策略是，在用户进入公众号会话页或公众号profile页时，如果发现上一次拉取菜单的请求在5分钟以前，就会拉取一下菜单，如果菜单有更新，就会刷新客户端的菜单。测试时可以尝试取消关注公众账号后再次关注，则可以看到创建后的效果。​
     *
     * @param menu
     * @return
     */
    public WeChatCommonResult createMenu(Menu menu) {
        if (menu.getButton().size() > 3) {
            return WeChatCommonResult.error("一级按钮个数最多3个");
        }
        WeChatCommonResult weChatCommonResult = getToken();
        if (!weChatCommonResult.isSuccess()) {
            return weChatCommonResult;
        }
        String token = weChatCommonResult.getData().toString();
        Map<String, String> query = new HashMap<>(7);
        query.put("access_token", token);
        Map<String, Object> buttonMap = new HashMap<>(7);
        List firstButton = new ArrayList<>();
        for (Button button : menu.getButton()) {
            /**
             * 赋值
             */
            Map<String, Object> menuMap = new HashMap<>();
            menuMap.put("name", button.getName());
            menuMap.put("type", button.getType());
            if (WeChatCommonConstants.MENU_CLICK.equals(button.getType())) {
                menuMap.put("key", ((ClickButton) button).getKey());
            } else if (WeChatCommonConstants.MENU_VIEW.equals(button.getType())) {
                menuMap.put("url", ((ViewButton) button).getUrl());
            } else if (WeChatCommonConstants.MENU_MINIPROGRAM.equals(button.getType())) {
                menuMap.put("url", ((MiniprogramButton) button).getUrl());
                menuMap.put("appid", ((MiniprogramButton) button).getAppid());
                menuMap.put("pagepath", ((MiniprogramButton) button).getPagepath());
            }

            if (button.getSub_button() == null) {
                firstButton.add(menuMap);
            } else {
                Map<String, Object> secondMap = new HashMap<>(7);
                secondMap.put("name", button.getName());
                List secondButton = new ArrayList<>();
                for (Button SubButton : button.getSub_button()) {
                    Map<String, Object> second = new HashMap<>(7);
                    second.put("type", SubButton.getType());
                    second.put("name", SubButton.getName());
                    if (WeChatCommonConstants.MENU_CLICK.equals(SubButton.getType())) {
                        second.put("key", ((ClickButton) SubButton).getKey());
                        secondButton.add(second);
                    } else if (WeChatCommonConstants.MENU_VIEW.equals(SubButton.getType())) {
                        second.put("url", ((ViewButton) SubButton).getUrl());
                        secondButton.add(second);
                    } else if (WeChatCommonConstants.MENU_MINIPROGRAM.equals(SubButton.getType())) {
                        second.put("url", ((MiniprogramButton) SubButton).getUrl());
                        second.put("appid", ((MiniprogramButton) SubButton).getAppid());
                        second.put("pagepath", ((MiniprogramButton) SubButton).getPagepath());
                        secondButton.add(second);
                    }
                }
                secondMap.put("sub_button", secondButton);
                firstButton.add(secondMap);
            }
            buttonMap.put("button", firstButton);
        }
        System.out.println(JSON.toJSON(buttonMap));
        try {
            String responseResult = HttpUtils.doPost(
                    WeChatCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WeChatCommonConstants.CREATE_MENU_URL,
                    WeChatCommonConstants.commonHeaders(),
                    query,
                    JSONObject.toJSONString(buttonMap)
            );
            WeChatCommonResult result = WeChatCommonConstants.analysisPubHttpResponse(responseResult);
            return result;
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 永久二维码
     *
     * @param sceneStr 场景 长度1-64
     * @return
     */
    public WeChatCommonResult getQRCodeByForever(String sceneStr) {
        String params = "{\"action_name\": \"QR_LIMIT_STR_SCENE\", \"action_info\": {\"scene\": {\"scene_str\": \"" + sceneStr + "\"}}}";
        return getQRCode(params);
    }

    /**
     * 临时二维码
     *
     * @param expireSeconds 单位s 二维码有效时间 二维码生成后的30天（即2592000秒）后过期;若超过30天自动设置30天
     * @param sceneStr      场景 长度1-64
     * @return
     */
    public WeChatCommonResult getQRCodeByTemporary(Integer expireSeconds, String sceneStr) {
        if (expireSeconds > 2592000) {
            expireSeconds = 2592000;
        }
        String params = "{\"expire_seconds\": " + expireSeconds + ", \"action_name\": \"QR_STR_SCENE\", \"action_info\": {\"scene\": {\"scene_str\": \"" + sceneStr + "\"}}}";
        return getQRCode(params);
    }

    private WeChatCommonResult getQRCode(String params) {
        Map<String, String> query = new HashMap<>(7);
        try {
            WeChatCommonResult weChatCommonResult = getToken();
            if (!weChatCommonResult.isSuccess()) {
                return weChatCommonResult;
            }
            String token = weChatCommonResult.getData().toString();
            query.put("access_token", token);
            String responseTicketResult = HttpUtils.doPost(
                    WeChatCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WeChatCommonConstants.GET_QRCODE_TICKET,
                    WeChatCommonConstants.commonHeaders(),
                    query,
                    params);
            WeChatCommonResult ticketResult = WeChatCommonConstants.analysisPubHttpResponse(responseTicketResult);
            if (!ticketResult.isSuccess()) {
                ticketResult.setData("获取Ticket报错，" + ticketResult.getData());
                return ticketResult;
            }
            String ticket = JSONObject.parseObject(JSONObject.toJSONString(ticketResult.getData())).getString("ticket");
            query.put("ticket", URLEncoder.encode(ticket));
            String qrCode = HttpUtils.doGet(WeChatCommonConstants.MP_WEI_XIN_SERVER_HOST,
                    WeChatCommonConstants.CREATE_QRCODE_URL,
                    WeChatCommonConstants.commonHeaders(), query);
            return WeChatCommonResult.ok(qrCode);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 模板消息发送
     *
     * @return
     */
    public WeChatCommonResult messageTemplateSend(MessageTemplateSend messageTemplateSend) {
        Map<String, String> query = new HashMap<>(2);
        WeChatCommonResult chatCommonResult = getToken();
        if (!chatCommonResult.isSuccess()) {
            return chatCommonResult;
        }
        String accessToken = chatCommonResult.getData().toString();
        query.put("access_token", accessToken);
        try {
            String body = JSONObject.toJSONString(messageTemplateSend);
            String responseResult = HttpUtils.doPost(
                    WeChatCommonConstants.API_WEI_XIN_SERVER_HOST,
                    WeChatCommonConstants.MESSAGE_TEMPLATE_SEND,
                    WeChatCommonConstants.commonHeaders(),
                    query,
                    body);
            return WeChatCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }


    /**
     * 获得授权认证连接
     * 网页授权 第一步
     *
     * @param redirectUri 重定向的域名一定要在公众号内进行配置
     * @param params
     * @param isSilence   是否静默授权  true：snsapi_base，false：snsapi_userinfo
     * @return
     */
    public String getConnectOauth2Url(String redirectUri, String params, Boolean isSilence) {
        String url = WeChatCommonConstants.AGREE_URL
                .replace("APPID", weiXinConfig.getAppKey())
                .replace("REDIRECT_URI", URLEncoder.encode(redirectUri))
                .replace("SCOPE", isSilence ? "snsapi_base" : "snsapi_userinfo");
        if (params != null) {
            url = url.replace("STATE", params);
        }
        return url;
    }

    /**
     * 通过code换取网页授权access_token
     * 网页授权    第二步
     *
     * @param code
     * @return
     */
    public WeChatCommonResult oauth2AccessToken(String code) {

        try {
            String url = WeChatCommonConstants.EXCHANGE_URL
                    .replace("APPID", weiXinConfig.getAppKey())
                    .replace("SECRET", weiXinConfig.getAppSecret())
                    .replace("CODE", code);
            String responseResult = HttpUtils.doGet(url, "", WeChatCommonConstants.commonHeaders(), null);
            return WeChatCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));
        }
    }

    /**
     * 拉取用户信息(需scope为 snsapi_userinfo)
     * 获得授权的用户信息
     *
     * @param openId
     * @param token
     * @return
     */
    public WeChatCommonResult getAccessUserInfo(String token, String openId) {

        Map<String, String> query = new HashMap<>(7);
        query.put("access_token", token);
        query.put("openid", openId);
        query.put("lang", "zh_CN");
        try {
            String responseResult = HttpUtils.doGet(WeChatCommonConstants.API_WEI_XIN_SERVER_HOST, WeChatCommonConstants.GET_PERSON_URL, WeChatCommonConstants.commonHeaders(), query);
            return WeChatCommonConstants.analysisPubHttpResponse(responseResult);
        } catch (Exception e) {
            return WeChatCommonResult.error(ExceptionUtil.formatException(e));

        }
    }


    public static void main(String[] args) {
     /*   WeiXinConfig weiXinConfig = new WeiXinConfig("wxa095e06a92f8d952", "8a9367ecd2d0bc6fa2ab02c8a8d8e44a");
        WeiXinPubServicesUtils weiXinPubServicesUtils = WeiXinPubServicesUtils.getInstance().config(weiXinConfig);*/
        WeiXinPubServicesUtil weiXinPubServicesUtil = WeiXinPubServicesUtil.getInstance().config();

        //获取Token
        System.out.println(JSON.toJSON(weiXinPubServicesUtil.getToken()));

        //获取用户基本信息
        /*System.out.println(JSON.toJSON(weiXinPubServicesUtils.getUserInfo("ofZpz1tFF7Ek0C5ci4VJ398upvEo")));
        System.out.println(JSON.toJSON(weiXinPubServicesUtils.getUserInfo("ofZpz1tFF7Ek0C5ci4VJ398upvEo", WeiXinLanguageEnum.WX_EN)));*/

        //获取用户列表
        /*System.out.println(weiXinPubServicesUtils.syncUser());
        System.out.println(weiXinPubServicesUtils.syncUser("ofZpz1r-vrs7QhBOSfayERT_8tKQ"));*/

        //更新用户备注
        //System.out.println(weiXinPubServicesUtils.updateUserRemark("ofZpz1tFF7Ek0C5ci4VJ398upvEo", "李剑"));

        //创建菜单
       /* Menu menu = new Menu();
        List<Button> buttons = new ArrayList<>();
        ViewButton button = new ViewButton();
        button.setName("测试");
        button.setType("view");
        button.setUrl("http://13123");
        buttons.add(button);
        menu.setButton(buttons);
        System.out.println(JSON.toJSON(weiXinPubServicesUtils.createMenu(menu)));*/

        //获取二维码 永久&&临时
        //System.out.println(weiXinPubServicesUtils.getQRCodeByForever("131321"));
        //System.out.println(weiXinPubServicesUtils.getQRCodeByTemporary(20, "131321"));

        //模板消息发送
        /*MessageTemplateSend messageTemplateSend = new MessageTemplateSend(
                "ofZpz1tFF7Ek0C5ci4VJ398upvEo",
                "-1a8qhMITj5PiUIQwSzKCsSe3WWi7FGb-Cic1Crocuk",
                "http://chiship.cn"
        );
        Map<String, MessageTemplateSendDataItem> data = new HashMap<>();
        MessageTemplateSendDataItem messageTemplateSendDataItem = new MessageTemplateSendDataItem("山东省济南市长清区", "#ff0000");
        data.put("address", messageTemplateSendDataItem);
        messageTemplateSend.setData(data);
        System.out.println(weiXinPubServicesUtils.messageTemplateSend(messageTemplateSend));*/
    }


}
