package cn.chiship.sdk.framework.ue;

import cn.chiship.sdk.framework.ue.define.ActionMap;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


import javax.servlet.http.HttpServletRequest;
import java.io.*;
import java.util.HashMap;
import java.util.Map;

/**
 * 配置管理器
 *
 * @author hancong03@baidu.com
 */
public final class ConfigManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConfigManager.class);
    private final String rootPath;
    private final String originalPath;
    private static final String CONFIG_FILE_NAME = "UEConfig.json";
    private String parentPath = null;
    private String jsonConfigFilePath;
    private JSONObject jsonConfig = null;
    /**
     * 涂鸦上传filename定义
     */
    private final static String SCRAWL_FILE_NAME = "scrawl";
    /**
     * 远程图片抓取filename定义
     */
    private final static String REMOTE_FILE_NAME = "remote";

    /**
     * 通过一个给定的路径构建一个配置管理器， 该管理器要求地址路径所在目录下必须存在config.properties文件
     *
     * @param rootPath
     * @param jsonConfigFilePath
     * @param contextPath
     * @param uri
     * @throws FileNotFoundException
     * @throws IOException
     */
    private ConfigManager(String rootPath, String jsonConfigFilePath, String contextPath, String uri)
            throws IOException {

        rootPath = rootPath.replace("\\", "/");

        this.rootPath = rootPath;

        this.originalPath = this.rootPath + uri;

        this.parentPath = new File(jsonConfigFilePath).getParent();

        this.jsonConfigFilePath = jsonConfigFilePath;

        this.initEnv();
    }

    /**
     * 配置管理器构造工厂
     *
     * @param rootPath           服务器根路径
     * @param contextPath        服务器所在项目路径
     * @param uri                当前访问的uri
     * @param jsonConfigFilePath json配置文件的绝对路径，和controller.jsp在同一目录
     * @return 配置管理器实例或者null
     */
    public static ConfigManager getInstance(String rootPath, String jsonConfigFilePath,
                                            String contextPath, String uri) {

        try {
            return new ConfigManager(rootPath, jsonConfigFilePath, contextPath, uri);
        } catch (Exception e) {
            LOGGER.error("发生异常", e);
            return null;
        }

    }

    /**
     * 验证配置文件加载是否正确
     */
    public boolean valid() {
        return this.jsonConfig != null;
    }

    public JSONObject getAllConfig() {

        return this.jsonConfig;

    }

    public Map<String, Object> getConfig(int type) {

        Map<String, Object> conf = new HashMap<>(2);
        String savePath = null;

        switch (type) {

            case ActionMap.UPLOAD_FILE:
                conf.put("isBase64", "false");
                conf.put("maxSize", this.jsonConfig.getLong("fileMaxSize"));
                conf.put("allowFiles", this.getArray("fileAllowFiles"));
                conf.put("fieldName", this.jsonConfig.getString("fileFieldName"));
                savePath = this.jsonConfig.getString("filePathFormat");
                break;

            case ActionMap.UPLOAD_IMAGE:
                conf.put("isBase64", "false");
                conf.put("maxSize", this.jsonConfig.getLong("imageMaxSize"));
                conf.put("allowFiles", this.getArray("imageAllowFiles"));
                conf.put("fieldName", this.jsonConfig.getString("imageFieldName"));
                savePath = this.jsonConfig.getString("imagePathFormat");
                break;

            case ActionMap.UPLOAD_VIDEO:
                conf.put("maxSize", this.jsonConfig.getLong("videoMaxSize"));
                conf.put("allowFiles", this.getArray("videoAllowFiles"));
                conf.put("fieldName", this.jsonConfig.getString("videoFieldName"));
                savePath = this.jsonConfig.getString("videoPathFormat");
                break;

            case ActionMap.UPLOAD_SCRAWL:
                conf.put("filename", ConfigManager.SCRAWL_FILE_NAME);
                conf.put("maxSize", this.jsonConfig.getLong("scrawlMaxSize"));
                conf.put("fieldName", this.jsonConfig.getString("scrawlFieldName"));
                conf.put("isBase64", "true");
                savePath = this.jsonConfig.getString("scrawlPathFormat");
                break;

            case ActionMap.CATCH_IMAGE:
                conf.put("filename", ConfigManager.REMOTE_FILE_NAME);
                conf.put("filter", this.getArray("catcherLocalDomain"));
                conf.put("maxSize", this.jsonConfig.getLong("catcherMaxSize"));
                conf.put("allowFiles", this.getArray("catcherAllowFiles"));
                conf.put("fieldName", this.jsonConfig.getString("catcherFieldName")
                        + "[]");
                savePath = this.jsonConfig.getString("catcherPathFormat");
                break;

            case ActionMap.LIST_IMAGE:
                conf.put("allowFiles", this.getArray("imageManagerAllowFiles"));
                conf.put("dir", this.jsonConfig.getString("imageManagerListPath"));
                conf.put("count", this.jsonConfig.getIntValue("imageManagerListSize"));
                break;

            case ActionMap.LIST_FILE:
                conf.put("allowFiles", this.getArray("fileManagerAllowFiles"));
                conf.put("dir", this.jsonConfig.getString("fileManagerListPath"));
                conf.put("count", this.jsonConfig.getIntValue("fileManagerListSize"));
                break;
            default:
        }

        conf.put("savePath", savePath);
        conf.put("rootPath", this.rootPath);

        return conf;

    }

    /**
     * Get rootPath from request,if not,find it from conf map.
     */
    public static String getRootPath(HttpServletRequest request,
                                     Map<String, Object> conf) {
        Object rootPath = request.getAttribute("rootPath");
        if (rootPath != null) {
            return rootPath + "" + File.separatorChar;
        } else {
            return conf.get("rootPath") + "";
        }
    }

    private void initEnv() throws IOException {

        File file = new File(this.originalPath);

        if (!file.isAbsolute()) {
            file = new File(file.getAbsolutePath());
        }

        if (this.parentPath == null) {
            this.parentPath = file.getParent();
        }

        String configContent = this.readFile(this.getConfigPath());

        try {
            JSONObject jsonConfig = JSON.parseObject(configContent);
            this.jsonConfig = jsonConfig;
        } catch (Exception e) {
            this.jsonConfig = null;
        }

    }

    private String getConfigPath() {
        String path = jsonConfigFilePath;
        if (new File(path).exists()) {
            return path;
        } else {
            return this.parentPath + File.separator
                    + ConfigManager.CONFIG_FILE_NAME;
        }
    }

    private String[] getArray(String key) {

        JSONArray jsonArray = this.jsonConfig.getJSONArray(key);
        String[] result = new String[jsonArray.size()];

        for (int i = 0, len = jsonArray.size(); i < len; i++) {
            result[i] = jsonArray.getString(i);
        }

        return result;

    }

    private String readFile(String path) throws IOException {

        StringBuilder builder = new StringBuilder();
        InputStreamReader reader = null;
        BufferedReader bfReader = null;
        try {
            reader = new InputStreamReader(new FileInputStream(path));
            bfReader = new BufferedReader(reader);

            String tmpContent;

            while ((tmpContent = bfReader.readLine()) != null) {
                builder.append(tmpContent);
            }
        } catch (UnsupportedEncodingException e) {
            // 忽略
        } finally {
            if (bfReader != null) {
                bfReader.close();
            }
            if (reader != null) {
                reader.close();
            }
        }

        return this.filter(builder.toString());

    }

    /**
     * 过滤输入字符串, 剔除多行注释以及替换掉反斜杠
     */
    private String filter(String input) {

        return input.replaceAll("/\\*[\\s\\S]*?\\*/", "");

    }

}