package cn.chiship.sdk.framework.base;

import cn.chiship.sdk.core.annotation.Authorization;
import cn.chiship.sdk.core.base.BaseResult;
import cn.chiship.sdk.core.base.BaseResultEnum;
import cn.chiship.sdk.core.base.constants.BaseConstants;
import cn.chiship.sdk.core.exception.custom.BusinessException;
import cn.chiship.sdk.core.util.OutUtil;
import cn.chiship.sdk.framework.util.BindingResultUtil;
import cn.chiship.sdk.framework.util.ServletUtil;
import cn.chiship.sdk.core.util.StringUtil;
import cn.chiship.sdk.core.vo.PageVo;
import com.alibaba.fastjson.JSONObject;

import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletResponse;

/**
 * @author lijian
 */
public abstract class BaseController<Record, Example> {
    public abstract BaseService<Record, Example> getService();

    @ApiOperation(value = "根据指定字段校验数据是否存在")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "主键", defaultValue = "", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "field", value = "字段", defaultValue = "", required = true, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "value", value = "值", defaultValue = "", required = true, dataType = "String", paramType = "query"),
    })
    @Authorization
    @GetMapping(value = "validateExistByField")
    public ResponseEntity<BaseResult> validateExistByField(
            @RequestParam(required = false, defaultValue = "", value = "id") String id,
            @RequestParam(value = "field") String field,
            @RequestParam(value = "value") String value) {
        return new ResponseEntity(getService().validateExistByField(id, field, value), HttpStatus.OK);
    }

    @ApiOperation(value = "导出指定格式的数据文件")
    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "paramsMap",
                    value = "数据集合(包含导出文件格式字段【exportType:csv、xls、xlsx.json】及导出数据的条件)",
                    required = true,
                    dataType = "JSONObject",
                    paramType = "body")
    })
    @PostMapping(value = "exportData")
    public void exportData(@RequestBody(required = true) JSONObject paramsMap, HttpServletResponse response) throws Exception {
        if (!paramsMap.containsKey("exportType")) {
            throw new BusinessException("缺少参数[exportType]");
        }
        String exportType = paramsMap.get("exportType").toString();
        if (StringUtil.isNullOrEmpty(exportType)) {
            BaseResult baseResult = BaseResult.error( "参数[exportType]不能为空");
            OutUtil.writeJSON(response, baseResult);

        }
        getService().exportData(response, paramsMap);


    }

    /**
     * 分页
     *
     * @param example 条件
     * @return
     */
    public PageVo page(Example example) {
        PageVo pageVo = new PageVo();

        Integer pageIndex;
        Integer pageSize;
        try {
            pageIndex = (ServletUtil.getParameterToInt(BaseConstants.PAGE_NUM));
            pageSize = (ServletUtil.getParameterToInt(BaseConstants.PAGE_SIZE));
            if (pageIndex == null) {
                pageIndex = 1;
            }
            if (pageSize == null) {
                pageSize = 20;
            }
        } catch (Exception e) {
            pageIndex = 1;
            pageSize = 20;
        }
        pageVo.setCurrent(pageIndex.longValue());
        pageVo.setSize(pageSize.longValue());
        return getService().selectPageByExample(pageVo, example);
    }

    public BaseResult validParams(BindingResult bindingResult) {
        return BindingResultUtil.format(bindingResult);
    }
}
