package cn.blankcat.websocket.v1;

import cn.blankcat.dto.audio.AudioAction;
import cn.blankcat.dto.channel.Channel;
import cn.blankcat.dto.forum.ForumAuditResult;
import cn.blankcat.dto.forum.Post;
import cn.blankcat.dto.forum.Reply;
import cn.blankcat.dto.guild.Guild;
import cn.blankcat.dto.interaction.Interaction;
import cn.blankcat.dto.member.Member;
import cn.blankcat.dto.message.Message;
import cn.blankcat.dto.message.MessageAudit;
import cn.blankcat.dto.message.MessageDelete;
import cn.blankcat.dto.websocket.*;
import cn.blankcat.openapi.v1.OpenClient;
import cn.blankcat.openapi.v1.service.WsService;
import cn.blankcat.token.Token;
import cn.blankcat.websocket.handler.AbstractWebsocketHandler;
import cn.blankcat.websocket.handler.WsHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

public class WsClient {

    public static final HashMap<Class<?>, ArrayList<WsHandler>> HANDLERS = new HashMap<>();
    private static final long concurrencyTimeWindowSec = 2;
    private static final Logger logger = LoggerFactory.getLogger("WsClient");

    static {
        HANDLERS.put(Guild.class, new ArrayList<>());
        HANDLERS.put(Message.class, new ArrayList<>());
        HANDLERS.put(MessageDelete.class, new ArrayList<>());
        HANDLERS.put(MessageAudit.class, new ArrayList<>());
        HANDLERS.put(Channel.class, new ArrayList<>());
        HANDLERS.put(Member.class, new ArrayList<>());
        HANDLERS.put(AudioAction.class, new ArrayList<>());
        HANDLERS.put(Thread.class, new ArrayList<>());
        HANDLERS.put(Post.class, new ArrayList<>());
        HANDLERS.put(Reply.class, new ArrayList<>());
        HANDLERS.put(ForumAuditResult.class, new ArrayList<>());
        HANDLERS.put(Interaction.class, new ArrayList<>());
        HANDLERS.put(WSHelloData.class, new ArrayList<>());
        HANDLERS.put(WSPayload.class, new ArrayList<>());
    }

    public static void startClient(){
        registryHandler();
        WsService service = OpenClient.getService(WsService.class);
        Token token = Token.loadFromConfig(null, Token.Type.TYPE_BOT);
        try {
            WebsocketAP apiInfo = service.ws().execute().body();
            // long startInterval = calcInterval(apiInfo.sessionStartLimit().maxConcurrency());
            logger.info("即将启动{}个sessions...", apiInfo.shards());
            for (int i = 0; i< apiInfo.shards(); i++){
                ShardConfig shards = new ShardConfig(i, apiInfo.shards());

                Session session = new Session("", apiInfo.url(), token, WSEvent.allEventToIntent(),
                        1, shards);

                WsRetrofit.startWs(session);
            }
        } catch (IOException e) {
            logger.error(e.getMessage());
        }
    }

    private long calcInterval(long maxConcurrency){
        if (maxConcurrency == 0){
            maxConcurrency = 1;
        }
        int i = Math.round((float) concurrencyTimeWindowSec / maxConcurrency);
        return i <= 0 ? 1 : i;
    }

    public static void registryHandler(){
        for ( Class<?> clazz : findSubclassesOf(AbstractWebsocketHandler.class)) {
            try {
                AbstractWebsocketHandler<?> handler = (AbstractWebsocketHandler<?>) clazz.getDeclaredConstructor().newInstance();
                handler.registry();
            } catch (Exception e) {
                logger.error(e.getMessage());
            }
        }
    }

    private static List<Class<?>> findSubclassesOf(Class<?> superClass) {
        List<Class<?>> subclasses = new ArrayList<>();
        Package[] packages = Package.getPackages();

        for (Package pkg : packages) {
            List<Class<?>> classes = findClassesInPackage(pkg.getName());

            for (Class<?> clazz : classes) {
                if (superClass.isAssignableFrom(clazz) && !superClass.equals(clazz)) {
                    subclasses.add(clazz);
                }
            }
        }
        return subclasses;
    }

    private static List<Class<?>> findClassesInPackage(String packageName) {
        List<Class<?>> classes = new ArrayList<>();
        String packagePath = packageName.replace('.', '/');
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        try {
            java.net.URL packageURL = classLoader.getResource(packagePath);
            java.io.File packageDir = new java.io.File(packageURL.getFile());
            if (packageDir.exists()) {
                String[] files = packageDir.list();
                for (String fileName : files) {
                    if (fileName.endsWith(".class")) {
                        String className = packageName + '.' + fileName.substring(0, fileName.length() - 6);
                        try {
                            Class<?> cls = Class.forName(className);
                            classes.add(cls);
                        } catch (Exception e) {
                            logger.error(e.getMessage());
                        }
                    }
                }
            }
        } catch (Exception e) {
            // 处理异常
        }
        return classes;
    }
}
