package cn.blankcat.websocket.v1;

import cn.blankcat.dto.websocket.*;
import cn.blankcat.websocket.handler.WsHandler;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import okhttp3.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.IOException;

public class MyListener extends WebSocketListener {

    private final Logger logger = LoggerFactory.getLogger("MyListener");
    private Session session;

    public MyListener(Session session) {
        this.session = session;
    }

    private void buildInHandler(WebSocket webSocket, WSPayload<?> wsPayload) throws JsonProcessingException {
        WSOPCode wsOPCode = WSOPCode.ofValue(wsPayload.opCode());
        switch (wsOPCode){
            case WSDispatchEvent:{
                if ("READY".equals(wsPayload.type())){
                    ObjectMapper objectMapper = new ObjectMapper();
                    WSReadyData readyData = objectMapper.convertValue(wsPayload.data(), WSReadyData.class);
                    ShardConfig shardConfig = new ShardConfig(readyData.shard()[0], readyData.shard()[1]);
                    session = new Session(readyData.sessionID(), session.url(), session.token(),
                            session.intent(), wsPayload.seq(), shardConfig);
                }
                if ("RESUMED".equals(wsPayload.type())){

                }
                break;
            }
            case WSHello: {
                WsRetrofit.hello(webSocket,session);
                break;
            }
            case WSHeartbeatAck:{
                // 心跳 ack 不需要业务处理
                break;
            }
            case WSReconnect:{
                // 达到连接时长，需要重新连接，此时可以通过 resume 续传原连接上的事件
                WsRetrofit.resume(webSocket,session);
                break;
            }
            case WSInvalidSession:{
                // 无效的 sessionLog，需要重新鉴权
                WsRetrofit.identify(webSocket,session);
                break;
            }
        }
    }

    @Override
    public void onMessage(WebSocket webSocket, String text) {
        logger.info("收到信息-->[{}]", text);
        ObjectMapper mapper = new ObjectMapper();
        try {
            WSPayload<?> wsPayload = mapper.readValue(text, WSPayload.class);
            // 每次收到消息都保存当前最新的seq值
            if (wsPayload.seq() > 0){
                session = new Session(session.id(), session.url(), session.token(),
                        session.intent(), wsPayload.seq(), session.shards());
            }
            WSEventType wsEventType = WSEventType.ofValue(wsPayload.type());
            Class<?> clazz = WSEvent.eventClassMap.get(wsEventType);
            if (clazz == null){
                buildInHandler(webSocket, wsPayload);
            }else {
                for (WsHandler handler : WsClient.HANDLERS.get(clazz)){
                    handler.handle(text, webSocket);
                }
            }
        } catch (JsonProcessingException e) {
            logger.error(e.getMessage());
        }
    }

    @Override
    public void onFailure(WebSocket webSocket, Throwable t, @Nullable Response response) {
        if (response != null && response.body() != null) {
            MediaType mediaType = response.body().contentType();
            try {
                String responseString = response.body().string();
                if (response.code() > 209) {
                    logger.warn("请求错误, 错误内容为:{}", responseString);
                }
                super.onFailure(webSocket, t, response.newBuilder().body(ResponseBody.create(mediaType, responseString)).build());
            } catch (IOException e) {
                logger.error(e.getMessage());
            }
        }
    }
}
