package cn.blankcat.token;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;

@Data
@AllArgsConstructor
@NoArgsConstructor
@JsonIgnoreProperties(ignoreUnknown = true)
public class Token {

    private static final Logger logger = LoggerFactory.getLogger("token");
    public static final Token EMPTY = new Token();

    public enum Type{
        TYPE_BOT("Bot"),
        TYPE_NORMAL("Bearer");

        private final String value;

        Type(String value){
            this.value = value;
        }
        @JsonValue
        public String getValue(){
            return value;
        }
    }

    private long appId;
    private String accessToken;
    private Type type;

    public String getRealString(){
        if (this.type == Type.TYPE_NORMAL){
            return this.type.value + " " + this.accessToken;
        }
        return this.type.value + " " + this.appId + "." + this.accessToken;
    }

    public static Token BotToken(long appId, String accessToken){
        Token token = new Token();
        token.appId = appId;
        token.accessToken = accessToken;
        token.type = Type.TYPE_BOT;
        return token;
    }

    public static Token UserToken(long appId, String accessToken){
        Token token = new Token();
        token.appId = appId;
        token.accessToken = accessToken;
        token.type = Type.TYPE_NORMAL;
        return token;
    }

    public static Token loadFromConfig(String filePath, Type tokenType){
        if (filePath == null || filePath.isEmpty()) {
            filePath = "bot.yaml";
        }
        ObjectMapper mapper = new ObjectMapper(new YAMLFactory());
        InputStream inputStream = Token.class
                .getClassLoader()
                .getResourceAsStream(filePath);
        mapper.findAndRegisterModules();
        try {
            Token token = mapper.readValue(inputStream, Token.class);
            if (token.getAccessToken().isEmpty() || token.getAccessToken().isBlank()){
                logger.warn("error read token from file ");
            }
            if (token.getAppId() ==0 ){
                logger.warn("error read appid from file ");
            }
            token.type = tokenType;
            return token;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return EMPTY;
    }
}
