package cn.blankcat.dto.channel;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonValue;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.util.Arrays;

@Data
@AllArgsConstructor
@NoArgsConstructor
@JsonIgnoreProperties(ignoreUnknown = true)
public class ChannelValueObject {

    // ChannelType 频道类型定义
    public enum ChannelType {
        CHANNEL_TYPE_TEXT(0),
        CHANNEL_TYPE_VOICE(2),
        CHANNEL_TYPE_CATEGORY(4),
        CHANNEL_TYPE_LIVE(10005),           // 直播子频道
        CHANNEL_TYPE_APPLICATION(10006),    // 应用子频道
        CHANNEL_TYPE_BBS(10007);            // 论坛子频道
        private final int value;

        ChannelType(int value) {
            this.value = value;
        }

        @JsonValue
        public int getValue() {
            return value;
        }

    }

    // ChannelSubType 子频道子类型定义
    public enum ChannelSubType {
        CHANNEL_SUBTYPE_CHAT(0),       // 闲聊，默认子类型
        CHANNEL_SUBTYPE_NOTICE(1),     // 公告
        CHANNEL_SUBTYPE_GUIDE(2),      // 攻略
        CHANNEL_SUBTYPE_TEAMGAME(3);   // 开黑
        private final int value;

        ChannelSubType(int value) {
            this.value = value;
        }

        @JsonValue
        public int getValue() {
            return value;
        }
    }

    // ChannelPrivateType 频道可见性类型定义
    public enum ChannelPrivateType {
        CHANNEL_PRIVATE_TYPE_PUBLIC(0),           // 公开频道
        CHANNEL_PRIVATE_TYPE_ONLY_ADMIN(1),       // 群主管理员可见
        CHANNEL_PRIVATE_TYPE_ADMIN_AND_MEMBER(2); // 群主管理员+指定成员
        private final int value;

        ChannelPrivateType(int value) {
            this.value = value;
        }

        @JsonValue
        public int getValue() {
            return value;
        }
    }

    // ChannelPrivateType 频道可见性类型定义
    public enum SpeakPermissionType {
        SPEAK_PERMISSION_TYPE_PUBLIC(1),           // 公开发言权限
        SPEAK_PERMISSION_TYPE_ADMIN_AND_MEMBER(2); // 指定成员可发言

        private final int value;

        SpeakPermissionType(int value) {
            this.value = value;
        }

        @JsonValue
        public int getValue() {
            return value;
        }

        public static SpeakPermissionType ofValue(int value){
            return Arrays.stream(SpeakPermissionType.values()).filter(SpeakPermissionType -> SpeakPermissionType.value == value).findFirst().orElse(null);
        }
    }


    // 频道名称
    private String name;
    // 频道类型
    private ChannelType type;
    // 排序位置
    private long position;
    // 父频道的ID
    @JsonProperty("parent_id")
    private String parentID;
    // 拥有者ID
    @JsonProperty("owner_id")
    private String ownerID;
    // 子频道子类型
    @JsonProperty("sub_type")
    private ChannelSubType subType;
    // 子频道可见性类型
    @JsonProperty("private_type")
    private ChannelPrivateType privateType;
    // 创建私密子频道的时候，同时带上 userID，能够将这些成员添加为私密子频道的成员
    // 注意：只有创建私密子频道的时候才支持这个参数
    @JsonProperty("private_user_ids")
    private String[] privateUserIds;
    // 发言权限
    @JsonProperty("speak_permission")
    private int speakPermission;
    // 应用子频道的应用ID，仅应用子频道有效，定义请参考
    // [文档](https://bot.q.qq.com/wiki/develop/api/openapi/channel/model.html)
    @JsonProperty("application_id")
    private String applicationID;
    // 机器人在此频道上拥有的权限, 定义请参考
    // [文档](https://bot.q.qq.com/wiki/develop/api/openapi/channel_permissions/model.html#permissions)
    private String permissions;
    // 操作人
    @JsonProperty("op_user_id")
    private String OpUserID;

}
